﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <functional>
#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/settings_ServiceTypes.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/settings/fwdbg/settings_SettingsCommon.h>
#include <nn/TargetConfigs/build_Base.h>

#include "settings_FirmwareDebugSettingsDatabase.h"
#include "settings_KeyValueStore.h"
#include "settings_Platform.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_Bsp0-os.horizon.h"
#endif

//!< ファームウェアデバッグ設定の警告ログを出力します。
#define NN_SETTINGS_FWDBG_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[firmware debug settings] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< 自身の設定の名前
const SettingsName PersonalSettingsName = {
    "settings_debug" };

//!< デバッグモードフラグのキー名
const SettingsItemKey DebugModeFlagKey = {
    "is_debug_mode_enabled" };

//!< HDMI-CEC 抑止フラグのキー名
const SettingsItemKey HdmiCecSuppressionFlagKey = {
    "is_hdmi_cec_suppression_enabled" };

//!< 設定名を検証結果を返します。
::nn::Result GetResultOfSettingsNameValidation(const char* name) NN_NOEXCEPT;

//!< 設定項目のキーの検証結果を返します。
::nn::Result GetResultOfSettingsItemKeyValidation(const char* key) NN_NOEXCEPT;

} // namespace

FwdbgSettingsItemKeyIterator::FwdbgSettingsItemKeyIterator() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_Impl()
{
    // 何もしない
}

FwdbgSettingsItemKeyIterator::FwdbgSettingsItemKeyIterator(
    FwdbgSettingsItemKeyIterator&& other) NN_NOEXCEPT
    : m_IsInitialized(::std::move(other.m_IsInitialized))
    , m_Impl(::std::move(other.m_Impl))
{
    other.m_IsInitialized = false;
}

FwdbgSettingsItemKeyIterator::~FwdbgSettingsItemKeyIterator() NN_NOEXCEPT
{
    this->Finalize();
}

::nn::Result FwdbgSettingsItemKeyIterator::Initialize(
    const SettingsName& name) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResultOfSettingsNameValidation(name.value));
    this->Finalize();
    NN_RESULT_DO(KeyValueStore(name).CreateKeyIterator(&m_Impl));
    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

void FwdbgSettingsItemKeyIterator::Finalize() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        DestroyKeyValueStoreKeyIterator(&m_Impl);
        m_IsInitialized = false;
    }
}

::nn::Result FwdbgSettingsItemKeyIterator::GoNext() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    NN_RESULT_DO(AdvanceKeyValueStoreKeyIterator(&m_Impl));
    NN_RESULT_SUCCESS;
}

::nn::Result FwdbgSettingsItemKeyIterator::GetKeySize(
    uint64_t* pOutCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullSettingsItemKeySizeBuffer());
    NN_RESULT_DO(GetKeyValueStoreKeyIteratorKeySize(pOutCount, m_Impl));
    NN_RESULT_SUCCESS;
}

::nn::Result FwdbgSettingsItemKeyIterator::GetKey(
    uint64_t* pOutCount, char* outBuffer, size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullSettingsItemKeySizeBuffer());
    NN_RESULT_THROW_UNLESS(
        outBuffer != nullptr, ResultNullSettingsItemKeyBuffer());
    NN_RESULT_DO(
        GetKeyValueStoreKeyIteratorKey(pOutCount, outBuffer, count, m_Impl));
    NN_RESULT_SUCCESS;
}

::nn::Result GetFwdbgDebugModeFlag(bool* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullDebugModeFlagBuffer());

    auto count = uint64_t();

    const ::nn::Result result = KeyValueStore(PersonalSettingsName).GetValue(
        &count,
        reinterpret_cast<char*>(pOutValue), sizeof(*pOutValue),
        DebugModeFlagKey);

    if (result.IsFailure() || count != sizeof(*pOutValue))
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        NN_SETTINGS_FWDBG_WARN(
            "IsDebugModeEnabled() failed. (%08x, %d%03d-%04d)\n",
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
#endif
        *pOutValue = true;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetFwdbgHdmiCecSuppressionFlagForInternal(
    bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto count = uint64_t();

    const ::nn::Result result = KeyValueStore(PersonalSettingsName).GetValue(
        &count,
        reinterpret_cast<char*>(pOutValue), sizeof(*pOutValue),
        HdmiCecSuppressionFlagKey);

    if (result.IsFailure() || count != sizeof(*pOutValue))
    {
#ifdef NN_BUILD_CONFIG_OS_HORIZON
        NN_SETTINGS_FWDBG_WARN(
            "Failed to get the \"%s.%s\" value. (%08x, %d%03d-%04d)\n",
            PersonalSettingsName.value,
            HdmiCecSuppressionFlagKey.value,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
#endif
        *pOutValue = false;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetFwdbgSettingsItemValueSize(
    uint64_t* pOutCount, const SettingsName& name, const SettingsItemKey& key
    ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullSettingsItemValueSizeBuffer());

    NN_RESULT_DO(GetResultOfSettingsNameValidation(name.value));
    NN_RESULT_DO(GetResultOfSettingsItemKeyValidation(key.value));

    ::nn::Result result = KeyValueStore(name).GetValueSize(pOutCount, key);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    if (result.IsFailure())
    {
        if (GetBuiltInBsp0SettingsItemValueSize(pOutCount,
                                                name.value, key.value))
        {
            NN_SETTINGS_FWDBG_WARN(
                "GetSettingsItemValueSize(\"%s\", \"%s\") "
                "returned the built-in value: %llu. (%08x, %d%03d-%04d)\n",
                name.value, key.value,
                *pOutCount,
                result.GetInnerValueForDebug(),
                ErrorCodePlatformId,
                result.GetModule(), result.GetDescription());

            result = ::nn::ResultSuccess();
        }
    }
#endif

    if (result.IsFailure())
    {
        NN_SETTINGS_FWDBG_WARN(
            "GetSettingsItemValueSize(\"%s\", \"%s\") "
            "failed. (%08x, %d%03d-%04d)\n",
            name.value, key.value,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        *pOutCount = 0;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetFwdbgSettingsItemValue(
    uint64_t* pOutCount, char* outBuffer, size_t count,
    const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullSettingsItemValueSizeBuffer());
    NN_RESULT_THROW_UNLESS(
        outBuffer != nullptr, ResultNullSettingsItemValueBuffer());

    NN_RESULT_DO(GetResultOfSettingsNameValidation(name.value));
    NN_RESULT_DO(GetResultOfSettingsItemKeyValidation(key.value));

    ::nn::Result result =
          KeyValueStore(name).GetValue(pOutCount, outBuffer, count, key);

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    if (result.IsFailure())
    {
        if (GetBuiltInBsp0SettingsItemValue(pOutCount, outBuffer, count,
                                            name.value, key.value))
        {
            NN_SETTINGS_FWDBG_WARN(
                "GetSettingsItemValue(\"%s\", \"%s\") "
                "returned the built-in value: \"%s\". (%08x, %d%03d-%04d)\n",
                name.value, key.value, outBuffer,
                result.GetInnerValueForDebug(),
                ErrorCodePlatformId,
                result.GetModule(), result.GetDescription());

            result = ::nn::ResultSuccess();
        }
    }
#endif

    if (result.IsFailure())
    {
        NN_SETTINGS_FWDBG_WARN(
            "GetSettingsItemValue(\"%s\", \"%s\") "
            "failed. (%08x, %d%03d-%04d)\n",
            name.value, key.value,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        *pOutCount = 0;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SetFwdbgSettingsItemValue(
    const SettingsName& name, const SettingsItemKey& key,
    const char* buffer, size_t count) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResultOfSettingsNameValidation(name.value));
    NN_RESULT_DO(GetResultOfSettingsItemKeyValidation(key.value));
    NN_RESULT_THROW_UNLESS(buffer != nullptr, ResultNullSettingsItemValue());
    NN_RESULT_DO(KeyValueStore(name).SetValue(key, buffer, count));
    NN_RESULT_SUCCESS;
}

::nn::Result ResetFwdbgSettingsItemValue(
    const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT
{
    NN_RESULT_DO(GetResultOfSettingsNameValidation(name.value));
    NN_RESULT_DO(GetResultOfSettingsItemKeyValidation(key.value));
    NN_RESULT_DO(KeyValueStore(name).ResetValue(key));
    NN_RESULT_SUCCESS;
}

::nn::Result ReadFwdbgSettings(
    uint64_t* pOutCount, char outBuffer[], size_t count,
    ::nn::settings::fwdbg::SettingsTarget target) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutCount != nullptr, ResultNullSettingsCountBuffer());
    NN_RESULT_THROW_UNLESS(
        outBuffer != nullptr, ResultNullSettingsBuffer());

    auto found = false;

    ::std::function<
        ::nn::Result(uint64_t*, char[], size_t) NN_NOEXCEPT> function;

    switch (target)
    {
    case ::nn::settings::fwdbg::SettingsTarget_SystemSaveFirmwareDebug:
        found = true;
        function = ReadKeyValueStoreSaveData;
        break;

    case ::nn::settings::fwdbg::SettingsTarget_SystemDataFirmwareDebug:
        found = true;
        function = ReadKeyValueStoreFirmwareDebug;
        break;

    case ::nn::settings::fwdbg::SettingsTarget_SystemDataPlatformConfigration:
        found = true;
        function = ReadKeyValueStorePlatformConfigration;
        break;

    default:
        break;
    }

    if (!found)
    {
        NN_SETTINGS_FWDBG_WARN(
            "ReadSettings() doesn't support %d\n", static_cast<int>(target));
    }
    else
    {
        const ::nn::Result result = function(pOutCount, outBuffer, count);

        if (result.IsFailure())
        {
            NN_SETTINGS_FWDBG_WARN(
                "ReadSettings() failed. (%08x, %d%03d-%04d)\n",
                result.GetInnerValueForDebug(),
                ErrorCodePlatformId,
                result.GetModule(), result.GetDescription());

            *pOutCount = 0;
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ResetFwdbgSettings(
    ::nn::settings::fwdbg::SettingsTarget target) NN_NOEXCEPT
{
    auto found = false;

    ::std::function<::nn::Result() NN_NOEXCEPT> function;

    switch (target)
    {
    case ::nn::settings::fwdbg::SettingsTarget_SystemSaveFirmwareDebug:
        found = true;
        function = ResetKeyValueStoreSaveData;
        break;

    default:
        break;
    }

    if (!found)
    {
        NN_SETTINGS_FWDBG_WARN(
            "ResetSettings() doesn't support %d\n", static_cast<int>(target));
    }
    else
    {
        const ::nn::Result result = function();

        if (result.IsFailure())
        {
            NN_SETTINGS_FWDBG_WARN(
                "ResetSettings() failed. (%08x, %d%03d-%04d)\n",
                result.GetInnerValueForDebug(),
                ErrorCodePlatformId,
                result.GetModule(), result.GetDescription());
        }
    }

    NN_RESULT_SUCCESS;
}

namespace {

//!< 指定された文字列が受け入れ可能な文字のみから成る否か表す値を返します。
bool ConsistsOfAcceptableCharacters(const char* str, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(str);

    if (length > 0 && str[length - 1] == '.')
    {
        return false;
    }

    for (size_t i = 0; i < length; ++i)
    {
        const char c = *(str++);

        if (('a' <= c && c <= 'z') || ('0' <= c && c <= '9'))
        {
            continue;
        }

        if (c == '-' || c == '.' || c == '_')
        {
            continue;
        }

        return false;
    }

    return true;
}

::nn::Result GetResultOfSettingsNameValidation(const char* name) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(name != nullptr, ResultNullSettingsName());
    const size_t length = ::strnlen(
        name,
        ::nn::settings::fwdbg::SettingsNameLengthMax + 1);
    NN_RESULT_THROW_UNLESS(0 < length , ResultEmptySettingsName());
    NN_RESULT_THROW_UNLESS(
        length <= ::nn::settings::fwdbg::SettingsNameLengthMax,
        ResultTooLongSettingsName());
    NN_RESULT_THROW_UNLESS(
        ConsistsOfAcceptableCharacters(name, length),
        ResultInvalidFormatSettingsName());
    NN_RESULT_SUCCESS;
}

::nn::Result GetResultOfSettingsItemKeyValidation(const char* key) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(key != nullptr, ResultNullSettingsItemKey());
    const size_t length = ::strnlen(
        key,
        ::nn::settings::fwdbg::SettingsItemKeyLengthMax + 1);
    NN_RESULT_THROW_UNLESS(0 < length , ResultEmptySettingsItemKey());
    NN_RESULT_THROW_UNLESS(
        length <= ::nn::settings::fwdbg::SettingsItemKeyLengthMax,
        ResultTooLongSettingsItemKey());
    NN_RESULT_THROW_UNLESS(
        ConsistsOfAcceptableCharacters(key, length),
        ResultInvalidFormatSettingsItemKey());
    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail
