﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkAssert.h>

#include <nn/pwm/detail/pwm_Log.h>
#include <nn/pwm/driver/pwm_Lib.h>
#include <nn/pwm/driver/pwm_Channel.h>
#include <nn/pwm/driver/pwm_ChannelDev.h>

#include "detail/pwm_ChannelNameMap.h"
#include "detail/pwm_ChannelSessionImpl-soc.tegra-x1.h"
#include "detail/pwm_DriverImpl-soc.tegra-x1.h"
#include "detail/pwm_DriverImplBlinkLed-soc.tegra-x1.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/spl/spl_Api.h>
#endif

namespace {

nn::pwm::driver::detail::DriverImpl g_DriverImpl;
nn::pwm::driver::detail::DriverImplBlinkLed g_DriverImplBlinkLed;

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
nn::spl::HardwareType g_HardwareType;
bool                  g_NeedSpiRead = true;

nn::spl::HardwareType GetHardwareType() NN_NOEXCEPT
{
    if (g_NeedSpiRead)
    {
        nn::Bit64 config;
        nn::spl::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GetConfig(&config, nn::spl::ConfigItem_HardwareType));
        nn::spl::Finalize();
        g_HardwareType = static_cast<nn::spl::HardwareType>(config);
        g_NeedSpiRead = false;
    }
    return g_HardwareType;
}
#endif

bool IsSupportedChannelName(nn::pwm::ChannelName name) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    auto hardwareType = GetHardwareType();
    return nn::pwm::driver::detail::ChannelSupportInfoMap[name].supportBoardFlag & 1 << hardwareType;
#else
    NN_UNUSED(name);
    return true;
#endif
}

}


namespace nn {
namespace pwm {
namespace driver {

NN_STATIC_ASSERT(sizeof(detail::ChannelSessionImplPadded) == detail::ChannelSessionSize);
NN_STATIC_ASSERT(sizeof(detail::ChannelSessionImpl) <= detail::ChannelSessionSize);
NN_STATIC_ASSERT(std::alignment_of<detail::ChannelSessionImplPadded>::value == detail::ChannelSessionAlignment);


void Initialize() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // Initialize で一度呼んでおくことで、タイミングで spl のセッションが枯渇するのを防ぐ
    GetHardwareType();
#endif
    g_DriverImpl.Initialize();
}

void Finalize() NN_NOEXCEPT
{
    g_DriverImpl.Finalize();
}

void OpenSessionForDev(ChannelSession* pOutSession, int channelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);

    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pOutSession);
    session.isOpen = true;
    session.channelIndex = channelIndex;
    session.isEnabled = false;
    session.period = 0;
    session.duty = 0;
    session.pDriverImpl = &g_DriverImpl; // OpenSessionForDev は BlinkLED には使えない

    // ドライバ本体の Open 処理
    session.pDriverImpl->Open(session);
}

void OpenSession(ChannelSession* pOutSession, ChannelName name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSession);
    NN_ABORT_UNLESS(IsSupportedChannelName(name));

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // ChannelName_BlinkLed は specs=NX の場合しか存在しないので分岐
    if (name == ChannelName_BlinkLed)
    {
        detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pOutSession);
        session.isOpen = true;
        session.channelIndex = 0;
        session.isEnabled = false;
        session.period = 0;
        session.duty = 0;
        session.pDriverImpl = &g_DriverImplBlinkLed;

        session.pDriverImpl->Open(session);
        return;
    }
#endif

    auto channelIndex = detail::GetChannelIndex(name);

    NN_ABORT_UNLESS(channelIndex != detail::UnsupportedChannelIndex,
        "PWM: Channel with specified name was not found\n");

    // 内部で pOutSession が初期化される
    OpenSessionForDev(pOutSession, channelIndex);
}

void CloseSession(ChannelSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    session.pDriverImpl->Close(session);
    session.isOpen = false;
}

void SetPeriod(ChannelSession* pSession, nn::TimeSpan period) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    session.pDriverImpl->SetPeriod(session, period);

    session.period = period;
}

nn::TimeSpan GetPeriod(ChannelSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    return session.pDriverImpl->GetPeriod(session);
}

void SetDuty(ChannelSession* pSession, int duty) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    session.pDriverImpl->SetDuty(session, duty);

    session.duty = duty;
}

int GetDuty(ChannelSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    return session.pDriverImpl->GetDuty(session);
}

void SetEnabled(ChannelSession* pSession, bool enable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    session.pDriverImpl->SetEnabled(session, enable);

    session.isEnabled = enable;
}

bool GetEnabled(ChannelSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession);
    detail::ChannelSessionImpl& session = detail::ToChannelSessionImpl(*pSession);
    NN_SDK_REQUIRES(session.isOpen);

    return session.pDriverImpl->GetEnabled(session);
}

void SuspendAllChannels() NN_NOEXCEPT
{
    g_DriverImpl.SuspendAllChannels();
}

void ResumeAllChannels() NN_NOEXCEPT
{
    g_DriverImpl.ResumeAllChannels();
}

} // driver
} // pwm
} // nn
