﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>

#include <nn/fan/fan.h>
#include <nn/gpio/gpio.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_ErrorReporter.h"
#include "psm_SupplyRouteDriver.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

const auto Vdd50AOffTime = ::nn::TimeSpan::FromMilliSeconds(20);
const auto SwitchingTime = ::nn::TimeSpan::FromMilliSeconds(500);

} // namespace

SupplyRouteDriver::SupplyRouteDriver() NN_NOEXCEPT
    : m_Vdd50State(Vdd50State::Unknown)
    , m_UpdateMutex(false)
{
    ::nn::fan::Initialize();
    ::nn::fan::OpenController(&m_FanController, ::nn::fan::FanName_Cpu);

    ::nn::gpio::Initialize();
    ::nn::gpio::OpenSession(&m_Vdd50ASession, ::nn::gpio::GpioPadName_Vdd50AEn);
    ::nn::gpio::SetDirection(&m_Vdd50ASession, ::nn::gpio::Direction_Output);
    ::nn::gpio::OpenSession(&m_Vdd50BSession, ::nn::gpio::GpioPadName_Vdd50BEn);
    ::nn::gpio::SetDirection(&m_Vdd50BSession, ::nn::gpio::Direction_Output);
}

SupplyRouteDriver::~SupplyRouteDriver() NN_NOEXCEPT
{
    ::nn::gpio::CloseSession(&m_Vdd50ASession);
    ::nn::gpio::CloseSession(&m_Vdd50BSession);
    ::nn::gpio::Finalize();

    ::nn::fan::CloseController(&m_FanController);
    ::nn::fan::Finalize();
}

::nn::Result SupplyRouteDriver::SetSupplyRoute(SupplyRoute supplyRoute) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_UpdateMutex)> updateLocker(m_UpdateMutex);

    auto newVdd50State = Vdd50State::Unknown;

    switch ( supplyRoute )
    {
    case SupplyRoute_None:
        newVdd50State = Vdd50State::None;
        break;
    case SupplyRoute_Vdd50A:
        newVdd50State = Vdd50State::Vdd50A;
        break;
    case SupplyRoute_Vdd50B:
        newVdd50State = Vdd50State::Vdd50B;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if ( IsSupplyRouteUpdated(newVdd50State) )
    {
        Disconnect();
        Connect(newVdd50State);
    }

    NN_RESULT_SUCCESS;
}

bool SupplyRouteDriver::IsSupplyRouteUpdated(Vdd50State newState) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(newState, Vdd50State::Unknown);

    // 現在の経路と異なる場合は更新する
    return newState != m_Vdd50State;
}

void SupplyRouteDriver::Disconnect() NN_NOEXCEPT
{
    if ( m_Vdd50State == Vdd50State::None )
    {
        // 現状態が None であれば切断操作は必要ない
        // m_LastVdd50OffTime も更新しない
        return;
    }

    ::nn::gpio::SetValue(&m_Vdd50ASession, ::nn::gpio::GpioValue_Low);
    ::nn::gpio::SetValue(&m_Vdd50BSession, ::nn::gpio::GpioValue_Low);
    m_LastVdd50OffTime = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());

    // SIGLO-83162: 現状態が VDD50A の時は DisableFan を VDD50A/VDD50B の操作の後に行う
    if ( m_Vdd50State == Vdd50State::Vdd50A )
    {
        ::nn::os::SleepThread(Vdd50AOffTime);
    }

    ::nn::fan::DisableFan(&m_FanController);

    m_Vdd50State = Vdd50State::None;
    GetErrorReporter().SetPowerSupplyPath(static_cast<int>(m_Vdd50State));
}

void SupplyRouteDriver::Connect(Vdd50State newState) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(newState, Vdd50State::Unknown);

    if ( newState == Vdd50State::None )
    {
        // 新状態が None であれば接続操作は必要ない（ログは一貫性を持たせるためにこちらに入れておく）
        NN_DETAIL_PSM_TRACE("VDD50A Off, VDD50B Off.\n");
        return;
    }

    // TODO: Connect を実行する TimerEvent の Handler を追加してこの ::nn::os::SleepThread を削除する
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // 前回 VDD50A, VDD50B を無効にしてからの経過時間が SwitchingTime に満たない場合だけスリープさせる
    auto currentTime = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());
    auto elapsedTime = currentTime - m_LastVdd50OffTime;
    if ( elapsedTime < SwitchingTime )
    {
        ::nn::os::SleepThread(SwitchingTime - elapsedTime);
    }
#endif

    switch ( newState )
    {
    case Vdd50State::Vdd50A:
        ::nn::gpio::SetValue(&m_Vdd50ASession, ::nn::gpio::GpioValue_High);
        NN_DETAIL_PSM_TRACE("VDD50A On, VDD50B Off.\n");
        break;
    case Vdd50State::Vdd50B:
        ::nn::gpio::SetValue(&m_Vdd50BSession, ::nn::gpio::GpioValue_High);
        NN_DETAIL_PSM_TRACE("VDD50A Off, VDD50B On.\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    ::nn::fan::EnableFan(&m_FanController);

    m_Vdd50State = newState;
    GetErrorReporter().SetPowerSupplyPath(static_cast<int>(m_Vdd50State));
}

}}}} // namespace nn::psm::driver::detail
