﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/psm/psm_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_SessionManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

int FindFreeSessionId(bool* isOpenTable, int sessionCountMax) NN_NOEXCEPT
{
    for (auto i = 0; i < sessionCountMax; i++)
    {
        if (!isOpenTable[i])
        {
            return i;
        }
    }

    // 全部オープン済だった
    return -1;
}

}

SessionManager::~SessionManager() NN_NOEXCEPT
{
    // 登録イベントの破棄
    for (auto it = m_StateChangeEventList.begin(); it != m_StateChangeEventList.end(); ++it)
    {
        if (it->IsBoundEvent())
        {
            it->DestroyEvent();
            it->DetachEvent();
        }
    }
    m_StateChangeEventList.clear();
}

::nn::Result SessionManager::OpenSession(Session* pOutSession) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutSession);
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = FindFreeSessionId(m_IsSessionOpen, SessionCountMax);
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmNoAvailableSessions());

    pOutSession->_sessionId = sessionId;
    m_IsSessionOpen[sessionId] = true;

    for (auto i = 0; i < EventType_Count; ++i)
    {
        m_IsEventEnabled[sessionId][i] = true;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SessionManager::CloseSession(const Session& session) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = session._sessionId;
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmInvalidArgument());
    NN_RESULT_THROW_UNLESS(
        m_IsSessionOpen[sessionId],
        ResultPsmInvalidArgument());

    auto& eventHolder = m_EventHolders[sessionId];
    if (eventHolder.IsBoundEvent())
    {
        m_StateChangeEventList.erase(
            m_StateChangeEventList.iterator_to(
                eventHolder));

        eventHolder.DestroyEvent();
        eventHolder.DetachEvent();
    }

    m_IsSessionOpen[sessionId] = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SessionManager::BindStateChangeEvent(
    ::nn::os::SystemEventType* pEvent,
    const Session& session) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = session._sessionId;
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmInvalidArgument());
    NN_RESULT_THROW_UNLESS(
        m_IsSessionOpen[sessionId],
        ResultPsmSessionNotOpen());

    auto& eventHolder = m_EventHolders[sessionId];
    NN_RESULT_THROW_UNLESS(
        !eventHolder.IsBoundEvent(),
        ResultPsmSessionAlreadyBound());

    ::nn::os::CreateSystemEvent(
        pEvent,
        ::nn::os::EventClearMode_ManualClear,
        true);

    eventHolder.AttachEvent(pEvent);

    m_StateChangeEventList.push_back(eventHolder);

    NN_RESULT_SUCCESS;
}

::nn::Result SessionManager::UnbindStateChangeEvent(
    const driver::Session& session) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = session._sessionId;
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmInvalidArgument());
    NN_RESULT_THROW_UNLESS(
        m_IsSessionOpen[sessionId],
        ResultPsmSessionNotOpen());

    auto& eventHolder = m_EventHolders[sessionId];
    NN_RESULT_THROW_UNLESS(
        eventHolder.IsBoundEvent(),
        ResultPsmSessionAlreadyUnbound());

    m_StateChangeEventList.erase(
        m_StateChangeEventList.iterator_to(
            eventHolder));

    eventHolder.DestroyEvent();
    eventHolder.DetachEvent();

    NN_RESULT_SUCCESS;
}

::nn::Result SessionManager::SetChargerTypeChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = session._sessionId;
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmInvalidArgument());

    m_IsEventEnabled[sessionId][EventType_ChargerTypeChange] = isEnabled;

    NN_RESULT_SUCCESS;
}

::nn::Result SessionManager::SetPowerSupplyChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = session._sessionId;
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmInvalidArgument());

    m_IsEventEnabled[sessionId][EventType_PowerSupplyChange] = isEnabled;

    NN_RESULT_SUCCESS;
}

::nn::Result SessionManager::SetBatteryVoltageStateChangeEventEnabled(
    const Session& session,
    bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto sessionId = session._sessionId;
    NN_RESULT_THROW_UNLESS(
        0 <= sessionId && sessionId < SessionCountMax,
        ResultPsmInvalidArgument());

    m_IsEventEnabled[sessionId][EventType_BatteryVoltageStateChange] = isEnabled;

    NN_RESULT_SUCCESS;
}

void SessionManager::SignalAllStateChangeEvents() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (auto&& e : m_StateChangeEventList)
    {
        ::nn::os::SignalSystemEvent(e.GetSystemEvent());
    }
}

void SessionManager::SignalStateChangeEvents(EventType event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(0 <= event && event < EventType_Count);
    ::std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    for (auto i = 0; i < SessionCountMax; ++i)
    {
        if (m_IsSessionOpen[i] && m_EventHolders[i].IsBoundEvent() && m_IsEventEnabled[i][event])
        {
            ::nn::os::SignalSystemEvent(m_EventHolders[i].GetSystemEvent());
        }
    }
}

}}}} // namespace nn::psm::driver::detail
