﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/spl/spl_Api.h>

#include "psm_ErrorReporter.h"
#include "psm_ModuleHolder.h"
#include "psm_PowerStateManager.h"
#include "psm_SupplyRouteManager.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

// サイクリックモード中に充電を開始する電池残量
// HiZ モードをスリープ復帰後すぐ解除して電池残量が 99% を超える可能性があるので
// 割り込みは 99% で発生させて判定は 99% より高めでおこなう
const double SleepChargeStartThresholdPercentage = 99.1;

} // namespace

::nn::Result PowerStateManager::Initialize(
    IChargerDriver* pChargerDriver,
    IFuelGaugeDriver* pFuelGaugeDriver,
    ChargeArbiter* pChargeArbiter,
    BatteryChargeManager* pBatteryChargeManager,
    IBatteryState* pBatteryState,
    SupplyRouteManager* pSupplyRouteManager,
    IFuelGaugeParameterManager* pFuelGaugeParameterManager,
    ModuleHolder* pModuleHolder,
    const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pChargerDriver);
    NN_SDK_ASSERT_NOT_NULL(pFuelGaugeDriver);
    NN_SDK_ASSERT_NOT_NULL(pChargeArbiter);
    NN_SDK_ASSERT_NOT_NULL(pBatteryChargeManager);
    NN_SDK_ASSERT_NOT_NULL(pBatteryState);
    NN_SDK_ASSERT_NOT_NULL(pSupplyRouteManager);
    NN_SDK_ASSERT_NOT_NULL(pFuelGaugeParameterManager);
    NN_SDK_ASSERT_NOT_NULL(pModuleHolder);
    NN_SDK_ASSERT_NOT_NULL(pSettingsHolder);

    m_pChargerDriver = pChargerDriver;
    m_pFuelGaugeDriver = pFuelGaugeDriver;
    m_pFuelGaugeParameterManager = pFuelGaugeParameterManager;
    m_pChargeArbiter = pChargeArbiter;
    m_pBatteryChargeManager = pBatteryChargeManager;
    m_pBatteryState = pBatteryState;
    m_PowerState = PowerState::FullAwake;
    m_pSupplyRouteManager = pSupplyRouteManager;
    m_pModuleHolder = pModuleHolder;
    m_pSettingsHolder = pSettingsHolder;
    m_IsControllerPowerSupplyNeeded = false;

    GetErrorReporter().SetControllerPowerSupplyAcquired(m_IsControllerPowerSupplyNeeded);

    NN_RESULT_SUCCESS;
}

void PowerStateManager::Finalize() NN_NOEXCEPT
{
    // 何もしません。
}

::nn::Result PowerStateManager::AcquireControllerPowerSupply() NN_NOEXCEPT
{
    m_IsControllerPowerSupplyNeeded = true;
    NN_RESULT_DO(m_pSupplyRouteManager->EnableVdd50());

    GetErrorReporter().SetControllerPowerSupplyAcquired(m_IsControllerPowerSupplyNeeded);

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::ReleaseControllerPowerSupply() NN_NOEXCEPT
{
    m_IsControllerPowerSupplyNeeded = false;

    GetErrorReporter().SetControllerPowerSupplyAcquired(m_IsControllerPowerSupplyNeeded);

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::Update(::nn::psc::PmState state, ::nn::psc::PmFlagSet flags) NN_NOEXCEPT
{
    switch (state)
    {
    case ::nn::psc::PmState_FullAwake:
        NN_RESULT_DO(HandleFullAwake(flags));
        break;
    case ::nn::psc::PmState_MinimumAwake:
        NN_RESULT_DO(HandleMinimumAwake(flags));
        break;
    case ::nn::psc::PmState_SleepReady:
        NN_RESULT_DO(HandleSleepReady(flags));
        break;
    case ::nn::psc::PmState_ShutdownReady:
        NN_RESULT_DO(HandleShutdownReady(flags));
        break;
    default:
        break;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::HandleFullAwake(
    ::nn::psc::PmFlagSet flags) NN_NOEXCEPT
{
    NN_UNUSED(flags);

    auto previousPowerState = m_PowerState;

    NN_RESULT_DO(m_pChargeArbiter->SetPowerState(PowerState::FullAwake));
    NN_RESULT_DO(EnterAwakeState());
    NN_RESULT_DO(SetupInterruptAndVdd50());

    if (previousPowerState != PowerState::FullAwake)
    {
        m_pBatteryChargeManager->Update(false);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::HandleMinimumAwake(
    ::nn::psc::PmFlagSet flags) NN_NOEXCEPT
{
    NN_UNUSED(flags);

    // アボートなどで充電できない状態を避けるため
    // 他のどの処理にも先行して HiZ モードを解除する
    NN_RESULT_DO(DisableHiZModeInterrupt());

    // 以前の状態が Awake だったか Sleep だったかが必要なので
    // m_PowerState は PowerState::MinimumAwake にしない

    m_pModuleHolder->Activate();

    NN_RESULT_DO(m_pChargeArbiter->SetPowerState(PowerState::MinimumAwake));
    // MinimumAwake では前回の VDD50 の状態を引き継ぐ

    NN_RESULT_DO(m_pModuleHolder->AllowOtg());

    m_pBatteryChargeManager->Update(true);

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::HandleSleepReady(
    ::nn::psc::PmFlagSet flags) NN_NOEXCEPT
{
    NN_UNUSED(flags);

    NN_RESULT_DO(m_pModuleHolder->DisallowOtg());

    NN_RESULT_DO(UpdateState());

    m_pModuleHolder->Deactivate();

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::HandleShutdownReady(
    ::nn::psc::PmFlagSet flags) NN_NOEXCEPT
{
    NN_UNUSED(flags);

    NN_RESULT_DO(m_pChargeArbiter->SetPowerState(PowerState::ShutdownChargeMain));

    NN_RESULT_DO(m_pModuleHolder->DisallowOtg());

    NN_RESULT_DO(m_pFuelGaugeParameterManager->SaveParameter());
    NN_RESULT_DO(m_pFuelGaugeParameterManager->FlushParameter());

    m_pModuleHolder->Deactivate();

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::UpdateState() NN_NOEXCEPT
{
    // 以前の状態が Awake だったか Sleep だったかが必要なので
    // m_PowerState は PowerState::MinimumAwake にしない
    NN_SDK_ASSERT(m_PowerState != PowerState::MinimumAwake);

    // 消費電力規制の対応のためスリープ中に充放電を繰り返す
    switch (m_PowerState)
    {
    case PowerState::FullAwake:
        // SleepCharge, SleepDischarge のうち適切な方に遷移する
        NN_RESULT_DO(EnterSleepChargeState());
        NN_RESULT_DO(TryEnterSleepDischargeState());
        break;
    case PowerState::SleepCharge:
        NN_RESULT_DO(TryEnterSleepDischargeState());
        break;
    case PowerState::SleepDischarge:
        NN_RESULT_DO(TryEnterSleepChargeState());
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    SetupInterruptAndVdd50();

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::TryEnterSleepDischargeState() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBatteryState);
    NN_SDK_ASSERT_NOT_NULL(m_pFuelGaugeDriver);

    // 下記の AND 条件が成立する場合に SleepDischarge に遷移する
    // - スリープ状態
    // - 充電器が接続されている
    // - 本体の電池が満充電
    // - ジョイコンが未装着もしくはジョイコンの電池が満充電
    auto chargerType = m_pBatteryState->GetChargerType();
    // 充電器が接続されている
    if (chargerType == ChargerType_EnoughPower
        || chargerType == ChargerType_LowPower)
    {
        IChargerDriver::Status status;
        NN_RESULT_DO(m_pChargerDriver->GetStatus(&status));
        // 本体の電池が満充電
        if (status == IChargerDriver::Status::ChargeTermination)
        {
            // ジョイコンが未装着もしくはジョイコンの電池が満充電
            if (!m_IsControllerPowerSupplyNeeded)
            {
                NN_RESULT_DO(EnterSleepDischargeState());
            }
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::TryEnterSleepChargeState() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pBatteryState);
    NN_SDK_ASSERT_NOT_NULL(m_pFuelGaugeDriver);

    // 本体の電池残量が一定値以下の場合に SleepCharge に遷移する
    auto batteryCharge = m_pBatteryState->GetRawBatteryChargePercentage();
    if (batteryCharge < SleepChargeStartThresholdPercentage)
    {
        NN_RESULT_DO(EnterSleepChargeState());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::EnterAwakeState() NN_NOEXCEPT
{
    m_PowerState = PowerState::FullAwake;
    NN_DETAIL_PSM_TRACE_V1("EnterAwakeState.\n");

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::EnterSleepDischargeState() NN_NOEXCEPT
{
    m_PowerState = PowerState::SleepDischarge;
    NN_DETAIL_PSM_TRACE_V1("EnterSleepDischargeState.\n");

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::EnterSleepChargeState() NN_NOEXCEPT
{
    m_PowerState = PowerState::SleepCharge;
    NN_DETAIL_PSM_TRACE_V1("EnterSleepChargeState.\n");

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::SetupInterruptAndVdd50() NN_NOEXCEPT
{
    switch (m_PowerState)
    {
    case PowerState::FullAwake:
        NN_RESULT_DO(DisableHiZModeInterrupt());
        NN_RESULT_DO(DisableLowBatteryInterrupt());
        NN_RESULT_DO(SetVdd50Enabled(true));
        break;
    case PowerState::SleepCharge:
        NN_RESULT_DO(DisableHiZModeInterrupt());
        NN_RESULT_DO(EnableLowBatteryInterrupt());
        NN_RESULT_DO(SetVdd50Enabled(m_IsControllerPowerSupplyNeeded));
        break;
    case PowerState::SleepDischarge:
        NN_RESULT_DO(EnableHiZModeInterrupt());
        NN_RESULT_DO(EnableLowBatteryInterrupt());
        NN_RESULT_DO(SetVdd50Enabled(false));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::EnableHiZModeInterrupt() NN_NOEXCEPT
{
    // psm では HiZ モードにせずセキュアモニタでスリープ直前に HiZ モードにする
    if ( m_pSettingsHolder->IsChargerDriverEnabled() )
    {
        nn::spl::SetConfig(nn::spl::ConfigItem_IsChargerHiZModeEnabled, 1);
    }
    NN_RESULT_DO(
        m_pFuelGaugeDriver->SetLowChargeThreshold(
            static_cast<int>(SleepChargeStartThresholdPercentage)));
    NN_RESULT_DO(m_pFuelGaugeDriver->DisableHighChargeInterrupt());
    NN_RESULT_DO(m_pFuelGaugeDriver->ClearChargeInterrupt());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::DisableHiZModeInterrupt() NN_NOEXCEPT
{
    // 起床時にセキュアモニタで HiZ モードを解除するが念のため psm でも HiZ モードを解除する
    NN_RESULT_DO(m_pChargerDriver->ExitHiZMode());
    if ( m_pSettingsHolder->IsChargerDriverEnabled() )
    {
        nn::spl::SetConfig(nn::spl::ConfigItem_IsChargerHiZModeEnabled, 0);
    }
    NN_RESULT_DO(m_pFuelGaugeDriver->DisableLowChargeInterrupt());

    // 電池残量が一定以上の場合、低速の充電から高速の充電に切り替える為に割込みの閾値を設定する。
    const double PowerSupplyStableThresholdPercentage = 3.0;
    if ( m_pBatteryState->GetRawBatteryChargePercentage() < PowerSupplyStableThresholdPercentage )
    {
        NN_RESULT_DO(m_pFuelGaugeDriver->SetHighChargeThreshold(static_cast<int>(PowerSupplyStableThresholdPercentage)));
    }
    else
    {
        NN_RESULT_DO(m_pFuelGaugeDriver->DisableHighChargeInterrupt());
    }

    NN_RESULT_DO(m_pFuelGaugeDriver->ClearChargeInterrupt());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::EnableLowBatteryInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(
        m_pFuelGaugeDriver->SetLowVoltageThreshold(
            BatteryState::ShutdownVoltageThresholdMilliVolt));
    NN_RESULT_DO(m_pFuelGaugeDriver->ClearVoltageInterrupt());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::DisableLowBatteryInterrupt() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pFuelGaugeDriver->SetLowVoltageThreshold(0));
    NN_RESULT_DO(m_pFuelGaugeDriver->ClearVoltageInterrupt());

    NN_RESULT_SUCCESS;
}

::nn::Result PowerStateManager::SetVdd50Enabled(bool isVdd50Enabled) NN_NOEXCEPT
{
    if (isVdd50Enabled)
    {
        NN_RESULT_DO(m_pSupplyRouteManager->EnableVdd50());
    }
    else
    {
        NN_RESULT_DO(m_pSupplyRouteManager->DisableVdd50());
    }

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
