﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

#include <nn/psc/psc_PmModule.h>
#include <nn/psm/psm_SystemTypes.h>
#include <nn/psm/psm_SystemProcessTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include "Handlers/psm_ChargerEventHandler.h"
#include "Handlers/psm_FuelGaugeEventHandler.h"
#include "Handlers/psm_PeriodicCustomParameterSaver.h"
#include "Handlers/psm_PeriodicStateUpdater.h"
#include "Handlers/psm_PeriodicSystemReportSaver.h"
#include "Handlers/psm_SystemPowerStateEventHandler.h"
#include "Handlers/psm_SystemPowerStateEventHandlerLow.h"
#include "Handlers/psm_UsbPowerDeliveryEventHandler.h"
#include "Handlers/psm_UsbPowerEventHandler.h"
#include "psm_BatteryChargeManager.h"
#include "psm_EventMonitor.h"
#include "psm_BatteryTemperatureManager.h"
#include "psm_BatteryVoltageLevelForCharge.h"
#include "psm_ChargeArbiter.h"
#include "psm_FuelGaugeParameterManager.h"
#include "psm_FuelGaugeParameterManagerStub.h"
#include "psm_IBatteryState.h"
#include "psm_IChargerDriver.h"
#include "psm_IFuelGaugeParameterManager.h"
#include "psm_ISupplyRouteDriver.h"
#include "psm_IUsbPowerManager.h"
#include "psm_ModuleHolder.h"
#include "psm_OverlayNotificationSender.h"
#include "psm_PowerStateManager.h"
#include "psm_SessionManager.h"
#include "psm_SettingsHolder-spec.nx.h"
#include "psm_SupplyRouteManager.h"
#include "psm_UsbPowerManager.h"
#include "psm_UsbPowerManagerStub.h"

namespace nn { namespace psm { namespace driver { namespace detail {

ModuleHolder::ModuleHolder() NN_NOEXCEPT
    : m_IsActive(false)
    , m_pSettingsHolder(nullptr)
    , m_ChargerEventHandler()
    , m_FuelGaugeEventHandler()
    , m_PeriodicCustomParameterSaver()
    , m_PeriodicStateUpdater()
    , m_PeriodicSystemReportSaver()
    , m_UsbPowerDeliveryEventHandler()
    , m_UsbPowerEventHandler()
    , m_SystemPowerStateEventHandler()
    , m_SystemPowerStateEventHandlerLow()
    , m_PowerStateManager()
    , m_TemperatureManager()
    , m_BatteryChargeManager()
    , m_pFuelGaugeParameterManager(nullptr)
    , m_FuelGaugeParameterManagerStorage()
    , m_pUsbPowerManager(nullptr)
    , m_UsbPowerManagerStorage()
    , m_BatteryState()
    , m_SupplyRouteManager()
    , m_NotificationSender()
    , m_pBatteryEventMonitor(nullptr)
    , m_pEventMonitorForUsbPowerDelivery(nullptr)
    , m_pFuelGaugeDriver(nullptr)
{
    // 何もしない
}

ModuleHolder::~ModuleHolder() NN_NOEXCEPT
{
    // 何もしない
}

void ModuleHolder::Initialize(
    IChargerDriver* pChargerDriver,
    IFuelGaugeDriver* pFuelGaugeDriver,
    ISupplyRouteDriver* pSupplyRouteDriver,
    ChargeArbiter* pChargeArbiter,
    SessionManager* pSessionManager,
    EventMonitor* pBatteryEventMonitor,
    EventMonitor* pEventMonitorForUsbPowerDelivery,
    const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pChargerDriver);
    NN_SDK_ASSERT_NOT_NULL(pFuelGaugeDriver);
    NN_SDK_ASSERT_NOT_NULL(pSupplyRouteDriver);
    NN_SDK_ASSERT_NOT_NULL(pChargeArbiter);
    NN_SDK_ASSERT_NOT_NULL(pSessionManager);
    NN_SDK_ASSERT_NOT_NULL(pBatteryEventMonitor);
    NN_SDK_ASSERT_NOT_NULL(pEventMonitorForUsbPowerDelivery);
    NN_SDK_ASSERT_NOT_NULL(pSettingsHolder);

    if ( pSettingsHolder->IsUsbPowerManagementSupported() )
    {
        m_pUsbPowerManager = new (&(m_UsbPowerManagerStorage.driver)) UsbPowerManager();
    }
    else
    {
        m_pUsbPowerManager = new (&(m_UsbPowerManagerStorage.stub)) UsbPowerManagerStub();
    }

    if ( pSettingsHolder->HasBattery() )
    {
        m_pFuelGaugeParameterManager = new (&(m_FuelGaugeParameterManagerStorage.driver)) FuelGaugeParameterManager();
    }
    else
    {
        m_pFuelGaugeParameterManager = new (&(m_FuelGaugeParameterManagerStorage.stub)) FuelGaugeParameterManagerStub();
    }

    // ErrorReporter 初期値設定
    GetErrorReporter().SetPowerRole(::nn::usb::UsbPowerRole_Unknown);
    GetErrorReporter().SetPowerSupplyType(::nn::usb::UsbChargerType_Unknown);
    GetErrorReporter().SetPowerSupplyVoltage(0);
    GetErrorReporter().SetPowerSupplyCurrent(0);
    GetErrorReporter().SetOtgRequested(false);

    m_IsActive = false;
    m_pFuelGaugeDriver = pFuelGaugeDriver;
    m_pBatteryEventMonitor = pBatteryEventMonitor;
    m_pEventMonitorForUsbPowerDelivery = pEventMonitorForUsbPowerDelivery;
    m_pSettingsHolder = pSettingsHolder;

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BatteryState.Initialize());

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_SupplyRouteManager.Initialize(pSupplyRouteDriver, m_pSettingsHolder));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_NotificationSender.Initialize(&m_BatteryState));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_BatteryChargeManager.Initialize(
            pFuelGaugeDriver,
            &m_BatteryState,
            &m_NotificationSender,
            m_pSettingsHolder));

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_TemperatureManager.Initialize(
            pFuelGaugeDriver,
            pChargeArbiter));

    m_FuelGaugeEventHandler.Initialize(&m_TemperatureManager, pFuelGaugeDriver);

    m_PeriodicStateUpdater.Initialize(pChargeArbiter, &m_BatteryChargeManager, &m_BatteryState, pChargerDriver, pSessionManager);

    NN_ABORT_UNLESS_RESULT_SUCCESS(pFuelGaugeDriver->EnableInterrupt());

    m_pUsbPowerManager->Initialize(pChargeArbiter, &m_SupplyRouteManager, &m_BatteryState, pChargerDriver, pSessionManager, &m_NotificationSender, m_pSettingsHolder);

    m_UsbPowerDeliveryEventHandler.Initialize(m_pUsbPowerManager);

    m_UsbPowerEventHandler.Initialize(m_pUsbPowerManager);

    m_ChargerEventHandler.Initialize(m_pUsbPowerManager, pChargerDriver);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pFuelGaugeParameterManager->Initialize(pFuelGaugeDriver));

    m_PeriodicCustomParameterSaver.Initialize(m_pFuelGaugeParameterManager);

    m_PeriodicSystemReportSaver.Initialize(pFuelGaugeDriver);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        m_PowerStateManager.Initialize(
            pChargerDriver,
            pFuelGaugeDriver,
            pChargeArbiter,
            &m_BatteryChargeManager,
            &m_BatteryState,
            &m_SupplyRouteManager,
            m_pFuelGaugeParameterManager,
            this,
            m_pSettingsHolder));

    m_SystemPowerStateEventHandler.Initialize(&m_PowerStateManager);

    m_SystemPowerStateEventHandlerLow.Initialize(pSettingsHolder);

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pUsbPowerManager->AllowOtg());

    if ( m_pSettingsHolder->HasBattery() )
    {
        m_PeriodicCustomParameterSaver.StartPeriodicTimerEvent();
        m_PeriodicStateUpdater.StartPeriodicTimerEvent();
        m_PeriodicSystemReportSaver.StartPeriodicTimerEvent();
    }

    m_pBatteryEventMonitor->LinkEventHandler(&m_SystemPowerStateEventHandler);

    m_pBatteryEventMonitor->LinkEventHandler(&m_SystemPowerStateEventHandlerLow);

    Activate();
}

void ModuleHolder::Finalize() NN_NOEXCEPT
{
    Deactivate();

    m_pBatteryEventMonitor->UnlinkEventHandler(&m_SystemPowerStateEventHandlerLow);

    m_pBatteryEventMonitor->UnlinkEventHandler(&m_SystemPowerStateEventHandler);

    if ( m_pSettingsHolder->HasBattery() )
    {
        m_PeriodicStateUpdater.StopTimerEvent();
        m_PeriodicCustomParameterSaver.StopTimerEvent();
        m_PeriodicSystemReportSaver.StopTimerEvent();
    }

    m_SystemPowerStateEventHandlerLow.Finalize();

    m_SystemPowerStateEventHandler.Finalize();

    m_PowerStateManager.Finalize();

    m_PeriodicSystemReportSaver.Finalize();

    m_PeriodicCustomParameterSaver.Finalize();

    m_ChargerEventHandler.Finalize();

    m_UsbPowerEventHandler.Finalize();

    m_UsbPowerDeliveryEventHandler.Finalize();

    m_pUsbPowerManager->Finalize();

    m_pFuelGaugeDriver->DisableInterrupt();

    m_PeriodicStateUpdater.Finalize();

    m_FuelGaugeEventHandler.Finalize();

    m_TemperatureManager.Finalize();
    m_BatteryChargeManager.Finalize();
    m_NotificationSender.Finalize();
    m_SupplyRouteManager.Finalize();
    m_BatteryState.Finalize();

    m_pFuelGaugeParameterManager->~IFuelGaugeParameterManager();
    m_pFuelGaugeParameterManager = nullptr;

    m_pUsbPowerManager->~IUsbPowerManager();
    m_pUsbPowerManager = nullptr;
}

void ModuleHolder::Activate() NN_NOEXCEPT
{
    if ( !m_IsActive )
    {
        // 先に登録された EventHandler の方が優先度が高くなります。
        // TODO: 優先度の概念が無いので追加順番に依存して優先度が決まります。優先度を指定できるようにするべきです。
        if ( m_pSettingsHolder->HasBattery() )
        {
            m_pBatteryEventMonitor->LinkEventHandler(&m_FuelGaugeEventHandler);
            m_pBatteryEventMonitor->LinkEventHandler(&m_ChargerEventHandler);
            m_pBatteryEventMonitor->LinkEventHandler(&m_PeriodicStateUpdater);
            m_pBatteryEventMonitor->LinkEventHandler(&m_PeriodicCustomParameterSaver);
            m_pBatteryEventMonitor->LinkEventHandler(&m_PeriodicSystemReportSaver);
        }

        if ( m_pSettingsHolder->IsUsbPowerManagementSupported() )
        {
            m_pBatteryEventMonitor->LinkEventHandler(&m_UsbPowerEventHandler);
            m_pEventMonitorForUsbPowerDelivery->LinkEventHandler(&m_UsbPowerDeliveryEventHandler);
        }

        m_IsActive = true;
    }
}

void ModuleHolder::Deactivate() NN_NOEXCEPT
{
    if ( m_IsActive )
    {
        if ( m_pSettingsHolder->HasBattery() )
        {
            m_pBatteryEventMonitor->UnlinkEventHandler(&m_FuelGaugeEventHandler);
            m_pBatteryEventMonitor->UnlinkEventHandler(&m_ChargerEventHandler);
            m_pBatteryEventMonitor->UnlinkEventHandler(&m_PeriodicStateUpdater);
            m_pBatteryEventMonitor->UnlinkEventHandler(&m_PeriodicCustomParameterSaver);
            m_pBatteryEventMonitor->UnlinkEventHandler(&m_PeriodicSystemReportSaver);
        }

        if ( m_pSettingsHolder->IsUsbPowerManagementSupported() )
        {
            m_pBatteryEventMonitor->UnlinkEventHandler(&m_UsbPowerEventHandler);
            m_pEventMonitorForUsbPowerDelivery->UnlinkEventHandler(&m_UsbPowerDeliveryEventHandler);
        }

        m_IsActive = false;
    }
}

::nn::Result ModuleHolder::GetBatteryChargePercentage(int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_BatteryState.GetBatteryChargePercentage();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetRawBatteryChargePercentage(double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_BatteryState.GetRawBatteryChargePercentage();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetBatteryAgePercentage(double* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_BatteryState.GetBatteryAgePercentage();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetChargerType(ChargerType* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_BatteryState.GetChargerType();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetBatteryVoltageState(BatteryVoltageState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_BatteryState.GetBatteryVoltageState();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::AcquireControllerPowerSupply() NN_NOEXCEPT
{
    NN_RESULT_DO(m_PowerStateManager.AcquireControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::ReleaseControllerPowerSupply() NN_NOEXCEPT
{
    NN_RESULT_DO(m_PowerStateManager.ReleaseControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::AllowOtg() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pUsbPowerManager->AllowOtg());

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::DisallowOtg() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pUsbPowerManager->DisallowOtg());

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::EnableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pUsbPowerManager->EnableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::DisableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pUsbPowerManager->DisableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::IsEnoughPowerSupplied(bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutValue);

    *pOutValue = m_BatteryState.IsEnoughPowerSupplied();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetBatteryChargeCalibratedEvent(::nn::os::SystemEventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    *pOutEventPointer = m_BatteryState.GetBatteryChargeCalibratedEvent();

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetPmModuleForTest(::nn::psc::PmModule** pOutPmModulePointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutPmModulePointer);

    NN_RESULT_DO(m_SystemPowerStateEventHandler.GetPmModuleForTest(pOutPmModulePointer));

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetPscEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_SystemPowerStateEventHandler.GetPscEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetFuelGaugeEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_FuelGaugeEventHandler.GetFuelGaugeInterruptForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetFuelGaugeTimerEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_PeriodicStateUpdater.GetPeriodicStateUpdateEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetUsbPdEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_UsbPowerDeliveryEventHandler.GetEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetUsbPmEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_UsbPowerEventHandler.GetUsbPmEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

::nn::Result ModuleHolder::GetEnoughPowerChargeEmulationEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    NN_RESULT_DO(m_UsbPowerEventHandler.GetEnoughPowerChargeEmulationEventForTest(pOutEventPointer));

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail
