﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/os/os_SdkMutex.h>

#include "Handlers/psm_EventHandler.h"
#include "psm_ErrorReporter.h"

namespace nn { namespace psm { namespace driver { namespace detail {

struct EventHandlerNode final
{
    bool valid;
    EventHandler* pEventHandler;

    void Clear() NN_NOEXCEPT
    {
        valid = false;
        pEventHandler = nullptr;
    }
};

//! イベントを受けて事前に指定されたハンドラに渡すクラスです。
class EventMonitor final
{
    NN_DISALLOW_COPY(EventMonitor);
    NN_DISALLOW_MOVE(EventMonitor);

private:
    static const int MaxEventHandler = 20;

public:
    EventMonitor() NN_NOEXCEPT;

    ~EventMonitor() NN_NOEXCEPT;

    //! 初期化します。
    void Initialize(int threadPriority, const char* pThreadName, void* pStack, size_t stackSize) NN_NOEXCEPT;

    //! 終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! ハンドラをリンクします。
    void LinkEventHandler(EventHandler* pEventHandler) NN_NOEXCEPT;

    //! ハンドラをアンリンクします。
    void UnlinkEventHandler(EventHandler* pEventHandler) NN_NOEXCEPT;

private:
    static void ThreadFunc(void* value) NN_NOEXCEPT;

    void InvokeEventLoop() NN_NOEXCEPT;

    bool HandleEvents() NN_NOEXCEPT;

private:
    //! スレッドオブジェクト
    ::nn::os::ThreadType m_Thread;

    //! 停止イベント
    ::nn::os::SystemEventType m_TerminationEvent;
    ::nn::os::MultiWaitHolderType m_TerminationMultiWaitHolder;

    //! 多重待ちオブジェクト
    ::nn::os::MultiWaitType m_MultiWait;

    //! イベントハンドラのリスト
    EventHandlerNode m_EventHandlerList[MaxEventHandler];

    //! EventMonitor から EventHolder を安全に Link/Unlink するための同期イベント
    ::nn::os::EventType m_ThreadSyncEvent;
    ::nn::os::MultiWaitHolderType m_ThreadSyncMultiWaitHolder;

    //! EventMonitor から EventHolder を安全に Link/Unlink するためのバリア
    ::nn::os::BarrierType m_ThreadSyncBarrier;

    //! EventMonitor オブジェクトの Link/Unlink 操作の排他制御を行う Mutex
    ::nn::os::SdkMutex m_LinkHandlerMutex;
};

}}}} // namespace nn::psm::driver::detail
