﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_BatteryChargeManager.h"
#include "psm_OverlayNotificationSender.h"
#include "psm_SettingsHolder-spec.nx.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

const int NotificationThresholdsPercentage[] =
{
    1,
    5,
    15,
};

} // namespace

BatteryChargeManager::BatteryChargeManager() NN_NOEXCEPT
    : m_pFuelGaugeDriver()
    , m_pBatteryState()
    , m_BatteryChargeLevel(BatteryChargeLevel::Enough)
    , m_pNotificationSender()
{
    // 何もしない
}

BatteryChargeManager::~BatteryChargeManager() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result BatteryChargeManager::Initialize(
    IFuelGaugeDriver* pFuelGaugeDriver,
    BatteryState* pBatteryState,
    OverlayNotificationSender* pNotificationSender,
    const SettingsHolder* pSettingsHolder) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFuelGaugeDriver);
    NN_SDK_ASSERT_NOT_NULL(pBatteryState);
    NN_SDK_ASSERT_NOT_NULL(pNotificationSender);

    m_pFuelGaugeDriver = pFuelGaugeDriver;
    m_pBatteryState = pBatteryState;
    m_pNotificationSender = pNotificationSender;
    m_pSettingsHolder = pSettingsHolder;

    m_BatteryChargeLevel = BatteryChargeLevel::Enough;

    // 起動直後にも正しい状態を取得できるように更新しておく
    UpdateBatteryCharge();

    NN_RESULT_SUCCESS;
}

void BatteryChargeManager::Finalize() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result BatteryChargeManager::Update(bool isPreviousLevelUsed) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pNotificationSender);

    UpdateBatteryCharge();

    auto batteryCharge = m_pBatteryState->GetBatteryChargePercentage();
    auto currentLevel = QuantizeBatteryCharge(batteryCharge);
    auto previousLevel = m_BatteryChargeLevel;

    m_BatteryChargeLevel = currentLevel;

    if (currentLevel != BatteryChargeLevel::Enough)
    {
        auto needsNotification = false;
        if (isPreviousLevelUsed)
        {
            // 電池残量が低くなる方向の電池残量レベルの変更時のみ通知が必要
            if (static_cast<int>(currentLevel) < static_cast<int>(previousLevel))
            {
                needsNotification = true;
            }
        }
        else
        {
            needsNotification = true;
        }

        if (needsNotification)
        {
            m_pNotificationSender->NotifyLowBattery();
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result BatteryChargeManager::UpdateBatteryCharge() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pFuelGaugeDriver);

    double rawBatteryChargePercentage;
    NN_RESULT_DO(
        m_pFuelGaugeDriver->GetBatteryChargePercentage(&rawBatteryChargePercentage));
    m_pBatteryState->SetRawBatteryChargePercentage(rawBatteryChargePercentage);

    double batteryAgePercentage;
    NN_RESULT_DO(
        m_pFuelGaugeDriver->GetBatteryAgePercentage(&batteryAgePercentage));
    m_pBatteryState->SetBatteryAgePercentage(batteryAgePercentage);

    int rawBatteryChargeMilliVoltage;
    NN_RESULT_DO(
        m_pFuelGaugeDriver->GetBatteryChargeMilliVoltage(&rawBatteryChargeMilliVoltage));
    m_pBatteryState->SetRawBatteryChargeMilliVoltage(rawBatteryChargeMilliVoltage);

    int averageCurrentMilliAmpere;
    NN_RESULT_DO(m_pFuelGaugeDriver->GetAverageCurrentMilliAmpere(&averageCurrentMilliAmpere));
    m_pBatteryState->SetAverageCurrentMilliAmpere(averageCurrentMilliAmpere);

    double voltageFuelGaugePercentage;
    NN_RESULT_DO(m_pFuelGaugeDriver->GetVoltageFuelGaugePercentage(&voltageFuelGaugePercentage));
    m_pBatteryState->SetVoltageFuelGaugePercentage(voltageFuelGaugePercentage);

    if ( m_pSettingsHolder->IsEvaluationLogEnabled() )
    {
        int fullCapacity;
        NN_RESULT_DO(m_pFuelGaugeDriver->GetFullCapacityMilliAmpereH(&fullCapacity));

        int remainingCapacity;
        NN_RESULT_DO(m_pFuelGaugeDriver->GetRemainingCapacityMilliAmpereH(&remainingCapacity));

        nn::os::Tick tick       = nn::os::GetSystemTick();
        int32_t hours           = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetHours());
        int32_t minutes         = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetMinutes() % 60);
        int32_t seconds         = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetSeconds() % 60);
        int32_t milliSeconds    = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetMilliSeconds() % 1000);

        // 浮動小数点を print すると極端にスタックを消費するので少数点以下は整数値で表現する。
        int rawBatteryChargePercentageInte = static_cast<int>(rawBatteryChargePercentage);
        int rawBatteryChargePercentageFrac = static_cast<int>(rawBatteryChargePercentage * 100) % 100;

        int batteryAgePercentageInte = static_cast<int>(batteryAgePercentage);
        int batteryAgePercentageFrac = static_cast<int>(batteryAgePercentage * 100) % 100;

        int voltageFuelGaugePercentageInte = static_cast<int>(voltageFuelGaugePercentage);
        int voltageFuelGaugePercentageFrac = static_cast<int>(voltageFuelGaugePercentage * 100) % 100;

        NN_DETAIL_PSM_TRACE("[evaluation], %d:%02d:%02d.%03d, Bat(%):%d.%02d, Bat(mV):%d, Age(%):%d.%02d, Iavg(mA):%d, SOCVF(%):%d.%02d, FullCap(mAh):%d, RemCapREP(mAh):%d\n",
            hours,
            minutes,
            seconds,
            milliSeconds,
            rawBatteryChargePercentageInte,
            rawBatteryChargePercentageFrac,
            rawBatteryChargeMilliVoltage,
            batteryAgePercentageInte,
            batteryAgePercentageFrac,
            averageCurrentMilliAmpere,
            voltageFuelGaugePercentageInte,
            voltageFuelGaugePercentageFrac,
            fullCapacity,
            remainingCapacity);

        // Default ビルド時の未使用変数の警告回避。
        NN_UNUSED(tick);
        NN_UNUSED(hours);
        NN_UNUSED(minutes);
        NN_UNUSED(seconds);
        NN_UNUSED(milliSeconds);

        NN_UNUSED(rawBatteryChargePercentageInte);
        NN_UNUSED(rawBatteryChargePercentageFrac);

        NN_UNUSED(batteryAgePercentageInte);
        NN_UNUSED(batteryAgePercentageFrac);

        NN_UNUSED(voltageFuelGaugePercentageInte);
        NN_UNUSED(voltageFuelGaugePercentageFrac);
    }

    NN_RESULT_SUCCESS;
}

BatteryChargeManager::BatteryChargeLevel
BatteryChargeManager::QuantizeBatteryCharge(int percentage) NN_NOEXCEPT
{
    const BatteryChargeLevel levels[] =
    {
        BatteryChargeLevel::Low1Percent,
        BatteryChargeLevel::Low5Percent,
        BatteryChargeLevel::Low15Percent,
    };
    const auto size = sizeof(levels) / sizeof(levels[0]);
    NN_STATIC_ASSERT(
        size == sizeof(NotificationThresholdsPercentage) / sizeof(NotificationThresholdsPercentage[0]));

    for (auto i = 0; i < size; ++i)
    {
        if (percentage <= NotificationThresholdsPercentage[i])
        {
            return levels[i];
        }
    }
    return BatteryChargeLevel::Enough;
}

}}}} // namespace nn::psm::driver::detail
