﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ファイル名等での実装分岐は必要になったときに準備する

#include <algorithm>

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/psm/detail/psm_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Ptm.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/srepo/srepo_SystemReport.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_SteadyClockTimePoint.h>

#include "../psm_IFuelGaugeDriver.h"
#include "psm_PeriodicSystemReportSaver.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

// 一週間経過するごと保存する
const ::nn::TimeSpan IntervalTimeSpan = ::nn::TimeSpan::FromHours(7 * 24);

} // namespace

PeriodicSystemReportSaver::PeriodicSystemReportSaver() NN_NOEXCEPT
    : m_pFuelGaugeDriver()
    , m_TimerEvent()
    , m_MultiWaitHolder()
    , m_Buffer()
{
    // 何もしない
}

PeriodicSystemReportSaver::~PeriodicSystemReportSaver() NN_NOEXCEPT
{
    // 何もしない
}

void PeriodicSystemReportSaver::Initialize(IFuelGaugeDriver* pFuelGaugeDriver) NN_NOEXCEPT
{
    m_pFuelGaugeDriver = pFuelGaugeDriver;

    ::nn::os::InitializeTimerEvent(&m_TimerEvent, ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_TimerEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::time::Initialize());
}

void PeriodicSystemReportSaver::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_PSM_ERROR_UNLESS_RESULT_SUCCESS(::nn::time::Finalize());

    ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);
    ::nn::os::FinalizeTimerEvent(&m_TimerEvent);
}

void PeriodicSystemReportSaver::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_MultiWaitHolder);
}

void PeriodicSystemReportSaver::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
}

void PeriodicSystemReportSaver::StartPeriodicTimerEvent() NN_NOEXCEPT
{
    ::nn::os::StartPeriodicTimerEvent(&m_TimerEvent, ::nn::TimeSpan(0), IntervalTimeSpan);
}

void PeriodicSystemReportSaver::StopTimerEvent() NN_NOEXCEPT
{
    ::nn::os::StopTimerEvent(&m_TimerEvent);
}

bool PeriodicSystemReportSaver::HandleEvent() NN_NOEXCEPT
{
    if ( ::nn::os::TryWaitTimerEvent(&m_TimerEvent) )
    {
        ::nn::os::ClearTimerEvent(&m_TimerEvent);

        SaveSystemReport();

        return true;
    }

    return false;
}

void PeriodicSystemReportSaver::SaveSystemReport() NN_NOEXCEPT
{
    // FuelGaugeParameterManager の Save 処理と近い処理だけどもオブジェクト間の依存を増やしたくないので独自に書く

    ::nn::settings::system::InitialLaunchSettings initialLaunchSettings;
    ::nn::settings::system::GetInitialLaunchSettings(&initialLaunchSettings);

    // 初回起動設定が完了している
    if ( !(initialLaunchSettings.flags.Test<::nn::settings::system::InitialLaunchFlag::IsCompleted>()) )
    {
        NN_DETAIL_PSM_INFO("InitialLaunch not completed.\n");
        return;
    }

    // 初回起動設定完了 TimeStamp が保存されている
    if ( !(initialLaunchSettings.flags.Test<::nn::settings::system::InitialLaunchFlag::HasTimeStamp>()) )
    {
        NN_DETAIL_PSM_INFO("TimeStamp of InitialLaunch not found.\n");
        return;
    }

    ::nn::time::SteadyClockTimePoint currentSteadyClockTimePoint;
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::time::StandardSteadyClock::GetCurrentTimePoint(&currentSteadyClockTimePoint));

    // int64_t の秒数
    int64_t seconds;
    auto result = nn::time::GetSpanBetween(&seconds, initialLaunchSettings.timeStamp, currentSteadyClockTimePoint);
    if ( !result.IsSuccess() )
    {
        // RTCリセット時など、SteadyClockTimePoint 同士を比較できない場合
        seconds = -1; // 不正値として -1 を入れる
    }

    // 電池残量計から初期値を考慮された cycles を取得する
    int cycles;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pFuelGaugeDriver->GetCycles(&cycles));

    // settings::system から parameter を取得する
    ::nn::settings::system::PtmFuelGaugeParameter ptmFuelGaugeParameter;
    ::nn::settings::system::GetPtmFuelGaugeParameter(&ptmFuelGaugeParameter);

    // battery_age の取得
    double batteryAgePercentage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pFuelGaugeDriver->GetBatteryAgePercentage(&batteryAgePercentage));

    // max_temperature の取得
    int maxTemperatureCelsius;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_pFuelGaugeDriver->GetMaxTemperatureCelsius(&maxTemperatureCelsius));

    // cycles の合計を SystemReport に保存する
    ::nn::srepo::SystemReport systemReport("battery_info");
    const ::nn::ApplicationId ApplicationId = { 0x0100000000000010ULL };
    NN_ABORT_UNLESS_RESULT_SUCCESS(systemReport.SetApplicationId(ApplicationId)); // ptm
    systemReport.SetBuffer(&m_Buffer, sizeof(m_Buffer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(systemReport.Add("elapsed_time_since_initial_boot_seconds", seconds));
    NN_ABORT_UNLESS_RESULT_SUCCESS(systemReport.Add("cycles", static_cast<int64_t>(cycles + ptmFuelGaugeParameter.cycles)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(systemReport.Add("battery_age_percentage", batteryAgePercentage));
    NN_ABORT_UNLESS_RESULT_SUCCESS(systemReport.Add("max_temperature_celsius", static_cast<int64_t>(maxTemperatureCelsius)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(systemReport.Save());
}

}}}} // namespace ::nn::psm::driver::detail
