﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/os.h>

#include <nn/psm/driver/detail/psm_BatteryChargePercentage.h>

#include <nn/result/result_HandlingUtility.h>

#include "../psm_BatteryChargeManager.h"
#include "../psm_BatteryState.h"
#include "../psm_BatteryVoltageLevelForCharge.h"
#include "../psm_ChargeArbiter.h"
#include "../psm_ChargerDriver.h"
#include "../psm_SessionManager.h"
#include "psm_PeriodicStateUpdater.h"

namespace nn { namespace psm { namespace driver { namespace detail {

namespace {

#if defined(NN_BUILD_CONFIG_OS_WIN)
// OS が Windows の場合はテストの時短を目的として更新間隔を短縮します。
const ::nn::TimeSpan StateUpdateInterval = ::nn::TimeSpan::FromMilliSeconds(10);
#else
const ::nn::TimeSpan StateUpdateInterval = ::nn::TimeSpan::FromMilliSeconds(1000);
#endif

} // namespace

PeriodicStateUpdater::PeriodicStateUpdater() NN_NOEXCEPT
    : m_pChargeArbiter(nullptr)
    , m_pBatteryChargeManager(nullptr)
    , m_pBatteryState(nullptr)
    , m_pChargerDriver(nullptr)
    , m_pSessionManager(nullptr)
    , m_BatteryVoltageState(BatteryVoltageState_Good)
    , m_BatteryVoltageLevelForCharge(BatteryVoltageLevelForCharge::DisallowChargeOnHighTemperature)
    , m_PeriodicStateUpdateEvent()
    , m_PeriodicStateUpdateMultiWaitHolder()
    , m_PeriodicStateUpdateEventForTest()
{
    // 何もしない。
}

PeriodicStateUpdater::~PeriodicStateUpdater() NN_NOEXCEPT
{
    // 何もしない。
}

void PeriodicStateUpdater::Initialize(ChargeArbiter* pChargeArbiter, BatteryChargeManager* pBatteryChargeManager, BatteryState* pBatteryState, IChargerDriver* pChargerDriver, SessionManager* pSessionManager) NN_NOEXCEPT
{
    m_pChargeArbiter = pChargeArbiter;
    m_pBatteryChargeManager = pBatteryChargeManager;
    m_pBatteryState = pBatteryState;
    m_pChargerDriver = pChargerDriver;
    m_pSessionManager = pSessionManager;

    ::nn::os::InitializeEvent(&m_PeriodicStateUpdateEventForTest, false, ::nn::os::EventClearMode_ManualClear);

    ::nn::os::InitializeTimerEvent(&m_PeriodicStateUpdateEvent, ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&m_PeriodicStateUpdateMultiWaitHolder, &m_PeriodicStateUpdateEvent);
}

void PeriodicStateUpdater::Finalize() NN_NOEXCEPT
{
    ::nn::os::FinalizeMultiWaitHolder(&m_PeriodicStateUpdateMultiWaitHolder);
    ::nn::os::FinalizeTimerEvent(&m_PeriodicStateUpdateEvent);

    ::nn::os::FinalizeEvent(&m_PeriodicStateUpdateEventForTest);
}

void PeriodicStateUpdater::LinkMultiWaitHolders(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_PeriodicStateUpdateMultiWaitHolder);
}

void PeriodicStateUpdater::UnlinkMultiWaitHolders() NN_NOEXCEPT
{
    ::nn::os::UnlinkMultiWaitHolder(&m_PeriodicStateUpdateMultiWaitHolder);
}

void PeriodicStateUpdater::StartPeriodicTimerEvent() NN_NOEXCEPT
{
    ::nn::os::StartPeriodicTimerEvent(&m_PeriodicStateUpdateEvent, StateUpdateInterval, StateUpdateInterval);
}

void PeriodicStateUpdater::StopTimerEvent() NN_NOEXCEPT
{
    ::nn::os::StopTimerEvent(&m_PeriodicStateUpdateEvent);
}

bool PeriodicStateUpdater::HandleEvent() NN_NOEXCEPT
{
    if ( ::nn::os::TryWaitTimerEvent(&m_PeriodicStateUpdateEvent) )
    {
        ::nn::os::ClearTimerEvent(&m_PeriodicStateUpdateEvent);

        NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessPeriodicStateUpdateEvent());

        // ウォッチドッグもまとめてリセットします。
        if ( m_pChargerDriver->IsWatchdogTimerEnabled() )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargerDriver->ResetWatchdogTimer());
        }

        ::nn::os::SignalEvent(&m_PeriodicStateUpdateEventForTest);

        return true;
    }

    return false;
}

::nn::Result PeriodicStateUpdater::GetPeriodicStateUpdateEventForTest(::nn::os::EventType** pOutEventPointer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutEventPointer);

    *pOutEventPointer = &m_PeriodicStateUpdateEventForTest;

    NN_RESULT_SUCCESS;
}

::nn::Result PeriodicStateUpdater::ProcessPeriodicStateUpdateEvent() NN_NOEXCEPT
{
    m_pBatteryChargeManager->Update(true);

    auto currentState = m_pBatteryState->GetBatteryVoltageState();
    auto previousState = m_BatteryVoltageState;

    m_BatteryVoltageState = currentState;

    if ( currentState != previousState )
    {
        m_pSessionManager->SignalStateChangeEvents(SessionManager::EventType_BatteryVoltageStateChange);
    }

    // 電池残量値の生値のオフセットの補正（電池僅少時に現在の RawBatteryChargePercentage をオフセットにします）
    if ( (previousState == BatteryVoltageState_Good || previousState == BatteryVoltageState_BoostPerformanceModeProhibited)
        && (currentState == BatteryVoltageState_SleepRequired) )
    {
        m_pBatteryState->UpdateRawBatteryChargeOffsetPercentage();
    }

    // 電池残量値の生値のオフセットの補正（電池十分時に現在の RawBatteryChargePercentage がオフセットよりも小さければ新たなオフセットにします）
    if ( (currentState == BatteryVoltageState_Good || currentState == BatteryVoltageState_BoostPerformanceModeProhibited) )
    {
        m_pBatteryState->TryDecreaseRawBatteryChargeOffsetPercentage();
    }

    // 電池電圧低下時に電池残量の生値が大きかったら、残量ずれ
    if ( (currentState == BatteryVoltageState_SleepRequired) && m_pBatteryState->GetRawBatteryChargePercentage() >= MinRawDefaultPercentage )
    {
        nn::os::SignalSystemEvent(m_pBatteryState->GetBatteryChargeCalibratedEvent());
    }

    auto currentLevel = m_pBatteryState->GetBatteryVoltageLevelForCharge();
    auto previousLevel = m_BatteryVoltageLevelForCharge;

    m_BatteryVoltageLevelForCharge = currentLevel;

    if ( currentLevel != previousLevel )
    {
        m_pChargeArbiter->SetBatteryVoltageLevel(currentLevel);
    }

    // BatteryChargeManager が BatteryState を更新した後に判定する。
    if ( m_pBatteryState->IsFullCharge() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargeArbiter->NotifyFullCharge());
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pChargeArbiter->NotifyNotFullCharge());
    }

    NN_RESULT_SUCCESS;
}


}}}} // namespace nn::psm::driver::detail
