﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdio>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/lmem/lmem_UnitHeap.h>
#include <nn/util/util_StringUtil.h>
#include <nn/profiler/profiler_Api.h>
#include <nn/htcs.h>

#include "profiler_CodeRewriting.h"
#include "profiler_CommMessages.h"
#include "profiler_Comms.h"
#include "profiler_DataStream.h"
#include "profiler_Defines.h"
#include "profiler_ForceSend.h"
#include "profiler_HeaderWriter.h"
#include "profiler_Hipc.h"
#include "profiler_IpcEventLoop.h"
#include "profiler_LibModule.h"
#include "profiler_LibPrivate.h"
#include "profiler_Logging.h"
#include "profiler_Memory.h"
#include "profiler_Messages.h"
#include "profiler_Nvn.h"
#include "profiler_RecordMethods.h"
#include "profiler_ResultPrivate.h"
#include "profiler_SamplingThread.h"
#include "profiler_StringTable.h"
#include "profiler_TargetApplication.h"
#include "profiler_Workarea.h"
#include "profiler_WriteToBuffer.h"


namespace nn { namespace profiler {


//-------------------------------------------------------------------
// Forward declarations
//-------------------------------------------------------------------
static void FillThreadList();
static void FillThreadNameBuffer(char* buffer, size_t bufferSize);
static char* AllocateCommMsgMem();
static void MessageSentCallbackToFreeMem(uint32_t msg, nn::Result result, void* data);


//-------------------------------------------------------------------
// Globals
//-------------------------------------------------------------------

const uint16_t ThreadListVersion = 2;
const uint32_t InstrumentedBufferVersion = 1;

// Size of buffer to store strings that are sent in messages
const size_t TEMP_BUFFER_SIZE = 1024;

const size_t SIZE_OF_COMM_UNIT_HEAP = (TEMP_BUFFER_SIZE * 8) + sizeof(nn::lmem::HeapCommonHead) + 256;


enum MessageQueryTypes
{
    QUERY_CURRENTLINE = 0
};

namespace // anonymous
{
    // The following value is not const because PCCOM does not support the
    //  sending of a uint32_t independently
    uint32_t ProfilerVersion = ProfilerRuntimeVersion;

    // Event signaled to being profiling
    nn::os::EventType s_beginEvent;

    // Event signaling that a transfer has completed
    nn::os::EventType s_xferCompleteEvent;

    // Event signaled to stop profiling
    nn::os::EventType s_stopEvent;

    // Pointer to where received settings should be stored
    SettingsFromThePcGui* gSettings = nullptr;

    void* s_commBufferMemory = nullptr;
    nn::lmem::HeapHandle s_commBuffers;

    nn::os::TransferMemoryType s_transferMemory;

    bool firstReadyToProfile = false;


    struct CommThreadList
    {
        uint32_t numThreads;
        uint32_t padding; // since 64-bit values are 8-byte aligned
        nn::os::ThreadId threadIDs[ThreadIdListSize];
        uint32_t coreIDs[ThreadIdListSize];
    };
    CommThreadList* threadInfo;


    char* s_threadNameSection = nullptr;
    const int ThreadNameSectionSize = 16 + ThreadIdListSize * (nn::os::ThreadNameLengthMax + sizeof(nn::os::ThreadId) + sizeof(int) + sizeof(int));


    struct ModuleData
    {
        uint64_t baseAddress;
        uint64_t size;
        uint64_t loadTime;
        uint64_t unloadTime;
        char name[MaximumFilePathLength];
        GnuBuildId buildId;
        uint32_t padding; // GnuBuildId size is not 8-byte multiple
    };

    struct ApplicationInfo
    {
        uint32_t version;
        uint32_t coreMask;
        uint32_t moduleCount;
        uint32_t padding;
        ModuleData moduleData[];
    };
    ApplicationInfo* s_appInfo = nullptr;
    size_t s_appInfoSize;

    bool s_initializedHtcs = false;

    void* Allocate(size_t size)
    {
        void *memory = nn::profiler::Memory::GetInstance()->Allocate(size);
        DEBUG_LOG("Allocated %lu bytes to address %p\n", size, memory);
        return memory;
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        DEBUG_LOG("Freeing address %p\n", p);
        nn::profiler::Memory::GetInstance()->Free(p);
    }



    static void FillApplicationInfo()
    {
        if (s_appInfo != nullptr)
        {
            Memory::GetInstance()->Free(s_appInfo);
            s_appInfo = nullptr;
        }

        int moduleCount = gPastModules.size();
        if (moduleCount <= 0)
        {
            BuildStaticModules();
        }
        moduleCount = gPastModules.size();
        NN_SDK_ASSERT(moduleCount > 0);

        s_appInfoSize = sizeof(ApplicationInfo) + (sizeof(ModuleData) * size_t(moduleCount));
        s_appInfo = reinterpret_cast<ApplicationInfo*>(Memory::GetInstance()->Allocate(s_appInfoSize, 32));

        memset(s_appInfo, 0, s_appInfoSize);

        s_appInfo->version = ProfilerApplicationInfoVersion;
        s_appInfo->coreMask = TargetApplication::GetCoreMask();
        s_appInfo->moduleCount = static_cast<uint32_t>(moduleCount);
        s_appInfo->padding = 0xaaaaaaaa;

        int index = 0;
        for (auto& x : gPastModules)
        {
            auto& data = s_appInfo->moduleData[index];
            data.baseAddress = x.Address;
            data.size = x.Size;
            data.loadTime = x.LoadTime;
            data.unloadTime = x.UnloadTime;
            memcpy(data.name, x.Name, MaximumFilePathLength);
            memcpy(&data.buildId, &x.BuildId, sizeof(data.buildId));
            ++index;
        }
        NN_SDK_ASSERT(index == moduleCount);
    }



    static void OnMessage(
        void* info,
        size_t length,
        uint32_t message)
    {
        DEBUG_LOG("== Message ==\n");
        switch (message)
        {
        case ProfilerCommMessage_Argv:
            {
                int index = 0;
                if (info != nullptr && length != 0)
                {
                    index = *reinterpret_cast<int*>(info);
                }

                int argc = nn::os::GetHostArgc();
                char** argv = nn::os::GetHostArgv();
                char* msgMem = AllocateCommMsgMem();
                char* writePtr = msgMem;
                memset(msgMem, 0, TEMP_BUFFER_SIZE);
                writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(argc));
                const char* appName = nullptr;
                size_t appNameLength;
                if (index == 0)
                {
                    TargetApplication::GetCurrent()->GetApplicationName(appName, appNameLength);
                }
                if (index == 0 && appName != nullptr)
                {
                    writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(index));
                    size_t copyLength = std::min(appNameLength, TEMP_BUFFER_SIZE - sizeof(int32_t) * 2);
                    memcpy(writePtr, appName, copyLength);
                }
                else if (0 <= index && index < argc)
                {
                    writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(index));
                    nn::util::Strlcpy(writePtr, argv[index], TEMP_BUFFER_SIZE - sizeof(int32_t) * 2);
                }
                else
                {
                    writePtr = WriteToBuffer(writePtr, static_cast<int32_t>(-1));
                }
                SendMessageToPc(ProfilerCommMessage_Argv, msgMem, TEMP_BUFFER_SIZE, false);
                MessageSentCallbackToFreeMem(ProfilerCommMessage_Argv, nn::ResultSuccess(), msgMem);
            }
            break;

        case ProfilerCommMessage_Version:
            SendMessageToPc(ProfilerCommMessage_Version, &ProfilerVersion, sizeof(ProfilerVersion), false);
            break;

        case ProfilerCommMessage_ApplicationInfo:
            DEBUG_LOG("APPLICATION INFO\n");
            {
                FillApplicationInfo();
                SendMessageToPc(ProfilerCommMessage_ApplicationInfo, s_appInfo, s_appInfoSize, false);
                SendReadyToProfile(SampleBuffers::GetInstance()->GetSize());
            }
            break;

        case ProfilerCommMessage_Start:
            if (gSettings != nullptr)
            {
                DEBUG_LOG("PROFILE START\n");
                if (info != nullptr && length != 0)
                {
                    if (length != sizeof(SettingsFromThePcGui))
                    {
                        ERROR_LOG("PCCOM: Settings may be too big [%lu] != [%lu]\n",
                            length,
                            sizeof(SettingsFromThePcGui));
                    }
                    NN_SDK_ASSERT((gSettings->flags & SettingsFromThePcGui::UseOutOfProcessSampling) == 0);
                    DUMP_CURRENT_LINE();
                    memcpy(gSettings, info, sizeof(SettingsFromThePcGui));
                    DUMP_CURRENT_LINE();
                    SignalBeginEvent();
                    DUMP_CURRENT_LINE();
                }
            }
            break;

        case ProfilerCommMessage_Stop:
            DEBUG_LOG("STOP PROFILING\n");
            //StopProfilingSamplingThreads();
            nn::os::SignalEvent(&s_stopEvent);
            break;

        case ProfilerCommMessage_TransferThreads:
            {
                FillThreadList();

                // Send a message back to the PC with the list of the threads
                //  Channel size is numThreads + 1 as the numThread variable is also
                //  being sent
                SendMessageToPc(ProfilerCommMessage_TransferThreads, threadInfo, sizeof(CommThreadList), false);
            }
            break;

        case ProfilerCommMessage_ReadyToProfile:
            {
                ProfilerStatus prevStatus = GetProfilerStatus();

                //if (prevStatus == ProfilerStatus_Profiling)
                //{
                //    SendProfilingHasBegun();
                //}
                //else if (prevStatus == ProfilerStatus_Active)
                //{
                //    if (SampleBuffers::GetInstance()->HasData())
                //    {
                //        SetProfilerStatus(ProfilerStatus_Transferring);
                //        nn::os::EventType* sendEvent = SendDataToPC();
                //        if (sendEvent != nullptr)
                //        {
                //            nn::os::WaitEvent(sendEvent);
                //            SampleBuffers::GetInstance()->Reset();
                //        }
                //        SetProfilerStatus(prevStatus);
                //    }
                //}
                //else if (prevStatus == ProfilerStatus_Transferring)
                //{
                //    SetProfilerStatus(ProfilerStatus_Active);
                //}

                if (prevStatus == ProfilerStatus_Active && firstReadyToProfile)
                {
                    if (SampleBuffers::GetInstance()->HasData())
                    {
                        SetProfilerStatus(ProfilerStatus_Transferring);
                        SendDataToPC();
                    }
                }

                firstReadyToProfile = false;
            }
            break;


        default:
            DEBUG_LOG("PCCOM: Unhandled Message: %d\n", message);
            break;
        }
    } // NOLINT(impl/function_size)



    static void OnSendDataComplete(
        uint32_t message,
        nn::Result result)
    {
        DEBUG_LOG("== Send Data Complete ==\n");
        if (result.IsSuccess())
        {
            nn::os::SignalEvent(&s_xferCompleteEvent);
        }
        else
        {
            ERROR_LOG("Error on data send\n");
            DumpResultInformation(LOG_AS_ERROR, result);
        }

        switch (message)
        {
        case ProfilerCommMessage_TransferData:
            Memory::GetInstance()->Free(s_threadNameSection);
            s_threadNameSection = nullptr;
            DestroyTransferMemory();
            SampleBuffers::GetInstance()->Reset();
            break;

        default:
            break;
        }
    }
} // anonymous namespace


//-------------------------------------------------------------------
//
//-------------------------------------------------------------------
void InitializeCommunicationsLayer()
{
    nn::Result result;

    Memory* memHandler = Memory::GetInstance();

    InitializeIpcEventLoop(
        OnSendDataComplete,
        OnMessage);

    {
        threadInfo = memHandler->Allocate<CommThreadList>();
        if (threadInfo == nullptr)
        {
            ERROR_LOG("Error getting memory for communication layer.\n");
            ProfilerPanic();
        }
        new (threadInfo) CommThreadList;
    }

    s_commBufferMemory = memHandler->Allocate(SIZE_OF_COMM_UNIT_HEAP);
    if (s_commBufferMemory == nullptr)
    {
        ERROR_LOG("Error allocating memory for comm buffer memory.\n");
        ProfilerPanic();
    }

    s_commBuffers = nn::lmem::CreateUnitHeap(
        s_commBufferMemory,
        SIZE_OF_COMM_UNIT_HEAP,
        TEMP_BUFFER_SIZE,
        nn::lmem::CreationOption_ThreadSafe,
        32,
        nn::lmem::InfoPlacement_Head);
    if (s_commBuffers == nullptr)
    {
        ERROR_LOG("Error creating unit heap.\n");
        ProfilerPanic();
    }

    firstReadyToProfile = true;

    // Initialize all events here
    nn::os::InitializeEvent(&s_beginEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&s_xferCompleteEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&s_stopEvent, false, nn::os::EventClearMode_ManualClear);

    if (nn::htcs::IsInitialized() == false)
    {
        nn::htcs::Initialize(Allocate, Deallocate);
        s_initializedHtcs = true;
    }
    else
    {
        s_initializedHtcs = false;
    }
}



void FinalizeCommunicationsLayer()
{
    // We finalize this first since it may depend on events and heap buffers
    if (s_initializedHtcs)
    {
        nn::htcs::Finalize();
    }

    // Now finalize in roughly reverse order of creation
    nn::os::FinalizeEvent(&s_beginEvent);
    nn::os::FinalizeEvent(&s_xferCompleteEvent);
    nn::os::FinalizeEvent(&s_stopEvent);

    nn::lmem::DestroyUnitHeap(s_commBuffers);

    Memory::GetInstance()->Free(s_appInfo);
    s_appInfo = nullptr;

    Memory::GetInstance()->Free(threadInfo);
    threadInfo = nullptr;

    Memory::GetInstance()->Free(s_commBufferMemory);
    s_commBufferMemory = nullptr;

    FinalizeIpcEventLoop();
}



nn::os::EventType* CheckForBegin(SettingsFromThePcGui *settings)
{
    // Maintain a copy of the settings pointer
    // The PCCOM Begin request will write the settings to this location
    gSettings = settings;

    return &s_beginEvent;
}



nn::os::EventType* SendDataToPC()
{
    nn::Result result;

    DEBUG_LOG("Attempting to send data to PC\n");

    // Make sure that the event is cleared before going into this
    // Users of the event can also clear the event
    nn::os::ClearEvent(&s_xferCompleteEvent);

    if (IsPCConnected() == false)
    {
        return nullptr;
    }

    {
        WorkArea* ws = GetWorkAreaForCore(SampleBufferIndex_Instrumentation);
        while (ws->recordCount > 0)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
        }
    }

    result = SetupTransferMemory(
        SampleBuffers::GetInstance()->GetStartAddress(),
        SampleBuffers::GetInstance()->GetSize());
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_FATAL, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    size_t totalSize = 0;
    uint32_t pieces = 0;

    const int SectionOffsetCount = SupportedCoreCount + 5;
    const int NsoOffsetIndex = SupportedCoreCount + 0;
    const int ThreadNameOffsetIndex = SupportedCoreCount + 1;
    const int UserMarkerNamesIndex = SupportedCoreCount + 2;
    const int NvnFunctionNamesIndex = SupportedCoreCount + 3;
    const int InstrumentationBufferIndex = SupportedCoreCount + 4;

    uint32_t sectionOffsets[SectionOffsetCount] = { 0 };
    uint32_t sectionCount = 0;

    // Add in master header
    Header* masterHeader = GetMasterHeader();
    totalSize += masterHeader->GetSize();
    pieces += 1;

    size_t coreBufferSizes[SupportedCoreCount] = { 0 };
    for (uint32_t core = 0; core < SupportedCoreCount; ++core)
    {
        auto coreBuffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex(core));
        size_t coreBufferTotalSize = coreBuffer->totalSize;
        if (coreBufferTotalSize > 0)
        {
            ++sectionCount;

            // Add in pieces for sample buffers
            sectionOffsets[core] = static_cast<uint32_t>(totalSize);
            totalSize += coreBufferTotalSize;
            pieces += coreBuffer->listCount;
            coreBufferSizes[core] = coreBufferTotalSize;

            // Add in core header
            Header *coreHeader = GetCoreHeader(core);
            totalSize += coreHeader->GetSize();
            pieces += 1;
        }
    }

    {
        FillApplicationInfo();
        ++sectionCount;
        sectionOffsets[NsoOffsetIndex] = static_cast<uint32_t>(totalSize);
        totalSize += s_appInfoSize;
        pieces += 1;
    }

    {
        s_threadNameSection = reinterpret_cast<char*>(Memory::GetInstance()->Allocate(ThreadNameSectionSize));
        FillThreadNameBuffer(s_threadNameSection, ThreadNameSectionSize);
        ++sectionCount;
        sectionOffsets[ThreadNameOffsetIndex] = static_cast<uint32_t>(totalSize);
        totalSize += ThreadNameSectionSize;
        pieces += 1;
    }

    StringTable* markerNames = GetMarkerNamesStringTable();
    {
        ++sectionCount;
        sectionOffsets[UserMarkerNamesIndex] = static_cast<uint32_t>(totalSize);
        totalSize += markerNames->GetTotalSize();
        pieces += 1;
    }

    const char* nvnFuncs;
    size_t nvnFuncSize;
    bool nvnFuncsFound = nn::profiler::GetNvnFunctionPointers(&nvnFuncs, &nvnFuncSize);
    if (nvnFuncsFound == false)
    {
        nvnFuncs = "\1\0\0\0\0\0\0\0\0";
        nvnFuncSize = sizeof(uint32_t) * 2;
    }
    {
        ++sectionCount;
        sectionOffsets[NvnFunctionNamesIndex] = static_cast<uint32_t>(totalSize);
        totalSize += nvnFuncSize;
        pieces += 1;
    }

    // Check for instrumented code/data
    size_t instrumentedSize = 0;
    {
        auto buffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex_Instrumentation);
        instrumentedSize = buffer->totalSize;
        if (instrumentedSize > 0)
        {
            ++sectionCount;
            sectionOffsets[InstrumentationBufferIndex] = static_cast<uint32_t>(totalSize);
            totalSize += instrumentedSize;
            pieces += buffer->listCount;
        }
    }

    // Provide offsets after the master header to each core's data
    {
        uint32_t oldMasterHeaderSize = static_cast<uint32_t>(masterHeader->GetSize());

        for (uint32_t core = 0; core < SupportedCoreCount; ++core)
        {
            if (sectionOffsets[core] > 0)
            {
                masterHeader->Write(
                    HeaderSpecialValues_CoreOffset,
                    core,
                    sectionOffsets[core] - oldMasterHeaderSize);
            }
        }

        masterHeader->Write(
            HeaderSpecialValues_NsoRegionOffset,
            sectionOffsets[NsoOffsetIndex] - oldMasterHeaderSize);

        masterHeader->Write(
            HeaderSpecialValues_ThreadListOffset,
            sectionOffsets[ThreadNameOffsetIndex] - oldMasterHeaderSize);

        masterHeader->Write(
            HeaderSpecialValues_UserMarkerNames,
            sectionOffsets[UserMarkerNamesIndex] - oldMasterHeaderSize);

        masterHeader->Write(
            HeaderSpecialvalues_UserMarkerNamesBase,
            reinterpret_cast<uint64_t>(markerNames->GetTableAddress()));

        masterHeader->Write(
            HeaderSpecialValues_NvnFuncNamesOffset,
            sectionOffsets[NvnFunctionNamesIndex] - oldMasterHeaderSize);

        if (instrumentedSize > 0)
        {
            NN_SDK_ASSERT(instrumentedSize <= UINT32_MAX);
            masterHeader->Write(
                HeaderSpecialValues_InstrumentedOffset,
                sectionOffsets[InstrumentationBufferIndex] - oldMasterHeaderSize,
                InstrumentedBufferVersion,
                static_cast<uint32_t>(instrumentedSize),
                gSettings->perf_counter_cycle);
        }

        masterHeader->WriteControlValueOnly(HeaderSpecialValues_HeaderEnd);

        totalSize -= oldMasterHeaderSize;
        totalSize += masterHeader->GetSize();
    }

    DEBUG_LOG("Starting multi-part send size = %lu; pieces = %d\n", totalSize, pieces);
    FORCE_SEND(StartMultipartMessageToPc(ProfilerCommMessage_TransferData, totalSize, pieces, true));

    // Write master header
    FORCE_SEND(SendMultipartMessageToPc(masterHeader->GetBuffer(), masterHeader->GetSize(), pieces - 1));
    --pieces;

    for (uint32_t core = 0; core < SupportedCoreCount; ++core)
    {
        size_t coreBufferSize = coreBufferSizes[core];
        auto coreBuffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex(core));
        if (coreBufferSize > 0)
        {
            // Write core header
            Header *coreHeader = GetCoreHeader(core);
            FORCE_SEND(SendMultipartMessageToPc(coreHeader->GetBuffer(), coreHeader->GetSize(), pieces - 1));
            --pieces;

            for (uint32_t listIdx = 0; listIdx < coreBuffer->listCount; ++listIdx)
            {
                auto data = coreBuffer->list[listIdx];
                size_t writeSize = std::min(data.size, coreBufferSize);
                coreBufferSize -= writeSize;
                FORCE_SEND(SendMultipartMessageToPc(data.memory, writeSize, pieces - 1));
                --pieces;
            }
        }
    }

    FORCE_SEND(SendMultipartMessageToPc(s_appInfo, s_appInfoSize, pieces - 1));
    --pieces;

    FORCE_SEND(SendMultipartMessageToPc(s_threadNameSection, ThreadNameSectionSize, pieces - 1));
    --pieces;

    Memory::GetInstance()->Free(s_threadNameSection);
    s_threadNameSection = nullptr;

    FORCE_SEND(SendMultipartMessageToPc(
        markerNames->GetTableAddress(),
        markerNames->GetTotalSize(),
        pieces - 1));
    --pieces;

    FORCE_SEND(SendMultipartMessageToPc(nvnFuncs, nvnFuncSize, pieces - 1));
    --pieces;

    if (instrumentedSize > 0)
    {
        auto buffer = SampleBuffers::GetInstance()->GetBuffer(SampleBufferIndex_Instrumentation);
        for (uint32_t listIdx = 0; listIdx < buffer->listCount; ++listIdx)
        {
            auto data = buffer->list[listIdx];
            size_t writeSize = std::min(data.size, instrumentedSize);
            instrumentedSize -= writeSize;
            FORCE_SEND(SendMultipartMessageToPc(data.memory, writeSize, pieces - 1));
            --pieces;
        }
    }

    NN_SDK_ASSERT(pieces == 0u);

    return &s_xferCompleteEvent;
} // NOLINT(impl/function_size)



nn::os::EventType* GetTransferCompleteEvent()
{
    return &s_xferCompleteEvent;
}



void SignalBeginEvent()
{
    nn::os::ClearEvent(GetStopEvent());
    nn::os::SignalEvent(&s_beginEvent);
}



nn::os::EventType* GetStopEvent()
{
    return &s_stopEvent;
}


void SendNotificationToPC(
    uint32_t code,
    nn::Result result)
{
    char* buffer = AllocateCommMsgMem();
    memset(buffer, 0, TEMP_BUFFER_SIZE);

    char* writePtr = buffer;
    writePtr = WriteToBuffer(writePtr, code);
    writePtr = WriteToBuffer(writePtr, result.GetInnerValueForDebug());

    FORCE_SEND(SendMessageToPc(ProfilerCommMessage_Error, buffer, TEMP_BUFFER_SIZE, false));
    MessageSentCallbackToFreeMem(ProfilerCommMessage_Error, nn::ResultSuccess(), buffer);
}



void SendReadyToProfile(size_t maxBuffSize)
{
    char* buffer = AllocateCommMsgMem();
    uint32_t* tempBuffer = reinterpret_cast<uint32_t*>(buffer);
    size_t bufferSize = 0;

    tempBuffer[0] = static_cast<uint32_t>(maxBuffSize);
    bufferSize += sizeof(*tempBuffer);

    INFO_LOG("Ready To Profile\n");

    SendMessageToPc(ProfilerCommMessage_ReadyToProfile, buffer, bufferSize, false);
    MessageSentCallbackToFreeMem(ProfilerCommMessage_ReadyToProfile, nn::ResultSuccess(), buffer);
}



void SendProfilingHasBegun()
{
    SendMessageToPc(ProfilerCommMessage_ProfilingStarted, nullptr, 0, false);
}



bool IsPCConnected()
{
    return IsPcSynced();
}


void SendCoreAvailable(uint32_t coreMask)
{
    char* buffer = AllocateCommMsgMem();
    char* tempBuffer = buffer;
    size_t bufferSize = 0;

    tempBuffer = WriteToBuffer(tempBuffer, coreMask);
    bufferSize = static_cast<size_t>(tempBuffer - buffer);

    //s_commLayer->Send(
    //    ProfilerCommMessageCoreAvailable,
    //    buffer,
    //    bufferSize,
    //    MessageSentCallbackToFreeMem);
    SendMessageToPc(ProfilerCommMessage_CoreAvailable, buffer, bufferSize, false);
    MessageSentCallbackToFreeMem(ProfilerCommMessage_CoreAvailable, nn::ResultSuccess(), buffer);
}


nn::Result SetupTransferMemory(void* startAddress, size_t size)
{
    nn::Result result = nn::os::CreateTransferMemory(
        &s_transferMemory,
        startAddress,
        size,
        nn::os::MemoryPermission_None);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_FATAL, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    auto handle = nn::os::DetachTransferMemory(&s_transferMemory);
    DEBUG_LOG("Original Transfer Memory Address %p\n", startAddress);

    result = TransferSampleBuffers(handle, reinterpret_cast<uintptr_t>(startAddress), size);
    if (result.IsFailure())
    {
        DumpResultInformation(LOG_AS_FATAL, result);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return nn::ResultSuccess();
}


nn::Result DestroyTransferMemory()
{
    if (s_transferMemory._state != nn::os::TransferMemoryType::State_Uninitialized)
    {
        nn::os::DestroyTransferMemory(&s_transferMemory);
    }
    return nn::ResultSuccess();
}


//-------------------------------------------------------------------
// Local functions
//-------------------------------------------------------------------
static void FillThreadList()
{
    TargetApplication *app = TargetApplication::GetCurrent();
    ThreadList* threadList = app->GetThreadList();
    threadList->ReadLock();
    int listCount = std::min(threadList->Count(), int(ThreadIdListSize));
    threadInfo->numThreads = uint32_t(listCount);
    for (int i = 0; i < listCount; ++i)
    {
        auto item = threadList->GetByIndex(i);
        threadInfo->threadIDs[i] = item->threadId;

        int idealCore = item->idealCore;
        if (idealCore < 0) { idealCore = 0; }
        threadInfo->coreIDs[i] = static_cast<uint32_t>(idealCore);
    }
    threadList->ReadUnlock();
}

static void FillThreadNameBuffer(char* buffer, size_t bufferSize)
{
    NN_SDK_ASSERT(bufferSize >= ThreadNameSectionSize);

    char* tnsWritePtr = buffer;
    char* endPtr = tnsWritePtr + bufferSize;

    memset(buffer, 0, bufferSize);

    // Write the header
    {
        tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint32_t>(HeaderSpecialValues_ThreadHeaderBegin));
        tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint32_t>(bufferSize - 8));
        tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(0xFFFF)); // version denotation
        tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(ThreadListVersion));
        tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(ThreadIdListSize));
        tnsWritePtr = WriteToBuffer(tnsWritePtr, static_cast<uint16_t>(nn::os::ThreadNameLengthMax));
    }

    // Write thread names
    {
        TargetApplication *app = TargetApplication::GetCurrent();
        ThreadList* threadList = app->GetThreadList();
        threadList->ReadLock();
        int listCount = std::min(threadList->Count(), int(ThreadIdListSize));
        threadInfo->numThreads = uint32_t(listCount);
        for (int i = 0; i < listCount; ++i)
        {
            auto item = threadList->GetByIndex(i);
            tnsWritePtr = WriteToBuffer(tnsWritePtr, item->threadId);
            tnsWritePtr = WriteToBuffer(tnsWritePtr, item->priority);
            tnsWritePtr = WriteToBuffer(tnsWritePtr, item->coreMask);
            strncpy(tnsWritePtr, item->threadName, nn::os::ThreadNameLengthMax);
            tnsWritePtr += nn::os::ThreadNameLengthMax;
        }
        threadList->ReadUnlock();
    }

    NN_SDK_ASSERT(tnsWritePtr <= endPtr);
    NN_UNUSED(endPtr);
}

static char* AllocateCommMsgMem()
{
    void* mem = nullptr;
    while (mem == nullptr)
    {
        mem = nn::lmem::AllocateFromUnitHeap(s_commBuffers);
        if (mem == nullptr)
        {
            nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(10));
        }
    }
    return reinterpret_cast<char*>(mem);
}

static void MessageSentCallbackToFreeMem(
    uint32_t msg,
    nn::Result result,
    void* data)
{
    NN_UNUSED(msg);
    NN_UNUSED(result);

    nn::lmem::FreeToUnitHeap(s_commBuffers, data);
}

} // profiler
} // nn
