﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/service/core/prepo_DebugSettings.h>
#include <nn/prepo/detail/service/core/prepo_ReportBuffer.h>
#include <nn/prepo/detail/service/core/prepo_WorkerThread.h>
#include <nn/prepo/detail/service/core/prepo_CategoryObject.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>
#include <nn/prepo/detail/service/core/prepo_SystemReport.h>
#include <nn/prepo/detail/service/core/prepo_UploadThread.h>
#include <nn/srepo/srepo_ApiAdmin.h>
#include <nn/fs/fs_PriorityPrivate.h>

namespace nn { namespace prepo { namespace detail { namespace service { namespace core {

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[16 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

namespace
{
    void ThreadFunction(void*) NN_NOEXCEPT
    {
        if (!DebugSettings::IsBackgroundProcessingEnabled())
        {
            NN_DETAIL_PREPO_INFO("[prepo] Exit worker thread because which is disabled by debug settings.\n");
            return;
        }

        nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Background);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Normal);
        };

        nn::os::Event reportDataAddEvent(nn::os::EventClearMode_ManualClear);
        ReportBuffer::GetInstance().RegisterAddEvent(reportDataAddEvent.GetBase());
        NN_UTIL_SCOPE_EXIT
        {
            ReportBuffer::GetInstance().UnregisterAddEvent();
        };

        nn::os::SystemEvent systemReportPushEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::srepo::RegisterPushEvent(&systemReportPushEvent, nn::os::EventClearMode_ManualClear));

        StatisticsManager::GetInstance().Load();

        nn::os::Event statisticsUpdateEvent(nn::os::EventClearMode_ManualClear);
        StatisticsManager::GetInstance().RegisterUpdateEvent(statisticsUpdateEvent.GetBase());
        NN_UTIL_SCOPE_EXIT
        {
            StatisticsManager::GetInstance().UnregisterUpdateEvent();
        };

        const auto StatisticsSaveInterval = DebugSettings::GetStatisticsSaveIntervalMin();
        nn::os::TimerEvent statisticsSaveEvent(nn::os::EventClearMode_ManualClear);
        nn::TimeSpan statisticsSaveTime;

        const auto StatisticsPostInterval = DebugSettings::GetStatisticsPostInterval();
        nn::os::TimerEvent statisticsPostEvent(nn::os::EventClearMode_ManualClear);
        statisticsPostEvent.StartPeriodic(nn::TimeSpan::FromSeconds(0), StatisticsPostInterval);

        while (!g_StopEvent.TryWait())
        {
            nn::os::WaitAny(
                g_StopEvent.GetBase(),
                reportDataAddEvent.GetBase(),
                systemReportPushEvent.GetBase(),
                statisticsUpdateEvent.GetBase(),
                statisticsSaveEvent.GetBase(),
                statisticsPostEvent.GetBase());

            // バッファにデータが残っている場合、ファイルに書き出す必要があるため、
            // reportDataAddEvent を優先して処理する。
            if (reportDataAddEvent.TryWait())
            {
                reportDataAddEvent.Clear();

                const auto result = ReportBuffer::GetInstance().Flush([](ReportCategory category, void* data, size_t dataSize, int count) -> nn::Result
                {
                    ReportDataSummary summary;
                    NN_RESULT_DO(CategoryObject::GetReportFileManager(category).WriteFile(&summary, data, dataSize, count));
                    if (summary.deletedCount > 0)
                    {
                        StatisticsManager::GetInstance().AddLostByStorageShortage(category, summary.deletedSize, summary.deletedCount);
                    }
                    NN_RESULT_SUCCESS;
                });
                if (result.IsFailure())
                {
                    detail::service::util::ReportError(result);
                    NN_DETAIL_PREPO_FATAL(NN_DETAIL_PREPO_STRING_FATAL(
                        "Failed to flush report data with error %03d-%04d\n" \
                        "Exit worker thread\n"),
                        result.GetModule(), result.GetDescription());
                    return;
                }

                UploadThread::RequestTransmissionForBackground();
            }

            if (systemReportPushEvent.TryWait())
            {
                systemReportPushEvent.Clear();

                const auto result = PopAndAddSystemReport();
                if (result.IsFailure())
                {
                    detail::service::util::ReportError(result);
                    NN_DETAIL_PREPO_FATAL(NN_DETAIL_PREPO_STRING_FATAL(
                        "Failed to add system report data with error %03d-%04d\n" \
                        "Exit worker thread\n"),
                        result.GetModule(), result.GetDescription());
                    return;
                }
            }

            if (statisticsUpdateEvent.TryWait())
            {
                statisticsUpdateEvent.Clear();

                const auto elapsedTime =
                    nn::os::GetSystemTick().ToTimeSpan() - statisticsSaveTime;

                if (elapsedTime < StatisticsSaveInterval)
                {
                    statisticsSaveEvent.StartOneShot(StatisticsSaveInterval - elapsedTime);
                }
                else
                {
                    statisticsSaveEvent.Signal();
                }
            }

            if (statisticsSaveEvent.TryWait())
            {
                statisticsSaveEvent.Clear();

                statisticsSaveTime = nn::os::GetSystemTick().ToTimeSpan();

                const auto result = StatisticsManager::GetInstance().Save();
                if (result.IsFailure())
                {
                    detail::service::util::ReportError(result);
                    NN_DETAIL_PREPO_FATAL(NN_DETAIL_PREPO_STRING_FATAL(
                        "Failed to flush report statistics with error %03d-%04d\n" \
                        "Exit worker thread\n"),
                        result.GetModule(), result.GetDescription());
                    return;
                }
            }

            if (statisticsPostEvent.TryWait())
            {
                statisticsPostEvent.Clear();

                const auto result = StatisticsManager::GetInstance().Post();
                if (result.IsFailure() && !ResultOutOfResource::Includes(result))
                {
                    detail::service::util::ReportError(result);
                    NN_DETAIL_PREPO_FATAL(NN_DETAIL_PREPO_STRING_FATAL(
                        "Failed to post report statistics with error %03d-%04d\n"),
                        result.GetModule(), result.GetDescription());
                }
            }
        }
    } // NOLINT(impl/function_size)
}

void WorkerThread::Start() NN_NOEXCEPT
{
    nn::os::CreateThread(&g_Thread, ThreadFunction, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(prepo, Worker));
    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(prepo, Worker));

    nn::os::StartThread(&g_Thread);
}

void WorkerThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
}

}}}}}
