﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/prepo/detail/ipc/prepo_IpcService.h>
#include <nn/prepo/detail/ipc/prepo_ServiceConfig.h>
#include <nn/prepo/detail/ipc/prepo_ServiceName.h>
#include <nn/prepo/detail/ipc/prepo_IPrepoService.sfdl.h>
#include <nn/prepo/prepo_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sm/sm_Result.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/util/util_LockGuard.h>

#if defined (NN_BUILD_CONFIG_OS_WIN)
#include <nn/prepo/detail/service/prepo_PrepoService.generic.h>
#include <nn/sf/sf_ObjectFactory.h>
#endif

namespace nn { namespace prepo { namespace detail { namespace ipc {

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

#if defined (NN_BUILD_CONFIG_OS_WIN)

    typedef nn::sf::UnmanagedServiceObject<detail::ipc::IPrepoService, detail::service::PrepoServiceForGeneric> PrepoServiceObject;

    std::aligned_storage<sizeof (PrepoServiceObject)>::type g_Storage;
    PrepoServiceObject* g_Object = nullptr;

#else

    nn::sf::ShimLibraryObjectHolder<IPrepoService>
        g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::sf::SimpleAllInOneHipcSubDomainClientManager<ObjectCountPerProcess>
        g_Domain = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;

#endif

    IPrepoService* g_Service = nullptr;
}

void Initialize() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

#if defined (NN_BUILD_CONFIG_OS_WIN)

    g_Object = new (&g_Storage) PrepoServiceObject();

    g_Service = g_Object->GetShared().Get();

#else

    const char* serviceNames[] =
    {
        ServiceNameForGeneric,
        ServiceNameForSystemUser,
        ServiceNameForManager,
        ServiceNameForAdministrator
    };

    bool isServiceFound = false;

    // 1 アプリに割り当てる権限は 1 つのみ。
    // 1 アプリが複数の権限を所持していた場合、最初に発見したもののみを利用する。
    for (auto name : serviceNames)
    {
        nn::Result result = g_Domain.InitializeShimLibraryHolder(&g_Holder, name);

        if (result.IsSuccess())
        {
            isServiceFound = true;
            break;
        }
        else if (!nn::sm::ResultNotPermitted::Includes(result))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    NN_ABORT_UNLESS(isServiceFound, "[prepo] Needs API permission.");

    g_Service = g_Holder.GetObject().Get();

    g_Domain.GetClientSessionManager().SetSessionCount(SessionCountPerProcess);

#endif
}

void Finalize() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

#if defined (NN_BUILD_CONFIG_OS_WIN)

    g_Service = nullptr;

    if (g_Object)
    {
        g_Object->~PrepoServiceObject();
        g_Object = nullptr;
    }

#else

    if (g_Service)
    {
        nn::sf::ReleaseSharedObject(g_Service);
        g_Service = nullptr;
    }

    g_Holder.FinalizeHolder();

#endif
}

IPrepoService* GetPrepoService() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_Mutex);

    return g_Service;
}

}}}}
