﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/perflog/Daemon.h>
#include "perflog_CreateDaemon.h"
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer
#include <nn/perflog/sfdl/perflog.sfdl.h>
#include <nn/os/os_Result.public.h>
#include <nn/nn_SdkLog.h>


namespace nn { namespace perflog {

namespace {
    // Shared pointer to the service object for implementing the shim library.
    nn::sf::SharedPointer<sf::IDaemon> g_Daemon;
}


nn::Result Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_Daemon);
    g_Daemon = CreateDaemonByHipc();
    return nn::ResultSuccess();
}


nn::Result Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);
    // Free the shared pointer by assigning nullptr.
    g_Daemon = nullptr;
    return nn::ResultSuccess();
}


nn::Result RegisterTrigger(
    const char* name,
    OSEvent*    pEvent ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);

    nn::sf::InBuffer nameInBuffer(name, strlen(name) + 1 );

    auto result = g_Daemon->RegisterTrigger(nameInBuffer,
         nn::sf::NativeHandle( nn::os::GetWritableHandleOfSystemEvent(pEvent),false) );
    return result;
}


nn::Result BeginDump(
    LogDumpCookie* pCookie ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);
    NN_SDK_ASSERT(pCookie);

    auto result = g_Daemon->BeginDump( pCookie );
    return result;
}

nn::Result DumpLine(
    LogDumpCookie cookie,
    uint64_t      threadId,
    uint64_t      timestampNs,
    uint32_t      id,
    const char*   context ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);

    nn::sf::InBuffer ctxInBuffer(context, strlen(context) + 1 );

    auto result = g_Daemon->DumpLine( cookie, threadId, timestampNs, id, ctxInBuffer );
    return result;
}

nn::Result EndDump(
    LogDumpCookie cookie ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);

    auto result = g_Daemon->EndDump(cookie );
    return result;
}

nn::Result TriggerDump(
    const char* mask  ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);

    nn::sf::InBuffer maskInBuffer(mask, strlen(mask) + 1 );

    auto result = g_Daemon->TriggerDump( maskInBuffer );
    return result;
}


nn::Result UnregisterTrigger(
    const char* name  ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Daemon);

    nn::sf::InBuffer nameInBuffer(name, strlen(name) + 1 );

    auto result = g_Daemon->UnregisterTrigger( nameInBuffer );
    return result;
}



} } // nn::perflog
