﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_INotifyService.sfdl.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_ServiceNames.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/pdm/srv/pdm_NotifyServiceImpl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/oe/oe_OperationModeApis.h>
#include <nn/util/util_Optional.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_SaveDataCommitThread.h>
#include <nn/pdm/detail/pdm_Time.h>
#include <nn/time.h>
#endif

namespace nn { namespace pdm {

namespace
{
    sf::ShimLibraryObjectHolder<pdm::detail::INotifyService> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    typedef void(*HolderInitializer)(sf::ShimLibraryObjectHolder<pdm::detail::INotifyService>&);

    detail::InitializationManager g_Initialization = NN_PDM_INITIALIZATION_INITIALIZER;

    void Initialize(HolderInitializer holderInitializer) NN_NOEXCEPT
    {
        g_Initialization.Initialize([&]
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
            detail::InitializeFs();
            detail::InitializeTime();
            detail::SaveDataCommitThread::Start();
#endif
            (*holderInitializer)(g_Holder);
        });
    }

    void Finalize() NN_NOEXCEPT
    {
        g_Initialization.Finalize([]
        {
#if defined( NN_BUILD_CONFIG_OS_WIN )
            detail::SaveDataCommitThread::Stop();
            detail::FinalizeFs();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
#endif
            g_Holder.FinalizeHolder();
        });
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )
    sf::UnmanagedServiceObject<pdm::detail::INotifyService, pdm::srv::NotifyServiceImpl> g_ServiceObject;

    void InitializeForDfc() NN_NOEXCEPT
    {
        Initialize([](sf::ShimLibraryObjectHolder<pdm::detail::INotifyService>& holder){
            holder.InitializeHolderDirectly(g_ServiceObject.GetShared());
        });
    }

#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    sf::SimpleAllInOneHipcClientManager<17> g_Manager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;

    void InitializeForHipc() NN_NOEXCEPT
    {
        Initialize([](sf::ShimLibraryObjectHolder<pdm::detail::INotifyService>& holder){
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_Manager.InitializeShimLibraryHolder(&holder, pdm::detail::ServiceNameForNotify));
        });
    }
#else
#error "unsupported os"
#endif
}

void InitializeForNotification() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    InitializeForDfc();
#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    InitializeForHipc();
#else
#error "unsupported os"
#endif
}

void FinalizeForNotification() NN_NOEXCEPT
{
    Finalize();
}

void NotifyAppletEvent(AppletEventType eventType, nn::ncm::ProgramId programId, uint32_t version, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder->NotifyAppletEvent(eventType, programId, version, static_cast<uint8_t>(appletId), storageId, logPolicy));
}

void NotifyLibraryAppletEvent(AppletEventType eventType, nn::ncm::ProgramId programId, applet::LibraryAppletMode libraryAppletMode, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder->NotifyLibraryAppletEvent(eventType, programId, static_cast<Bit8>(libraryAppletMode), static_cast<uint8_t>(appletId), storageId, logPolicy));
}

void NotifyOperationModeChangeEvent(oe::OperationMode operationMode) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder->NotifyOperationModeChangeEvent(static_cast<pdm::OperationMode>(operationMode)));
}

void NotifyPowerStateChangeEvent(PowerStateChangeEventType eventType) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder->NotifyPowerStateChangeEvent(eventType));
}

void NotifyClearAllEvent() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder->NotifyClearAllEvent());
}

void NotifyEventForDebug(const PlayEvent playEvent[], int count) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder->NotifyEventForDebug(sf::InArray<PlayEvent>(playEvent, count)));
}

Result SuspendUserAccountEventService(const account::Uid& user) NN_NOEXCEPT
{
    return g_Holder->SuspendUserAccountEventService(user);
}

Result ResumeUserAccountEventService(const account::Uid& user) NN_NOEXCEPT
{
    return g_Holder->ResumeUserAccountEventService(user);
}

}}
