﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/pdm/pdm_PrivateTypes.h>
#include <nn/pdm/pdm_SystemTypes.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/detail/pdm_Fs.h>
#include <nn/pdm/detail/pdm_Log.h>
#include <nn/pdm/detail/pdm_PlayEventBuffer.h>
#include <nn/pdm/detail/pdm_AccountPlayEventBuffer.h>
#include <nn/pdm/detail/pdm_Util.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace pdm { namespace detail {

PlayEventBuffer::PlayEventBuffer() NN_NOEXCEPT
    : EventEntryFileStream(sizeof(Header), sizeof(PlayEvent))
    , m_pPlayEventAddedEvent(nullptr)
    , m_IsInitialized(false)
{
}

void PlayEventBuffer::Initialize(uint32_t eventCountMax) NN_NOEXCEPT
{
    if (m_IsInitialized || OpenResult::AlreadyOpened == Open(eventCountMax, detail::MountName))
    {
        NN_DETAIL_PDM_INFO("[pdm] PlayEventBuffer: Already opened.\n");
        return;
    }
    m_IsInitialized = true;
    const auto header = GetHeader();
    NN_DETAIL_PDM_TRACE("[pdm] PlayEventBuffer: Open successful. Start = %d, Count = %d.\n", header.start, header.count);
    NN_UNUSED(header);
}

void PlayEventBuffer::Clear() NN_NOEXCEPT
{
    Discard();
    detail::CommitSaveData();
    m_EntryCache.Clear();
    NN_DETAIL_PDM_INFO("[pdm] PlayEventBuffer: PlayEvent is cleared.\n");
}

void PlayEventBuffer::Add(const pdm::PlayEvent& playEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);     // Only once initialization 想定.
    auto addedLock = m_EntryCache.Add(playEvent, [&]() -> void
    {
        Flush();
    });

    // AccountPlayEventBuffer へお裾分け。
    AccountPlayEventProvider::GetInstance().Import(playEvent);

    os::Event* pNotifier;
    if (detail::IsPowerOffEvent(playEvent))
    {
        // 電源OFF のイベント時は電源が切れる前に確実に保存するためここで書き込んでセーブデータをコミットする。
        Flush();
        detail::CommitSaveData();
    }
    else if (nullptr != (pNotifier = m_pPlayEventAddedEvent))
    {
        pNotifier->Signal();
    }
}

void PlayEventBuffer::Flush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);     // Only once initialization 想定.
    auto flushLock = m_EntryCache.SwapBuffer();
    if (false == flushLock.owns_lock())
    {
        // 追加するべきイベントがない。
        NN_DETAIL_PDM_TRACE("[pdm] PlayEventBuffer: There is no event in buffer to flush.\n");
        return;
    }
    Header header;
    auto pFlushBuffer = m_EntryCache.GetFlushBuffer();
    {
        auto fileLock = Write(pFlushBuffer->playEvent, pFlushBuffer->filledCount);
        header = GetHeaderUnsafe();
    }
    NN_DETAIL_PDM_TRACE("[pdm] PlayEventBuffer: Flush completed. Now Start = %d, Count = %d.\n", header.start, header.count);
    NN_UNUSED(header);
}

void PlayEventBuffer::SetPlayEventAddedEvent(os::Event* addedEvent) NN_NOEXCEPT
{
    m_pPlayEventAddedEvent = addedEvent;
}

uint32_t PlayEventBuffer::GetFilledBufferSpaceCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);   // Only once initialization 想定.
    return m_EntryCache.GetCachedEntryCount();
}

void PlayEventBuffer::GetReportData(ReportData* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    auto lock = AcquireLockForFile();
    pOut->count = GetCountUnsafe();
    pOut->startIndex = GetStartIndexUnsafe();
    pOut->lastIndex = GetLastIndexUnsafe();
}

}}}
