﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/pcv/pcv.h>

#include <nn/pcv/detail/pcv_IPcvService.h>
#include <nn/pcv/pcv_IArbitrationManager.sfdl.h>
#include <nn/pcv/pcv_IImmediateManager.sfdl.h>

#include "detail/pcv_ServiceName.h"

namespace nn {
namespace pcv {

namespace {

// Initialize の参照カウント。
// IsInitialized 関数のサポートの為に必要。
int g_InitializeCount = 0;

// 参照カウントを守る Mutex。
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

const int NumberOfClients = 3;

sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;

sf::ShimLibraryObjectHolder<detail::IPcvService> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
sf::ShimLibraryObjectHolder<IArbitrationManager> g_ArbitrationHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
sf::ShimLibraryObjectHolder<IImmediateManager> g_ImmediateHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    g_InitializeCount++;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ClientManager.InitializeShimLibraryHolder(&g_Holder, detail::PcvServiceName));
}

bool IsInitialized() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if ( g_InitializeCount == 0 )
    {
        return false;
    }

    return true;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    g_Holder.FinalizeHolder();
}

Result SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetPowerEnabled(static_cast<std::int32_t>(moduleId), enabled);
}

Result GetPossibleClockRates(ClockRatesListType* pOutType, ClockHz* pOutRates,
                             int* pOutCount, Module moduleId, int maxCount) NN_NOEXCEPT
{
    nn::sf::OutArray<uint32_t> outArray(pOutRates, MaxNumClockRates);
    int32_t type;
    int limitedMaxCount = (MaxNumClockRates < maxCount) ? MaxNumClockRates : maxCount;

    auto result = g_Holder.GetObject()->GetPossibleClockRates(&type, outArray,
        pOutCount, static_cast<std::int32_t>(moduleId), limitedMaxCount);

    *pOutType = static_cast<ClockRatesListType>(type);

    return result;
}

Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetClockEnabled(static_cast<std::int32_t>(moduleId), enabled);
}

Result SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetClockRate(static_cast<std::int32_t>(moduleId), static_cast<std::uint32_t>(clockRateHz));
}

Result GetClockRate(ClockHz* pClockRateHz, Module moduleId) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetClockRate(pClockRateHz, static_cast<std::int32_t>(moduleId));
}

Result GetState(ModuleState *pOutState, Module moduleId) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetState(pOutState, static_cast<std::int32_t>(moduleId));
}

Result SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetMinVClockRate(static_cast<std::int32_t>(moduleId), static_cast<std::uint32_t>(clockRateHz));
}

Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetReset(static_cast<std::int32_t>(moduleId), asserted);
}

Result SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetVoltageEnabled(static_cast<std::int32_t>(powerDomain), enabled);
}

Result GetVoltageEnabled(bool* pEnabled, PowerDomain powerDomain) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetVoltageEnabled(pEnabled, static_cast<std::int32_t>(powerDomain));
}

Result GetVoltageRange(MicroVolt* pMinVolt, MicroVolt* pMaxVolt, MicroVolt* pStepVolt, PowerDomain powerDomain) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetVoltageRange(pMinVolt, pMaxVolt, pStepVolt, static_cast<std::int32_t>(powerDomain));
}

Result SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetVoltageValue(static_cast<std::int32_t>(powerDomain), static_cast<std::uint32_t>(microVolt));
}

Result GetVoltageValue(MicroVolt* pMicroVolt, PowerDomain powerDomain) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetVoltageValue(pMicroVolt, static_cast<std::int32_t>(powerDomain));
}

Result GetTemperatureThresholds(TemperatureThreshold* pThresholds, int* pOutCount, int maxCount) NN_NOEXCEPT
{
    nn::sf::OutArray<TemperatureThreshold> outArray(pThresholds, MaxNumTemperatureThresholds);
    int limitedMaxCount = (MaxNumTemperatureThresholds < maxCount) ? MaxNumClockRates : maxCount;

    return g_Holder.GetObject()->GetTemperatureThresholds(outArray, pOutCount, limitedMaxCount);
}

Result SetTemperature(MilliC temperatureMilliC) NN_NOEXCEPT
{
    return g_Holder.GetObject()->SetTemperature(temperatureMilliC);
}

Result GetPowerClockInfoEvent(nn::os::SystemEventType* pOutSystemEvent, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT
{
    ::nn::sf::NativeHandle handle;

    Result result = g_Holder->GetPowerClockInfoEvent(&handle);

    if (result.IsSuccess())
    {
        ::nn::os::AttachReadableHandleToSystemEvent(pOutSystemEvent, handle.GetOsHandle(), handle.IsManaged(), eventClearMode);
    }

    handle.Detach();

    return result;
}

Result GetOscillatorClock(uint32_t* pClockRateHz) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetOscillatorClock(pClockRateHz);
}

Result GetDvfsTable(uint32_t* pOutClocks, int32_t* pOutVoltages, int32_t* pOutCount, Module moduleId, int32_t maxCount) NN_NOEXCEPT
{
    nn::sf::OutArray<uint32_t> outClocks(pOutClocks, DvfsTableMaxDataCount);
    nn::sf::OutArray<int32_t> outVoltages(pOutVoltages, DvfsTableMaxDataCount);

    int limitedMaxCount = (MaxNumClockRates < maxCount) ? DvfsTableMaxDataCount : maxCount;

    auto result = g_Holder.GetObject()->GetDvfsTable(outClocks, outVoltages, pOutCount, static_cast<std::int32_t>(moduleId), limitedMaxCount);

    return result;
}

Result GetModuleStateTable(ModuleState* pOutModuleStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    nn::sf::OutArray<ModuleState> outModuleStates(pOutModuleStates, ModuleStateTableSize);

    int limitedMaxCount = (ModuleStateTableSize < maxCount) ? ModuleStateTableSize : maxCount;

    return g_Holder.GetObject()->GetModuleStateTable(outModuleStates, pOutCount, limitedMaxCount);
}

Result GetPowerDomainStateTable(PowerDomainState* pOutPowerDomainStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    nn::sf::OutArray<PowerDomainState> outPowerDomainStates(pOutPowerDomainStates, PowerDomainStateTableSizeMax);

    int limitedMaxCount = (PowerDomainStateTableSizeMax < maxCount) ? PowerDomainStateTableSizeMax : maxCount;

    return g_Holder.GetObject()->GetPowerDomainStateTable(outPowerDomainStates, pOutCount, limitedMaxCount);
}

Result GetFuseInfo(uint32_t* pOutValues, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    nn::sf::OutArray<uint32_t> outValues(pOutValues, FuseValueCount);

    int limitedMaxCount = (FuseValueCount < maxCount) ? FuseValueCount : maxCount;

    return g_Holder.GetObject()->GetFuseInfo(outValues, pOutCount, limitedMaxCount);
}

Result GetDramId(uint32_t* pOutDramId) NN_NOEXCEPT
{
    return g_Holder.GetObject()->GetDramId(pOutDramId);
}

void InitializeForImmediate() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ClientManager.InitializeShimLibraryHolder(&g_ImmediateHolder, detail::ImmediateServiceName));
}

void FinalizeForImmediate() NN_NOEXCEPT
{
    g_ImmediateHolder.FinalizeHolder();
}

Result SetClockRateImmediately(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    return g_ImmediateHolder.GetObject()->SetClockRate(moduleId, clockRateHz);
}

void InitializeForArbitration() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ClientManager.InitializeShimLibraryHolder(&g_ArbitrationHolder, detail::ArbitrationServiceName));
}

void FinalizeForArbitration() NN_NOEXCEPT
{
    g_ArbitrationHolder.FinalizeHolder();
}

Result ReleaseControl(Module moduleId) NN_NOEXCEPT
{
    return g_ArbitrationHolder.GetObject()->ReleaseControl(moduleId);
}

Result PowerOn(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    return g_Holder.GetObject()->PowerOn(powerControlTarget, microVolt);
}

Result PowerOff(PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    return g_Holder.GetObject()->PowerOff(powerControlTarget);
}

Result ChangeVoltage(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    return g_Holder.GetObject()->ChangeVoltage(powerControlTarget, microVolt);
}

Result IsPoweredOn(bool* pOutIsPoweredOn, PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    return g_Holder.GetObject()->IsPoweredOn(pOutIsPoweredOn, powerControlTarget);
}

} // namespace pcv
} // namespace nn
