﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   PCV ドライバの API インタフェース部分。
 * @details
 */

#include <nn/pcv/detail/pcv_Log.h>
#include <nn/pcv/driver/pcv.h>
#include <nn/pcv/driver/pcv_Suspend.h>
#include <nn/result/result_HandlingUtility.h>

#include "detail/pcv_CheckDefaultStates.h"
#include "detail/pcv_Driver.h"
#include "detail/pcv_PowerController.h"

namespace nn {
namespace pcv {
namespace driver {

void Initialize() NN_NOEXCEPT
{
    detail::Initialize();

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    // Check default states of respective clock domains.
    if ( !detail::AreDefaultStatesExpected() )
    {
        NN_DETAIL_PCV_WARN("Some modules have unexpected states!\n");
    }

#endif

    detail::InitializePowerController();
}

bool IsInitialized() NN_NOEXCEPT
{
    return detail::IsInitialized();
}

void Finalize() NN_NOEXCEPT
{
    detail::FinalizePowerController();

    detail::Finalize();
}

Result SetPowerEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    return detail::SetPowerEnabled(moduleId, enabled);
}

Result GetPossibleClockRates(ClockRatesListType* pOutType, ClockHz* pOutRates,
                             int* pOutCount, Module moduleId, int maxCount) NN_NOEXCEPT
{
    return detail::GetPossibleClockRates(pOutType, pOutRates, pOutCount, moduleId, maxCount);
}

Result SetClockEnabled(Module moduleId, bool enabled) NN_NOEXCEPT
{
    return detail::SetClockEnabled(moduleId, enabled);
}

Result SetClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    return detail::SetClockRate(moduleId, clockRateHz);
}

Result GetClockRate(ClockHz* pClockRateHz, Module moduleId) NN_NOEXCEPT
{
    return detail::GetClockRate(pClockRateHz, moduleId);
}

Result GetState(ModuleState *pState, Module moduleId) NN_NOEXCEPT
{
    return detail::GetState(pState, moduleId);
}

Result SetMinVClockRate(Module moduleId, ClockHz clockRateHz) NN_NOEXCEPT
{
    return detail::SetMinVClockRate(moduleId, clockRateHz);
}

Result SetReset(Module moduleId, bool asserted) NN_NOEXCEPT
{
    return detail::SetReset(moduleId, asserted);
}

Result SetVoltageEnabled(PowerDomain powerDomain, bool enabled) NN_NOEXCEPT
{
    // TODO: SIGLO-79931 のワークアラウンドです。
    // Raptor の display ドライバの修正が完了され次第削除します。
    if ( (powerDomain == PowerDomain_Max77620_Ldo8) && detail::IsSioMcuSupported() )
    {
        NN_DETAIL_PCV_INFO("%s(PowerDomain_Max77620_Ldo8, %s) rejected.\n", __FUNCTION__, enabled ? "true" : "false");
        NN_RESULT_SUCCESS;
    }

    return detail::SetVoltageEnabled(powerDomain, enabled);
}

Result GetVoltageEnabled(bool* pEnabled, PowerDomain powerDomain) NN_NOEXCEPT
{
    return detail::GetVoltageEnabled(pEnabled, powerDomain);
}

Result GetVoltageRange(MicroVolt* pMinVolt, MicroVolt* pMaxVolt, MicroVolt* pStepVolt, PowerDomain powerDomain) NN_NOEXCEPT
{
    return detail::GetVoltageRange(pMinVolt, pMaxVolt, pStepVolt, powerDomain);
}

Result SetVoltageValue(PowerDomain powerDomain, MicroVolt microVolt) NN_NOEXCEPT
{
    // TODO: SIGLO-79931 のワークアラウンドです。
    // Raptor の display ドライバの修正が完了され次第削除します。
    if ( (powerDomain == PowerDomain_Max77620_Ldo8) && detail::IsSioMcuSupported() )
    {
        NN_DETAIL_PCV_INFO("%s(PowerDomain_Max77620_Ldo8, %d) rejected.\n", __FUNCTION__, microVolt);
        NN_RESULT_SUCCESS;
    }

    return detail::SetVoltageValue(powerDomain, microVolt);
}

Result GetVoltageValue(MicroVolt* pMicroVolt, PowerDomain powerDomain) NN_NOEXCEPT
{
    return detail::GetVoltageValue(pMicroVolt, powerDomain);
}

Result GetTemperatureThresholds(TemperatureThreshold* pThresholds, int* pOutCount, int maxCount) NN_NOEXCEPT
{
    return detail::GetTemperatureThresholds(pThresholds, pOutCount, maxCount);
}

Result SetTemperature(MilliC temperatureMilliC) NN_NOEXCEPT
{
    return detail::SetTemperature(temperatureMilliC);
}

Result SuspendClocks() NN_NOEXCEPT
{
    return detail::SuspendClocks();
}

Result ResumeClocks() NN_NOEXCEPT
{
    return detail::ResumeClocks();
}

Result SuspendVoltage() NN_NOEXCEPT
{
    return detail::SuspendVoltage();
}

Result ResumeVoltage() NN_NOEXCEPT
{
    return detail::ResumeVoltage();
}

Result PowerOn(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    return detail::PowerOn(powerControlTarget, microVolt);
}

Result PowerOff(PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    return detail::PowerOff(powerControlTarget);
}

Result ChangeVoltage(PowerControlTarget powerControlTarget, MicroVolt microVolt) NN_NOEXCEPT
{
    return detail::ChangeVoltage(powerControlTarget, microVolt);
}

Result IsPoweredOn(bool* pOutIsPoweredOn, PowerControlTarget powerControlTarget) NN_NOEXCEPT
{
    return detail::IsPoweredOn(pOutIsPoweredOn, powerControlTarget);
}

nn::os::SystemEventType* GetPowerClockInfoEvent() NN_NOEXCEPT
{
    return detail::GetPowerClockInfoEvent();
}

uint32_t GetOscillatorClock() NN_NOEXCEPT
{
    return detail::GetOscillatorClock();
}

Result GetDvfsTable(uint32_t* pOutClocks, int32_t* pOutVoltages, int32_t* pOutCount, Module moduleId, int32_t maxCount) NN_NOEXCEPT
{
    return detail::GetDvfsTable(pOutClocks, pOutVoltages, pOutCount, moduleId, maxCount);
}

void GetModuleStateTable(ModuleState* pOutModuleStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    return detail::GetModuleStateTable(pOutModuleStates, pOutCount, maxCount);
}

void GetPowerDomainStateTable(PowerDomainState* pOutPowerDomainStates, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    return detail::GetPowerDomainStateTable(pOutPowerDomainStates, pOutCount, maxCount);
}

void GetFuseInfo(uint32_t* pOutValues, int32_t* pOutCount, int32_t maxCount) NN_NOEXCEPT
{
    return detail::GetFuseInfo(pOutValues, pOutCount, maxCount);
}

void GetDramId(uint32_t* pOutDramId) NN_NOEXCEPT
{
    return detail::GetDramId(pOutDramId);
}

} // namespace driver
} // namespace pcv
} // namespace nn
