﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_EventObjectHolder.h>
#include <nn/pctl/detail/pctl_ResultHandler.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>

#include <nn/util/util_LockGuard.h>

#include <cstring>

namespace nn { namespace pctl {

nn::Result UnlockRestrictionTemporarily(const char* code) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(code);

    // NULL文字込みの長さとする
    size_t codeLength = std::strlen(code) + 1;
    auto result = detail::ipc::GetService()->UnlockRestrictionTemporarily(nn::sf::InArray<char>(code, codeLength));
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);

    return result;
}

nn::Result UnlockSystemSettingsRestriction(const char* code) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(code);

    // NULL文字込みの長さとする
    size_t codeLength = std::strlen(code) + 1;
    auto result = detail::ipc::GetService()->UnlockSystemSettingsRestriction(nn::sf::InArray<char>(code, codeLength));
    NN_PCTL_DETAIL_ABORT_IF_UNEXPECTED_RESULT(result);

    return result;
}

void SetPinCode(const char* code) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(code == nullptr || (std::strlen(code) >= 4 && std::strlen(code) <= 8), "The length of 'code' must be between 4 and 8");

    // code != nullptr の場合はNULL文字込みの長さとする
    // code == nullptr の場合は長さ 0 とする
    size_t codeLength = (code != nullptr ? std::strlen(code) + 1 : 0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->SetPinCode(nn::sf::InArray<char>(code, codeLength)));
}

int GetPinCodeLength() NN_NOEXCEPT
{
    int value = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetPinCodeLength(&value));
    return value;
}

int GetPinCode(char* outCode, int maxLengthWithNull) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCode);
    NN_SDK_REQUIRES_GREATER(maxLengthWithNull, 0);
    if (maxLengthWithNull <= 0)
    {
        return 0;
    }

    int length = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetPinCode(&length, nn::sf::OutArray<char>(outCode, static_cast<size_t>(maxLengthWithNull))));
    return length;
}

void NotifyWrongPinCodeInputManyTimes() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->NotifyWrongPinCodeInputManyTimes());
}

void GenerateInquiryCode(InquiryCode* pOutCodeData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCodeData);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GenerateInquiryCode(pOutCodeData));
}

bool CheckMasterKey(const InquiryCode& codeData, const char* masterKey) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(masterKey);

    // NULL文字込みの長さとする
    size_t masterKeyLength = std::strlen(masterKey) + 1;
    bool value = false;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->CheckMasterKey(&value, codeData, nn::sf::InArray<char>(masterKey, masterKeyLength)));
    return value;
}

nn::os::SystemEvent* GetPinCodeChangedEvent() NN_NOEXCEPT
{
    // プロセス単位のインスタンスとする
    NN_FUNCTION_LOCAL_STATIC(detail::EventObjectHolder, eventChanged);
    NN_FUNCTION_LOCAL_STATIC(bool, eventUnavailable, = false);

    NN_UTIL_LOCK_GUARD(eventChanged);

    if (!eventUnavailable && !eventChanged.IsInitialized())
    {
        nn::sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ipc::GetService()->GetPinCodeChangedEvent(&eventHandle));
        // InvalidNativeHandle の場合は利用できない状態とする
        if (eventHandle.GetOsHandle() == nn::os::InvalidNativeHandle)
        {
            eventUnavailable = true;
        }
        else
        {
            eventChanged.InitializeWithReadableHandle(
                eventHandle.GetOsHandle(),
                eventHandle.IsManaged(),
                nn::os::EventClearMode::EventClearMode_AutoClear
                );
            eventHandle.Detach();
            eventUnavailable = false;
        }
    }
    return eventUnavailable ? nullptr : eventChanged.GetSystemEvent();
}

}}
