﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_UpdateDevice.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/watcher/pctl_Notifications.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_UpdateDeviceDispatcher.h>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

////////////////////////////////////////////////////////////////////////////////

nn::Result OnlineCheckExecutor::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, bufferInfo, m_pCancelable));

    NN_RESULT_DO(
        dispatcher::OnlineCheckDispatcher::Execute(bufferInfo, m_pCancelable, tokenHolder.GetToken(), m_DeviceId)
        );
    NN_RESULT_SUCCESS;
}

void OnlineCheckExecutor::Cancel() NN_NOEXCEPT
{
    m_pCancelable->Cancel();
}

////////////////////////////////////////////////////////////////////////////////

nn::Result UpdateDeviceExecutor::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    if (g_pWatcher->GetNetworkManager().IsLastUpdateDeviceSuccess())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Already posted (update device).\n");
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_PCTL_TRACE("[pctl] Start post update device.\n");

    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, bufferInfo, m_pCancelable));

    NN_RESULT_DO(
        ExecuteUpdateDevice(bufferInfo, m_pCancelable, tokenHolder, m_DeviceId, &m_EtagSynchronized, &m_NotificationToken)
        );

    NN_RESULT_SUCCESS;
}

void UpdateDeviceExecutor::Cancel() NN_NOEXCEPT
{
    m_pCancelable->Cancel();
}

// static
nn::Result UpdateDeviceExecutor::ExecuteUpdateDevice(common::NetworkBuffer& bufferInfo,
    common::Cancelable* pCancelable,
    TokenHolder& tokenHolder,
    ServerDeviceId deviceId,
    const EtagInfo* etagSynchronized,
    nn::npns::NotificationToken* pNotificationTokenBuffer) NN_NOEXCEPT
{
    if (!g_pWatcher->GetNetworkManager().IsNotificationTokenRetrieved())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Refresh notification token.\n");
        NN_RESULT_DO(
            CreateNotificationToken::Execute(*pNotificationTokenBuffer, pCancelable)
        );
        g_pWatcher->GetNetworkManager().StoreNotificationToken(*pNotificationTokenBuffer);
    }

    NN_DETAIL_PCTL_TRACE("[pctl] Post update device.\n");

    NN_RESULT_DO(
        dispatcher::UpdateDeviceDispatcher::Execute(bufferInfo, pCancelable, tokenHolder.GetToken(), deviceId, etagSynchronized)
    );

    g_pWatcher->GetNetworkManager().SetLastUpdateDeviceSuccess(true);
    NN_DETAIL_PCTL_TRACE("[pctl] Post update device done.\n");

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////

nn::Result UpdateDeviceAlarmSettingExecutor::Execute(common::NetworkBuffer& bufferInfo) NN_NOEXCEPT
{
    // 「現在時刻」からリセットタイミングを取得
    nn::time::PosixTime timeInvisibleUntil;
    NN_RESULT_DO(WatcherEventManager::GetTimeToResetAlarmDisable(&timeInvisibleUntil, m_TimeNow));

    NN_DETAIL_PCTL_TRACE("[pctl] Post update device for alarm setting.\n");

    TokenHolder tokenHolder;
    NN_RESULT_DO(g_pWatcher->GetNetworkManager().AcquireAuthenticationToken(tokenHolder, bufferInfo, m_pCancelable));

    NN_RESULT_TRY(
        dispatcher::UpdateDeviceAlarmSettingDispatcher::Execute(bufferInfo, m_pCancelable, tokenHolder.GetToken(), m_DeviceId,
            m_IsDisabled, timeInvisibleUntil)
        )
        NN_RESULT_CATCH(nn::pctl::ResultServerOverTheTimeLimit)
        {
            // 設定は反映させないが想定内のResultであるため成功として返す
            NN_DETAIL_PCTL_TRACE("[pctl The alarm setting is no longer valid (ResultServerOverTheTimeLimit). Do nothing.\n");
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    // 設定に成功したときのみ結果を反映させる
    NN_DETAIL_PCTL_TRACE("[pctl] Post update device for alarm setting done. Update local status.\n");
    g_pWatcher->GetWatcherEventManager().SetAlarmDisabled(m_IsDisabled, m_TimeNow);

    NN_RESULT_SUCCESS;
}

void UpdateDeviceAlarmSettingExecutor::Cancel() NN_NOEXCEPT
{
    m_pCancelable->Cancel();
}

}}}}}
