﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_NetworkManager.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_Result.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ResultPrivate.h>
#include <nn/npns/npns_Api.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/os/os_Tick.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/common/pctl_SystemInfo.h>
#include <nn/pctl/detail/service/overlay/pctl_OverlaySender.h>
#include <nn/pctl/detail/service/watcher/pctl_Authentication.h>
#include <nn/pctl/detail/service/watcher/pctl_DeviceEvent.h>
#include <nn/pctl/detail/service/watcher/pctl_IntermittentOperation.h>
#include <nn/pctl/detail/service/watcher/pctl_MiiDownloader.h>
#include <nn/pctl/detail/service/watcher/pctl_Pairing.h>
#include <nn/pctl/detail/service/watcher/pctl_RetrieveOwners.h>
#include <nn/pctl/detail/service/watcher/pctl_RetrieveSettings.h>
#include <nn/pctl/detail/service/watcher/pctl_UnlinkDevice.h>
#include <nn/pctl/detail/service/watcher/pctl_UpdateDevice.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    static const char NetworkConfigFileName[] = "watcher.dat";
    static const char NetworkDeviceStatusFileName[] = "netDevice.dat";

    const char* GetCurrentEnvironment() NN_NOEXCEPT
    {
        static nn::nsd::EnvironmentIdentifier envData = { { 0 } };
        if (envData.value[0] == 0)
        {
            nn::nsd::GetEnvironmentIdentifier(&envData);
        }
        return envData.value;
    }

    // outFileName は NetworkManager::MaxMiiImageFileNameLength の長さだけ保持可能なバッファー
    void GenerateMiiFileName(char* outFileName, NintendoAccountId naId, const char* miiUrl) NN_NOEXCEPT
    {
        // miiUrl における '?' の位置
        const char* queryStringPosition = std::strchr(miiUrl, '?');
        if (queryStringPosition == nullptr)
        {
            queryStringPosition = miiUrl + std::strlen(miiUrl);
        }
        // miiUrl におけるファイル名部分(パスを除いたもの)の位置
        const char* fileNamePosition = queryStringPosition;
        while (fileNamePosition > miiUrl)
        {
            if (fileNamePosition[-1] == '/')
            {
                break;
            }
            --fileNamePosition;
        }

        // 長さが長すぎる場合は先頭の文字を削る
        // (22 = 16 (NA-ID) + 4 ("mii/") + 2 (separator, null-char))
        if (static_cast<size_t>(queryStringPosition - fileNamePosition) > NetworkManager::MaxMiiImageFileNameLength - 22)
        {
            fileNamePosition = queryStringPosition - NetworkManager::MaxMiiImageFileNameLength + 22;
        }
        // (block)
        {
            int r = nn::util::SNPrintf(outFileName, NetworkManager::MaxMiiImageFileNameLength,
                "mii/%016llx-", static_cast<uint64_t>(naId));
            NN_SDK_ASSERT(r == 4 + 16 + 1);
            NN_UNUSED(r);
        }
        // (block)
        {
            int copyLen = static_cast<int>(queryStringPosition - fileNamePosition) + 1;
            nn::util::Strlcpy(outFileName + 21, fileNamePosition, copyLen);
        }
    }
}

NetworkManager::NetworkManager(void* threadStackBuffer, size_t stackBufferSize) NN_NOEXCEPT :
    m_TaskThread(),
    m_EventHolderExit(),
    m_EventHolderMessage(),
    m_EventHolderNpns(),
    m_SysEventUnlinked(nn::os::EventClearMode_ManualClear, true),
    m_pNextAsyncContext(nullptr),
    m_pRunningBackgroundContext(nullptr),
    m_pNextBackgroundAsyncContext(nullptr),
    m_LastBackgroundTaskResult(nn::ResultSuccess()),
    m_IsThreadRunning(false),
    m_IsNetworkDisabledOnInit(false),
    m_IsBackgroundCanceled(false),
    m_IsNotificationTokenRetrieved(false), // 初期値=コールドブート時は常に false
    m_IsLastUpdateDeviceSuccess(false), // 初期値は常に false
    m_IsOverlayDisabled(false),
    m_TokenRefCount(),
    //m_Config(),
    m_Cancelable(),
    m_CancelableForBackground(),
    m_CancelableForIntermittent(),
    m_pBuffer(nullptr),
    m_MiiImageFileName(nullptr),
    m_MiiImageFileNameBuffer(nullptr),
    m_MiiImageContentType(nullptr)
{
    std::memset(&m_Config, 0, sizeof(m_Config));
    std::memset(&m_NotificationToken, 0, sizeof(m_NotificationToken));
    m_TokenRefCount = 0;

    m_IsNetworkDisabledOnInit = g_pMain->GetSettingsManager().IsAllFeaturesDisabled();

    if (!m_IsNetworkDisabledOnInit)
    {
        nn::os::InitializeMultiWait(&m_MultiWait);

        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderExit.m_WaitHolder);
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderMessage.m_WaitHolder);

        m_pBuffer = AllocateMemoryBlock<common::NetworkBuffer>();
        NN_ABORT_UNLESS_NOT_NULL(m_pBuffer);

        m_MiiImageFileName = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * MaxMiiImageFileNameLength));
        NN_ABORT_UNLESS_NOT_NULL(m_MiiImageFileName);
        *m_MiiImageFileName = 0;

        m_MiiImageFileNameBuffer = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * MaxMiiImageFileNameLength));
        NN_ABORT_UNLESS_NOT_NULL(m_MiiImageFileNameBuffer);
        *m_MiiImageFileNameBuffer = 0;

        m_MiiImageContentType = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * MaxMiiImageContentTypeLength));
        NN_ABORT_UNLESS_NOT_NULL(m_MiiImageContentType);
        *m_MiiImageContentType = 0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_TaskThread, StaticThreadProc, this,
            threadStackBuffer, stackBufferSize, NN_SYSTEM_THREAD_PRIORITY(pctl, NetworkManager)));
        nn::os::SetThreadNamePointer(&m_TaskThread, NN_SYSTEM_THREAD_NAME(pctl, NetworkManager));
    }
}

NetworkManager::~NetworkManager() NN_NOEXCEPT
{
    // NetworkManager 内で無効化したまま終了している場合は有効化しておく
    if (m_IsOverlayDisabled)
    {
        overlay::EnableOverlayNotification();
    }

    // コンストラクターで disabled でなければ
    // 途中で disabled になっても m_IsNetworkDisabledOnInit は false のまま
    if (!m_IsNetworkDisabledOnInit)
    {
        m_Cancelable.Cancel();
        // バックグラウンドタスクもキャンセルする
        m_CancelableForBackground.Cancel();
        m_CancelableForIntermittent.Cancel();
        if (m_IsThreadRunning)
        {
            m_EventHolderExit.m_Event.Signal();
            //nn::os::WaitThread(&m_TaskThread);
        }
        nn::os::DestroyThread(&m_TaskThread);

        // "mii/" とそれ以下のファイルを削除する(エラーは無視)
        common::FileSystem::DeleteDirectory("mii", true);

        FreeMemoryBlock(m_MiiImageContentType);
        FreeMemoryBlock(m_MiiImageFileNameBuffer);
        FreeMemoryBlock(m_MiiImageFileName);
        FreeMemoryBlock(m_pBuffer);

        nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
        nn::os::FinalizeMultiWait(&m_MultiWait);
    }
}

// static
nn::Result NetworkManager::LoadConfig(bool* pOutRead, Config& outConfig) NN_NOEXCEPT
{
    size_t size = 0;
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, NetworkConfigFileName);
        NN_RESULT_THROW_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            result);
        if (result.IsSuccess())
        {
            NN_RESULT_DO(stream.Read(&size, 0, &outConfig, sizeof(Config)));
        }
    }
    if (size != sizeof(Config))
    {
        std::memset(&outConfig, 0, sizeof(Config));
        if (pOutRead != nullptr)
        {
            *pOutRead = false;
        }
        NN_DETAIL_PCTL_TRACE("[pctl] Watcher config not loaded. (size = %llu)\n", static_cast<uint64_t>(size));
    }
    else
    {
        if (pOutRead != nullptr)
        {
            *pOutRead = true;
        }
        std::memset(outConfig._reserved2, 0, sizeof(outConfig._reserved2));
        NN_DETAIL_PCTL_TRACE("[pctl] Loaded watcher config:\n");
        NN_DETAIL_PCTL_TRACE("[pctl]   isPairingActive: %s\n", outConfig.isPairingActive != 0 ? "true" : "false");
        NN_DETAIL_PCTL_TRACE("[pctl]   statusFlags: 0x%02lX\n", static_cast<uint32_t>(outConfig.statusFlags));
        NN_DETAIL_PCTL_TRACE("[pctl]   deviceId: 0x%016llX\n", outConfig.deviceId);
        NN_DETAIL_PCTL_TRACE("[pctl]   environment: %s\n", outConfig.environment.value);
        NN_DETAIL_PCTL_TRACE("[pctl]   tokenRetrieved: %lld\n", outConfig.token.tokenRetrieved.value);
        NN_DETAIL_PCTL_TRACE("[pctl]   tokenExpiresAt: %lld\n", outConfig.token.tokenExpiresAt);
    }
    NN_RESULT_SUCCESS;
}

// static
nn::Result NetworkManager::SaveConfig(const Config& config) NN_NOEXCEPT
{
    {
        common::FileStream stream;
        NN_RESULT_DO(common::FileSystem::OpenWrite(&stream, NetworkConfigFileName, sizeof(Config)));
        NN_RESULT_DO(stream.Write(0, &config, sizeof(Config)));
        NN_RESULT_DO(stream.Flush());
    }
    NN_RESULT_DO(common::FileSystem::Commit());

    NN_RESULT_SUCCESS;
}

// static
void NetworkManager::ClearPairingInfoFromConfig(Config& config) NN_NOEXCEPT
{
    config.isPairingActive = 0;
    config.statusFlags = 0;
    config.deviceId = 0;
    config.etagForSettings.lastUpdatedAt = 0;
    // DeviceAuthNToken, NotificationToken はリセットしない
}

void NetworkManager::LoadConfig() NN_NOEXCEPT
{
    if (m_IsNetworkDisabledOnInit)
    {
        return;
    }

    {
        NN_UTIL_LOCK_GUARD(m_MutexSaveData);
        LoadConfigNoLock();
        LoadDeviceStatusNoLock();
        // "mii/" とそれ以下のファイルを一旦削除する(エラーは無視)
        common::FileSystem::DeleteDirectory("mii", true);
        // "mii/" ディレクトリを作成
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::CreateDirectory("mii"));
    }

    // 連携解除済み状態である場合はシグナルしておく
    if ((m_Config.statusFlags & StatusFlags_NeedsNotifyUnlinked) != 0)
    {
        m_SysEventUnlinked.Signal();
    }
    // 連携済みで旧バージョンから初めて起動した場合は、このタイミングでのpdmのデータをすべてスキップする
    if (IsPairingActive() &&
        m_Config.lastPdmAppEventOffset == 0 && m_Config.lastPdmAccountEventOffset == 0)
    {
        ApplicationStateManager::SkipAllPlayDataBasedEvents(false);
    }
}

void NetworkManager::StartThread() NN_NOEXCEPT
{
    if (m_IsNetworkDisabledOnInit)
    {
        return;
    }

    nn::os::StartThread(&m_TaskThread);
    m_IsThreadRunning = true;
}

nn::Result NetworkManager::AcquireAuthenticationToken(TokenHolder& holder, common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    NN_UTIL_LOCK_GUARD(m_MutexToken);

    // 環境の変更を検出した場合は authToken がクリアされる
    ClearTokensIfEnvironmentChanged();

    // (m_Config.token.tokenExpiresAt <= 0 の場合最後の else if が true にならないはず)
    if (m_Config.token.authToken[0] != 0 && m_Config.token.tokenExpiresAt > 0)
    {
        // 無効化までの時間が問題なければそのtokenを使う
        nn::time::SteadyClockTimePoint pointNow;
        auto result = nn::time::StandardSteadyClock::GetCurrentTimePoint(&pointNow);
        if (result.IsSuccess())
        {
            int64_t seconds = 0;
            result = nn::time::GetSpanBetween(&seconds, m_Config.token.tokenRetrieved, pointNow);
            if (result.IsFailure())
            {
                NN_DETAIL_PCTL_INFO("[pctl] nn::time::GetSpanBetween returns 0x%08lX\n", result.GetInnerValueForDebug());
            }
            // 無効化までの時刻が1時間を超えていればそのまま使う
            // (無効化までの時刻が1時間以下の場合は取り直すようにする)
            else if (seconds >= 0 && m_Config.token.tokenExpiresAt - seconds > 60 * 60)
            {
                NN_DETAIL_PCTL_TRACE("[pctl] Use cached auth token: seconds from retrieved = %lld, expiresAt = %lld\n",
                    seconds, m_Config.token.tokenExpiresAt, m_Config.token.authToken);
                holder.SetTokenData(m_Config.token.authToken, &m_TokenRefCount);
                NN_RESULT_SUCCESS;
            }
        }
        else
        {
            NN_DETAIL_PCTL_WARN("[pctl] Failed to retrieve steady clock (0x%08X). Device authentication token will be retrieved every time.\n", result.GetInnerValueForDebug());
        }
    }

    // 参照中の場合は待機する: m_MutexToken のロックはここでしか行わないのでロックしたままにする
    while (m_TokenRefCount > 0)
    {
        NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), nn::pctl::ResultCanceled());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    // 有効な値が残っている場合は一旦無効化した値をセーブデータに保存
    if (m_Config.token.authToken[0] != 0)
    {
        NN_UTIL_LOCK_GUARD(m_MutexSaveData);
        m_Config.token.authToken[0] = 0;
        SaveConfigNoLock();
    }

    NN_RESULT_DO(Authentication::GetInstance().RetrieveDeviceAuthenticationToken(bufferInfo, pCancelable));
    if (m_Config.token.authToken[0] == 0)
    {
        NN_SDK_ASSERT(false, "RetrieveDeviceAuthenticationToken succeeded but token is not valid");
        NN_RESULT_THROW(nn::pctl::ResultUnexpected());
    }

    holder.SetTokenData(m_Config.token.authToken, &m_TokenRefCount);
    NN_RESULT_SUCCESS;
}

void NetworkManager::StoreDeviceAuthenticationToken(const AuthenticationTokenData& token) NN_NOEXCEPT
{
    NN_STATIC_ASSERT((std::is_same<decltype(m_Config.token), AuthenticationTokenData>::value));

    // AcquireAuthenticationToken 内部の Authentication::RetrieveDeviceAuthenticationToken からのみ
    // 呼び出される想定なのでここでは RefCount チェックを行わない
    // (RefCount も 0 のはず)
    NN_SDK_ASSERT(m_MutexToken.IsLockedByCurrentThread());
    NN_SDK_ASSERT(m_TokenRefCount == 0);
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);
    std::memcpy(&m_Config.token, &token, sizeof(token));
    SaveConfigNoLock();
}

void NetworkManager::GetNotificationToken(nn::npns::NotificationToken& token) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(
        (std::is_same<std::remove_reference<decltype(token)>::type, decltype(m_NotificationToken)>::value)
    );

    // MEMO: 永続化している NotificationToken があるのであれば
    //       ここで ClearTokensIfEnvironmentChanged を呼び出す
    //       (ClearTokensIfEnvironmentChanged 内の処理も修正する)

    // m_IsNotificationTokenRetrieved は特に見ない
    std::memcpy(&token, &m_NotificationToken, sizeof(m_NotificationToken));
}

void NetworkManager::StoreNotificationToken(const nn::npns::NotificationToken& token) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(
        (std::is_same<
            std::remove_const<std::remove_reference<decltype(token)>::type>::type,
            decltype(m_NotificationToken)
        >::value)
    );

    std::memcpy(&m_NotificationToken, &token, sizeof(token));
    m_IsNotificationTokenRetrieved = true;
    // NotificationTokenは現状保存していないのでコメントアウト
    //{
    //    NN_UTIL_LOCK_GUARD(m_MutexSaveData);
    //    SaveConfigNoLock();
    //}
}

void NetworkManager::StorePairingInfo(ServerDeviceId deviceId) NN_NOEXCEPT
{
    // これまでに溜められていたpdmイベントをスキップする
    m_Config.lastPdmAppEventOffset = 0;
    m_Config.lastPdmAccountEventOffset = 0;
    ApplicationStateManager::SkipAllPlayDataBasedEvents(true); // このタイミングでは未連携状態なので true を指定

    {
        NN_UTIL_LOCK_GUARD(m_MutexSaveData);

        m_Config.isPairingActive = 1;
        m_Config.deviceId = deviceId;
        m_Config.statusFlags &= ~StatusFlags_NeedsNotifyUnlinked;
        m_SysEventUnlinked.Clear();
        SaveConfigNoLock();
    }
}

void NetworkManager::ClearPairingInfo(bool notifyEvent) NN_NOEXCEPT
{
    // バックグラウンド処理をキャンセルする
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        // 既にバックグラウンド処理がセットされていたらそれは完全に破棄する(実行しない)
        if (m_pNextBackgroundAsyncContext != nullptr)
        {
            m_pNextBackgroundAsyncContext->CloseContext();
            m_pNextBackgroundAsyncContext = nullptr;
        }
    }
    m_CancelableForBackground.Cancel();
    m_CancelableForIntermittent.Cancel();
    // 保持データから連携情報を削除して保存する
    {
        NN_UTIL_LOCK_GUARD(m_MutexSaveData);

        ClearPairingInfoFromConfig(m_Config);
        // 通知を行う場合はフラグも立てておく(再起動後もSignalされるようにする)
        if (notifyEvent)
        {
            m_Config.statusFlags |= StatusFlags_NeedsNotifyUnlinked;
        }
        SaveConfigNoLock();
    }
    // 貯めていたイベントも不要になるので削除
    g_pWatcher->GetWatcherEventStorage().DiscardAllEvents();
    // 連携解除に伴う処理を実行
    g_pWatcher->GetWatcherEventManager().ProcessPairingDeleted();
    // 連係解除時には制限対象外リストもクリア
    g_pMain->GetSettingsManager().ClearExemptApplicationList();
    // 再連携時に NotificationToken を取り直すようにする
    m_IsNotificationTokenRetrieved = false;

    if (notifyEvent)
    {
        m_SysEventUnlinked.Signal();
    }
}

void NetworkManager::StoreEtagForSettings(const EtagInfo* etagForSettings) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);

    if (etagForSettings == nullptr)
    {
        m_Config.etagForSettings.lastUpdatedAt = 0;
    }
    else
    {
        std::memcpy(&m_Config.etagForSettings, etagForSettings, sizeof(EtagInfo));
    }
    SaveConfigNoLock();
}

bool NetworkManager::RefreshDeviceStatus() NN_NOEXCEPT
{
    // 無効化中は何もしない(この処理が必要ない)
    if (IsNetworkFeatureDisabled())
    {
        return false;
    }

    NN_UTIL_LOCK_GUARD(m_MutexSaveData);
    bool result = false; // 不一致(= 更新あり)なら true

    // 言語
    {
        nn::settings::Language lang;
        if (common::GetSystemLanguage(&lang))
        {
            auto& data = LanguageValues[lang];
            size_t length = data.length + 1; // NULL文字込み
            if (length > std::extent<decltype(m_DeviceStatus.language)>::value)
            {
                length = std::extent<decltype(m_DeviceStatus.language)>::value;
            }
            if (std::strncmp(m_DeviceStatus.language, data.value, std::extent<decltype(m_DeviceStatus.language)>::value) != 0)
            {
                result = true;
                std::memcpy(m_DeviceStatus.language, data.value, sizeof(char) * length);
            }
        }
        else
        {
            if (m_DeviceStatus.language[0] != 0)
            {
                result = true;
                m_DeviceStatus.language[0] = 0;
            }
        }
    }
    // 地域
    {
        nn::settings::system::RegionCode region;
        nn::settings::system::GetRegionCode(&region);
        if (region != m_DeviceStatus.regionCode)
        {
            result = true;
            m_DeviceStatus.regionCode = region;
        }
    }
    // シリアル番号
    {
        nn::settings::system::SerialNumber num;
#if NN_BUILD_CONFIG_OS_WIN
        nn::util::Strlcpy(num.string, "XNX0000000001", 13 + 1);
#else
        nn::settings::system::GetSerialNumber(&num);
#endif
        if (std::strncmp(m_DeviceStatus.serialNumber.string, num.string,
            std::extent<decltype(m_DeviceStatus.serialNumber.string)>::value) != 0)
        {
            result = true;
            std::memcpy(&m_DeviceStatus.serialNumber, &num, sizeof(m_DeviceStatus.serialNumber));
        }
    }
    // ファームウェアバージョン
    {
        char versionText[common::MaxSystemVersionStringLength];
        auto version = common::GetSystemVersion(versionText);
        if (m_DeviceStatus.systemVersion != version)
        {
            result = true;
            m_DeviceStatus.systemVersion = version;
            NN_STATIC_ASSERT(sizeof(m_DeviceStatus.systemVersionText) == sizeof(versionText));
            std::memcpy(m_DeviceStatus.systemVersionText, versionText, sizeof(versionText));
        }
    }

    if (result)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(SaveDeviceStatus(m_DeviceStatus));
    }

    // NotificationToken
    if (!IsNotificationTokenRetrieved())
    {
        // NotificationToken未取得の場合は更新されるようにする
        // - 更新ありとマークするのみ(実際の更新には通信処理が必要)
        // ※ SaveDeviceStatus は不要なのでこの位置に記述
        // MEMO: 起動時は未取得状態になるのでここは true になる
        NN_DETAIL_PCTL_TRACE("[pctl] Needs to refresh notification token\n");
        result = true;
    }

    return result;
}

void NetworkManager::GetDeviceStatus(DeviceStatus& outDeviceStatus) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);

    std::memcpy(&outDeviceStatus, &m_DeviceStatus, sizeof(DeviceStatus));
}

void NetworkManager::UpdateLastPlayDataManagerOffsets(uint32_t appEventOffset, uint32_t accountEventOffset) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);

    m_Config.lastPdmAppEventOffset = appEventOffset;
    m_Config.lastPdmAccountEventOffset = accountEventOffset;
    SaveConfigNoLock();
}

nn::Result NetworkManager::RequestPairing(common::AsyncContext** ppContext, ServerDeviceId* outDeviceId, size_t* outActualOwnerCount,
    const char* pairingCode, size_t codeLength, ServerDeviceOwner* pOwners, size_t maxOwnerCount) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();
    watcher::RequestPairing* pContext = new watcher::RequestPairing(&m_Cancelable);
    pContext->SetParameters(outDeviceId, outActualOwnerCount, pairingCode, codeLength, pOwners, maxOwnerCount);
    auto result = pContext->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pContext;
        return result;
    }

    *ppContext = pContext;
    return SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::RequestAuthorizePairing(common::AsyncContext** ppContext, ServerDeviceId deviceId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();
    watcher::AuthorizePairing* pContext = new watcher::AuthorizePairing(&m_Cancelable);
    pContext->SetParameters(deviceId);
    auto result = pContext->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pContext;
        return result;
    }

    *ppContext = pContext;
    return SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::RequestRetrieveOwners(common::AsyncContext** ppContext, size_t* outActualOwnerCount,
    ServerDeviceOwner* pOwners, size_t maxOwnerCount, ServerDeviceId deviceId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();
    watcher::RetrieveDeviceOwnersExecutor* pContext = new watcher::RetrieveDeviceOwnersExecutor(&m_Cancelable);
    pContext->SetParameters(outActualOwnerCount, pOwners, maxOwnerCount, deviceId);
    auto result = pContext->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pContext;
        return result;
    }

    *ppContext = pContext;
    return SetNextAsyncContext(pContext);
}

void NetworkManager::RequestOnlineCheckResponseBackground(ServerDeviceId deviceId) NN_NOEXCEPT
{
    // 無効化中は無視してキャンセル扱いにする
    if (m_EventHolderExit.m_Event.TryWait() || IsNetworkFeatureDisabled())
    {
        return;
    }

    m_CancelableForBackground.Reset();
    watcher::OnlineCheckExecutor* pContext;
    // この処理はNPNS由来で行われる処理のため、半起床中に受信したら
    // 間欠起動処理として扱うようにする
    if (IntermittentOperation::IsInHalfAwake())
    {
        pContext = new IntermittentContext<OnlineCheckExecutor, common::Cancelable*>(&m_CancelableForBackground);
    }
    else
    {
        pContext = new OnlineCheckExecutor(&m_CancelableForBackground);
    }
    pContext->SetParameters(deviceId);
    pContext->SetBackgroundMode(true);

    SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::RequestPostEvents(common::AsyncContext** ppContext, ServerDeviceId deviceId, size_t* outCount, EventData* pEventData, size_t eventDataSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();

    watcher::UploadDeviceEventsExecutor* pContext = new watcher::UploadDeviceEventsExecutor(&m_Cancelable);
    pContext->SetParameters(deviceId, outCount, pEventData, eventDataSize);
    pContext->SetBackgroundMode(false);

    // バックグラウンド取得待ちを解除する
    if (IsWatcherAvailable())
    {
        g_pWatcher->GetWatcherEventManager().ClearWaitingForRetrieveSettingsBackground();
    }
    // フォアグラウンドでの待機用にシステムイベントを生成する
    auto result = pContext->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pContext;
        return result;
    }

    if (ppContext != nullptr)
    {
        *ppContext = pContext;
    }
    return SetNextAsyncContext(pContext);
}

void NetworkManager::RequestPostEventsBackground(ServerDeviceId deviceId) NN_NOEXCEPT
{
    // 無効化中は無視してキャンセル扱いにする
    if (m_EventHolderExit.m_Event.TryWait() || IsNetworkFeatureDisabled())
    {
        return;
    }

    m_CancelableForBackground.Reset();
    watcher::UploadDeviceEventsExecutor* pContext = new watcher::UploadDeviceEventsExecutor(&m_CancelableForBackground);
    pContext->SetParameters(deviceId);
    pContext->SetBackgroundMode(true);

    SetNextAsyncContext(pContext);
}

void NetworkManager::RequestUpdateDeviceBackground() NN_NOEXCEPT
{
    // 無効化中・未連携時は無視してキャンセル扱いにする
    // (無効化中はデバイス情報のデータは更新されず、未連携時はPostできない)
    if (IsNetworkFeatureDisabled() || !IsPairingActive())
    {
        return;
    }

    // 未了状態としてフラグを落とす
    // (このメソッドが呼ばれているのは更新を行いたいからであるので、
    // 下記の終了処理中であった場合に更新できなくなるのを防ぐ)
    SetLastUpdateDeviceSuccess(false);

    // 終了処理中は何もしない(キャンセル扱い)
    if (m_EventHolderExit.m_Event.TryWait())
    {
        return;
    }

    m_CancelableForBackground.Reset();
    watcher::UpdateDeviceExecutor* pContext = new watcher::UpdateDeviceExecutor(&m_CancelableForBackground);
    pContext->SetParameters(m_Config.deviceId, m_Config.etagForSettings);
    pContext->SetBackgroundMode(true);

    SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::RequestUnlinkDevice(common::AsyncContext** ppContext, ServerDeviceId deviceId, bool isAlwaysClearPairingInfo) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();
    watcher::UnlinkDeviceExecutor* pContext = new watcher::UnlinkDeviceExecutor(&m_Cancelable);
    pContext->SetParameters(deviceId, isAlwaysClearPairingInfo);
    auto result = pContext->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pContext;
        return result;
    }

    *ppContext = pContext;
    return SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::RequestSynchronizeSettingsInternal(common::AsyncContext** ppContext,
    ServerDeviceId deviceId, bool forceRetrieve, int backgroundTryCount, bool fromNpns) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    const bool isBackgroundTask = (backgroundTryCount > 0);
    auto pCancelable = (isBackgroundTask ? &m_CancelableForBackground : &m_Cancelable);
    pCancelable->Reset();

    watcher::RetrieveSettingsExecutor* pContext;
    // NPNS由来であり、半起床中である場合は間欠起動処理として扱うようにする
    if (isBackgroundTask && fromNpns && IntermittentOperation::IsInHalfAwake())
    {
        pContext = new IntermittentContext<RetrieveSettingsExecutor, common::Cancelable*>(pCancelable);
    }
    else
    {
        pContext = new RetrieveSettingsExecutor(pCancelable);
    }
    pContext->SetBackgroundMode(isBackgroundTask);
    // バックグラウンド/フォアグラウンドにかかわらずオーバーレイ通知を行う(第4引数)
    pContext->SetParameters(&m_Config.etagForSettings,
        deviceId,
        backgroundTryCount,
        true,
        forceRetrieve);
    if (!isBackgroundTask)
    {
        // フォアグラウンドの場合はバックグラウンド取得待ちを解除する
        if (IsWatcherAvailable())
        {
            g_pWatcher->GetWatcherEventManager().ClearWaitingForRetrieveSettingsBackground();
        }
        // フォアグラウンドでの待機用にシステムイベントを生成する
        auto result = pContext->InitializeSystemEvent();
        if (result.IsFailure())
        {
            delete pContext;
            return result;
        }
    }

    if (ppContext != nullptr)
    {
        // isBackgroundTask == true の場合は返さない(使わないはず)
        *ppContext = isBackgroundTask ? nullptr : pContext;
    }
    return SetNextAsyncContext(pContext);
}

void NetworkManager::RequestApplyAlarmSettingBackground(bool isDisabled) NN_NOEXCEPT
{
    // 無効化中・未連携時は無視してキャンセル扱いにする
    // (無効化中はデバイス情報のデータは更新されず、未連携時はPostできない)
    if (IsNetworkFeatureDisabled() || !IsPairingActive())
    {
        return;
    }

    // 何らかの理由で取得できない場合は何もしない
    nn::time::PosixTime timeNow;
    if (!common::GetNetworkTime(&timeNow))
    {
        return;
    }

    // 終了処理中は何もしない(キャンセル扱い)
    if (m_EventHolderExit.m_Event.TryWait())
    {
        return;
    }

    m_CancelableForBackground.Reset();
    watcher::UpdateDeviceAlarmSettingExecutor* pContext;
    // この処理はNPNS由来で行われる処理のため、半起床中に受信したら
    // 間欠起動処理として扱うようにする
    if (IntermittentOperation::IsInHalfAwake())
    {
        pContext = new IntermittentContext<UpdateDeviceAlarmSettingExecutor, common::Cancelable*>(&m_CancelableForBackground);
    }
    else
    {
        pContext = new UpdateDeviceAlarmSettingExecutor(&m_CancelableForBackground);
    }
    // 無効 == !visible
    pContext->SetParameters(m_Config.deviceId, isDisabled, timeNow);
    pContext->SetBackgroundMode(true);

    SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::RequestUpdateExemptionList(common::AsyncContext** ppContext,
    ServerDeviceId deviceId, nn::ncm::ApplicationId applicationId, bool isExempt) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();

    watcher::RetrieveSettingsExecutor* pContext = new RetrieveSettingsExecutor(&m_Cancelable);
    pContext->SetBackgroundMode(false);

    pContext->SetParameters(
        deviceId,
        true,
        applicationId,
        isExempt);

    // バックグラウンド取得待ちを解除する
    if (IsWatcherAvailable())
    {
        g_pWatcher->GetWatcherEventManager().ClearWaitingForRetrieveSettingsBackground();
    }
    // フォアグラウンドでの待機用にシステムイベントを生成する
    auto result = pContext->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pContext;
        return result;
    }

    if (ppContext != nullptr)
    {
        *ppContext = pContext;
    }
    return SetNextAsyncContext(pContext);
}

nn::Result NetworkManager::TryToDownloadMiiImage(common::AsyncContext** ppContext, UserData& user) NN_NOEXCEPT
{
    // Mii URIがない場合はダウンロード不要
    if (user.miiUri[0] == 0)
    {
        *ppContext = nullptr;
        NN_RESULT_SUCCESS;
    }

    // 無効化中は何もない扱いにする
    if (IsNetworkFeatureDisabled())
    {
        user.miiUri[0] = 0;
        *ppContext = nullptr;
        NN_RESULT_SUCCESS;
    }

    // Mii用の一時ファイル名を取得
    GenerateMiiFileName(m_MiiImageFileNameBuffer, user.accountId, user.miiUri);

    // 既に同一ファイルをダウンロード済みであれば何もしない
    // 別ファイルである場合は古いファイルを消してダウンロードし直す
    if (std::strcmp(m_MiiImageFileName, m_MiiImageFileNameBuffer) == 0)
    {
        *ppContext = nullptr;
        NN_RESULT_SUCCESS;
    }

    if (*m_MiiImageFileName != 0)
    {
        // (エラーは見ない)
        common::FileSystem::Delete(m_MiiImageFileName);
        *m_MiiImageFileName = 0;
    }

    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());

    PostponeBackgroundTask();
    m_Cancelable.Reset();
    watcher::MiiDownloaderExecutor* pDownloader = new watcher::MiiDownloaderExecutor(&m_Cancelable);
    pDownloader->SetParameters(m_MiiImageFileNameBuffer, user.miiUri, m_MiiImageContentType);
    auto result = pDownloader->InitializeSystemEvent();
    if (result.IsFailure())
    {
        delete pDownloader;
        return result;
    }

    *ppContext = pDownloader;
    return SetNextAsyncContext(pDownloader);
}

nn::Result NetworkManager::FinishDownloadMiiImage(common::AsyncContext* pContext, UserData& user) NN_NOEXCEPT
{
    nn::Result result;
    if (!pContext->IsFinished(&result))
    {
        result = nn::pctl::ResultCanceled();
    }
    pContext->CloseContext();

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::pctl::ResultUnexpectedServerError404)
        {
            // 画像が見つからないので再実行時にダウンロードされないように書き換える
            user.miiUri[0] = 0;
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY

    // 同じサイズのデータなので入れ替えるだけで十分
    std::swap(m_MiiImageFileName, m_MiiImageFileNameBuffer);
    NN_RESULT_SUCCESS;
}

nn::Result NetworkManager::GetMiiImage(uint32_t* outActualSize, UserData& user, void* imageBuffer, size_t bufferSize) NN_NOEXCEPT
{
    return GetMiiImageInternal(outActualSize, nullptr, user, imageBuffer, bufferSize, nullptr, 0);
}

nn::Result NetworkManager::GetMiiImageContentType(uint32_t* outActualLength, UserData& user, char* contentTypeStringBuffer, size_t maxLength) NN_NOEXCEPT
{
    return GetMiiImageInternal(nullptr, outActualLength, user, nullptr, 0, contentTypeStringBuffer, maxLength);
}

nn::Result NetworkManager::GetMiiImageInternal(uint32_t* outActualSize, uint32_t* outActualContentTypeLength,
    UserData& user, void* imageBuffer, size_t bufferSize,
    char* contentTypeStringBuffer, size_t contentTypeMaxLength) NN_NOEXCEPT
{
    // Mii URIがない場合は成功扱い
    if (user.miiUri[0] == 0)
    {
        if (outActualSize != nullptr)
        {
            *outActualSize = 0;
        }
        if (outActualContentTypeLength != nullptr)
        {
            *outActualContentTypeLength = 0;
        }
        if (contentTypeMaxLength > 0)
        {
            *contentTypeStringBuffer = 0;
        }
        NN_RESULT_SUCCESS;
    }

    // ファイルにダウンロード済みなのでそこから取得する
    if (outActualSize != nullptr)
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, m_MiiImageFileName);
        if (result.IsFailure())
        {
            NN_DETAIL_PCTL_ERROR("Cannot open temporal mii-image file: result = 0x%08x, file = %s\n",
                result.GetInnerValueForDebug(), m_MiiImageFileName);
            NN_RESULT_THROW(nn::pctl::ResultResponseFormatError());
        }
        {
            int64_t s = 0;
            result = stream.GetFileSize(&s);
            if (result.IsFailure())
            {
                NN_DETAIL_PCTL_ERROR("Cannot get file size for temporal mii-image: result = 0x%08x, file = %s\n",
                    result.GetInnerValueForDebug(), m_MiiImageFileName);
                NN_RESULT_THROW(nn::pctl::ResultResponseFormatError());
            }
            // ダウンロード時点で uint32_t で扱えるサイズかどうかチェックしているため
            // ここでは abort で止める
            NN_ABORT_UNLESS(nn::util::IsIntValueRepresentable<uint32_t>(s));
            *outActualSize = static_cast<uint32_t>(s);
        }
        if (imageBuffer != nullptr && bufferSize != 0)
        {
            size_t read = 0;
            result = stream.Read(&read, 0, imageBuffer, bufferSize);
            if (result.IsFailure())
            {
                NN_DETAIL_PCTL_ERROR("Cannot read temporal mii-image: result = 0x%08x, file = %s\n",
                    result.GetInnerValueForDebug(), m_MiiImageFileName);
                NN_RESULT_THROW(nn::pctl::ResultResponseFormatError());
            }
        }
    }
    // Content-Type は m_MiiImageFileName と同調してバッファーにコピーしているのでそれを返す
    if (outActualContentTypeLength != nullptr)
    {
        *outActualContentTypeLength = static_cast<uint32_t>(nn::util::Strnlen(m_MiiImageContentType, MaxMiiImageContentTypeLength - 1));
    }
    if (contentTypeStringBuffer != nullptr && contentTypeMaxLength != 0)
    {
        if (contentTypeMaxLength > MaxMiiImageContentTypeLength)
        {
            contentTypeMaxLength = MaxMiiImageContentTypeLength;
        }
        nn::util::Strlcpy(contentTypeStringBuffer, m_MiiImageContentType, static_cast<int>(contentTypeMaxLength));
    }
    NN_RESULT_SUCCESS;
}

nn::Result NetworkManager::StartIntermittentOperation() NN_NOEXCEPT
{
    // 無効化中は無視してキャンセル扱いにする
    NN_RESULT_THROW_UNLESS(!m_EventHolderExit.m_Event.TryWait(), nn::pctl::ResultCanceled());
    NN_RESULT_THROW_UNLESS(!IsNetworkFeatureDisabled(), nn::pctl::ResultCanceled());
    // 未連携時は何もしない
    NN_RESULT_THROW_UNLESS(IsPairingActive(), nn::pctl::ResultCanceled());

    // バックグラウンドタスクであるが既存のタスクはキャンセルする
    PostponeBackgroundTask();
    m_CancelableForBackground.Reset();
    m_CancelableForIntermittent.Reset();

    watcher::IntermittentOperation* pContext;
    NN_RESULT_DO(watcher::IntermittentOperation::CreateInstance(&pContext, GetSavedDeviceId(), &m_CancelableForIntermittent));

    // バックグラウンドタスクとして動作させる
    pContext->SetBackgroundMode(true);
    // 共有不要な接続要求を利用する
    pContext->SetSharableConnectionNecessary(false);

    return SetNextAsyncContext(pContext);
}

void NetworkManager::OnEnterSleep() NN_NOEXCEPT
{
    // オーバーレイ通知は無効化する
    // (半起床中に通知が発行された場合はスリープ復帰後に行われるようにする)
    if (!m_IsOverlayDisabled)
    {
        overlay::DisableOverlayNotification();
        m_IsOverlayDisabled = true;
    }
}

void NetworkManager::OnAwakeFromSleep() NN_NOEXCEPT
{
    // スリープ復帰では特に取り直しをしないようにする
    // (コールドブートでは取り直す)
    //// このセッションにおける通知トークン取得状態を未取得状態とする
    //m_IsNotificationTokenRetrieved = false;

    // オーバーレイ通知無効化を戻す
    if (m_IsOverlayDisabled)
    {
        overlay::EnableOverlayNotification();
        m_IsOverlayDisabled = false;
    }

    // オーバーレイ通知の確認を行う
    CheckNotifyNecessaryFlags();
}

void NetworkManager::DisableNetworkFeature() NN_NOEXCEPT
{
    if (m_IsNetworkDisabledOnInit)
    {
        return;
    }

    // スレッドを停止させる(破棄しない)
    m_Cancelable.Cancel();
    m_CancelableForBackground.Cancel();
    m_CancelableForIntermittent.Cancel();
    if (m_IsThreadRunning)
    {
        m_EventHolderExit.m_Event.Signal();
        nn::os::WaitThread(&m_TaskThread);
        m_IsThreadRunning = false;
    }

    // データをクリアする
    std::memset(&m_Config, 0, sizeof(m_Config));
}

bool NetworkManager::IsNetworkFeatureDisabled() const NN_NOEXCEPT
{
    return m_IsNetworkDisabledOnInit || g_pMain->GetSettingsManager().IsAllFeaturesDisabled();
}

bool NetworkManager::GetSettingsUpdatedTime(nn::time::PosixTime* outTime) const NN_NOEXCEPT
{
    if (!IsPairingActive() || m_Config.etagForSettings.etag[0] == 0 || m_Config.etagForSettings.lastUpdatedAt == 0)
    {
        return false;
    }
    // 'lastUpdatedAt' は 1970/01/01 00:00:00 (UTC) からの経過秒数であるため
    // nn::time::PosixTime が指すデータと同じ
    outTime->value = static_cast<int64_t>(m_Config.etagForSettings.lastUpdatedAt);
    return true;
}

void NetworkManager::HandlePairingDeletedStatus() NN_NOEXCEPT
{
    // サーバー上で連携状態が残っていないためローカル分を消す
    // (システムイベントによる通知も実施)
    ClearPairingInfo(true);
    // オーバーレイ通知を発行
    overlay::NotifyUnlink(nn::ovln::format::PctlUnlinkReasonFlag_IndirectUnlink);
}

void NetworkManager::SetNotifyNecessaryFlagForSettingChanged(nn::ovln::format::PctlSettingTypeFlagSet changedTypeFlags) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);
    // どの項目に変更があったかをマージする
    // MEMO: Reset は NetworkManager::CheckNotifyNecessaryFlags で行う
    m_Config.changedTypeFlags |= changedTypeFlags;
    m_Config.notifyNecessaryFlags |= NotifyNecessaryFlags_SettingChanged;
    SaveConfigNoLock();
}

void NetworkManager::SetNotifyNecessaryFlagForUnlinked(nn::ovln::format::PctlUnlinkReasonFlag reasonFlags) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);
    m_Config.unlinkReasonFlags = reasonFlags;
    m_Config.notifyNecessaryFlags |= NotifyNecessaryFlags_Unlinked;
    SaveConfigNoLock();
}

void NetworkManager::SetNotifyNecessaryFlagForAlarmSettingChanged(nn::ovln::format::PctlAlarmSettingFlag flags) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_MutexSaveData);
    m_Config.alarmSettingFlags = flags;
    m_Config.notifyNecessaryFlags |= NotifyNecessaryFlags_AlarmSettingChanged;
    SaveConfigNoLock();
}

void NetworkManager::SetLastUpdateDeviceSuccess(bool isSuccess) NN_NOEXCEPT
{
    // 値に変更が無い場合は何もしない
    if (isSuccess == m_IsLastUpdateDeviceSuccess)
    {
        return;
    }

    NN_DETAIL_PCTL_TRACE("[pctl] SetLastUpdateDeviceSuccess(%s)\n", isSuccess ? "true" : "false");

    m_IsLastUpdateDeviceSuccess = isSuccess;
}

// static
void NetworkManager::ClearUnlinkedFlag() NN_NOEXCEPT
{
    if (IsWatcherAvailable())
    {
        // インスタンスが利用できる場合は保持しているデータの書き換えとイベントのシグナルクリアを行う
        auto& manager = g_pWatcher->GetNetworkManager();
        NN_UTIL_LOCK_GUARD(manager.m_MutexSaveData);
        if ((manager.m_Config.statusFlags & StatusFlags_NeedsNotifyUnlinked) == 0)
        {
            return;
        }
        manager.m_Config.statusFlags &= ~StatusFlags_NeedsNotifyUnlinked;
        manager.SaveConfigNoLock();
        manager.m_SysEventUnlinked.Clear();
    }
    else
    {
        // インスタンスが利用できない場合はセーブデータの書き換えの身を行う
        Config config;
        auto result = LoadConfig(config);
        NN_ABORT_UNLESS(result.IsSuccess(), "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        if ((config.statusFlags & StatusFlags_NeedsNotifyUnlinked) == 0)
        {
            return;
        }
        config.statusFlags &= ~StatusFlags_NeedsNotifyUnlinked;
        NN_ABORT_UNLESS_RESULT_SUCCESS(SaveConfig(config));
    }
}

nn::Result NetworkManager::WaitForBackgroundTaskForTest() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        {
            NN_UTIL_LOCK_GUARD(m_MutexBackgroundTask);
            if (m_pRunningBackgroundContext == nullptr)
            {
                break;
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    return m_LastBackgroundTaskResult;
}


void NetworkManager::LoadConfigNoLock() NN_NOEXCEPT
{
    bool read = false;
    auto result = LoadConfig(&read, m_Config);
    NN_ABORT_UNLESS(result.IsSuccess(), "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
    NN_UNUSED(read);
}

void NetworkManager::LoadDeviceStatusNoLock() NN_NOEXCEPT
{
    bool read = false;
    auto result = LoadDeviceStatus(&read, m_DeviceStatus);
    NN_ABORT_UNLESS(result.IsSuccess(), "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
    NN_UNUSED(read);
}

void NetworkManager::SaveConfigNoLock() NN_NOEXCEPT
{
    // 無効化中は保存しない
    if (IsNetworkFeatureDisabled())
    {
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(SaveConfig(m_Config));
}

// static
nn::Result NetworkManager::LoadDeviceStatus(bool* pOutRead, DeviceStatus& outDeviceStatus) NN_NOEXCEPT
{
    size_t size = 0;
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, NetworkDeviceStatusFileName);
        NN_RESULT_THROW_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            result);
        if (result.IsSuccess())
        {
            NN_RESULT_DO(stream.Read(&size, 0, &outDeviceStatus, sizeof(DeviceStatus)));
        }
    }
    if (size != sizeof(DeviceStatus))
    {
        std::memset(&outDeviceStatus, 0, sizeof(DeviceStatus));
        if (pOutRead != nullptr)
        {
            *pOutRead = false;
        }
    }
    else
    {
        if (pOutRead != nullptr)
        {
            *pOutRead = true;
        }
    }
    NN_RESULT_SUCCESS;
}

// static
nn::Result NetworkManager::SaveDeviceStatus(const DeviceStatus& deviceStatus) NN_NOEXCEPT
{
    {
        common::FileStream stream;
        NN_RESULT_DO(common::FileSystem::OpenWrite(&stream, NetworkDeviceStatusFileName, sizeof(DeviceStatus)));
        NN_RESULT_DO(stream.Write(0, &deviceStatus, sizeof(DeviceStatus)));
        NN_RESULT_DO(stream.Flush());
    }
    NN_RESULT_DO(common::FileSystem::Commit());

    NN_RESULT_SUCCESS;
}

void NetworkManager::ClearTokensIfEnvironmentChanged() NN_NOEXCEPT
{
    // 環境が一致している場合は何もしない
    if (nn::util::Strncmp(m_Config.environment.value, GetCurrentEnvironment(),
        std::extent<decltype(m_Config.environment.value)>::value) == 0)
    {
        return;
    }

    NN_DETAIL_PCTL_TRACE("[pctl] Environment changed: %s -> %s\n",
        m_Config.environment.value, GetCurrentEnvironment());

    m_Config.token.authToken[0] = 0;
    // MEMO: NotificationTokenを永続化している場合はここを修正
    //m_IsNotificationTokenRetrieved = false;
    //m_NotificationToken.data[0] = 0;
    nn::util::Strlcpy(m_Config.environment.value, GetCurrentEnvironment(),
        std::extent<decltype(m_Config.environment.value)>::value);
    // 一旦保存しておく
    {
        NN_UTIL_LOCK_GUARD(m_MutexSaveData);
        SaveConfigNoLock();
    }
}

nn::Result NetworkManager::Connect(nn::nifm::NetworkConnection& connection, common::AsyncContext* pContext) NN_NOEXCEPT
{
    // システムプロセス用に接続を確立
    auto result = nn::nifm::SetRequestRequirementPreset(connection.GetRequestHandle(),
        pContext->IsSharableConnectionNecessary() ?
            nn::nifm::RequirementPreset::RequirementPreset_InternetForSystemProcessSharable : // 共存可能なプリセット(通常時、アプリが通信中でも通信できるようにする)
            nn::nifm::RequirementPreset::RequirementPreset_InternetForSystemProcess // 共存不可のプリセット(半起床中の通信など共存させない場合に使用)
        );
    NN_SDK_ASSERT(result.IsSuccess(),
        "[pctl] nn::nifm::SetRequestRequirementPreset failed: 0x%08x", result.GetInnerValueForDebug());
    NN_UNUSED(result);
    NN_DETAIL_PCTL_TRACE("[pctl] NetworkConnection::SubmitRequest()\n");
    connection.SubmitRequest();

    while (NN_STATIC_CONDITION(true))
    {
        if (connection.GetSystemEvent().TryWait())
        {
            // イベントがシグナルされている場合は IsRequestOnHold() == false のはず
            NN_SDK_ASSERT(!connection.IsRequestOnHold(),
                "Event for NetworkConnection was triggered but request is on hold");
            // IsAvailable() == true → 使用可能(受理された)
            if (connection.IsAvailable())
            {
                break;
            }
            // 利用要求が途切れて IsAvailable() == false である場合は拒否されているのでエラーを返す
            NN_DETAIL_PCTL_TRACE("[pctl] Network connection is not available (result = 0x%08lX)\n",
                connection.GetResult().GetInnerValueForDebug());
            NN_RESULT_THROW(nn::pctl::ResultConnectionNotAccepted());
        }

        // 終了イベントがトリガーされていたら Cancel 終了
        if (m_EventHolderExit.m_Event.TryWait())
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Network connection canceled (Exit triggered)\n");
            NN_RESULT_THROW(nn::pctl::ResultCanceled());
        }

        if (pContext->IsFinished())
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Task is finished before connection established\n");
            NN_RESULT_SUCCESS;
        }

        if (pContext->IsCancelTriggered())
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Network connection canceled (Canceled by caller)\n");
            NN_RESULT_THROW(nn::pctl::ResultCanceled());
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    NN_DETAIL_PCTL_TRACE("[pctl] Network connection established\n");
    NN_RESULT_SUCCESS;
}

void NetworkManager::ExecuteContext(common::AsyncContext* pContext) NN_NOEXCEPT
{
    ExecuteContext(pContext, *m_pBuffer);
}

void NetworkManager::ExecuteContext(common::AsyncContext* pContext, common::NetworkBuffer& buffer) NN_NOEXCEPT
{
    // 無効化中は無視してキャンセル扱いにする
    if (IsNetworkFeatureDisabled())
    {
        pContext->Cancel();
        pContext->SetFinished(nn::pctl::ResultCanceled());
        return;
    }

    nn::nifm::NetworkConnection connection(nn::os::EventClearMode::EventClearMode_AutoClear);
    auto result = Connect(connection, pContext);
    if (result.IsFailure())
    {
        NN_DETAIL_PCTL_TRACE("[pctl] Connection failed (result = 0x%08lX)\n", result.GetInnerValueForDebug());
        pContext->Cancel();
        pContext->SetFinished(result);
        return;
    }

    if (!pContext->IsFinished())
    {
        result = pContext->Execute(buffer);
        if (nn::pctl::ResultServerInvalidToken::Includes(result))
        {
            NN_DETAIL_PCTL_TRACE("[pctl] Token expired (expiresAt = %lld)\n", m_Config.token.tokenExpiresAt);
#if !defined(NN_SDK_BUILD_RELEASE)
            {
                nn::time::SteadyClockTimePoint pointNow;
                result = nn::time::StandardSteadyClock::GetCurrentTimePoint(&pointNow);
                if (result.IsSuccess())
                {
                    NN_DETAIL_PCTL_TRACE("[pctl]   retrieved = %lld, now = %lld\n", m_Config.token.tokenRetrieved.value, pointNow.value);
                }
            }
#endif
            // token を無効化して(expire させて)もう一度実行する
            {
                NN_UTIL_LOCK_GUARD(m_MutexToken);
                m_Config.token.tokenExpiresAt = 0;
            }
            result = pContext->Execute(buffer);
        }
        pContext->SetFinished(result);
    }
}

nn::Result NetworkManager::SetNextAsyncContext(common::AsyncContext* pContext) NN_NOEXCEPT
{
    // 無効化中は実行しない(ただし本来は呼び出し元で止めるはず)
    if (IsNetworkFeatureDisabled())
    {
        NN_DETAIL_PCTL_WARN("SetNextAsyncContext is called with disabled status\n");
        pContext->CloseContext();
        NN_RESULT_THROW(nn::pctl::ResultCanceled());
    }

    if (pContext->IsBackgroundMode())
    {
        // バックグラウンドタスクの場合は、次に実行したいタスクが無ければ
        // 通常の処理として、既に存在する場合はバックグラウンド処理として待機させる
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        // (Signalされていた場合は既に他のAsyncContextが存在する)
        if (!m_EventHolderMessage.m_Event.TryWait())
        {
            m_pNextAsyncContext = pContext;
            m_EventHolderMessage.m_Event.Signal();
        }
        else
        {
            // 既にバックグラウンド処理がセットされていたらそれは完全に破棄する(実行しない)
            if (m_pNextBackgroundAsyncContext != nullptr)
            {
                m_pNextBackgroundAsyncContext->CloseContext();
            }
            m_pNextBackgroundAsyncContext = pContext;
        }
    }
    else
    {
        while (NN_STATIC_CONDITION(true))
        {
            {
                NN_UTIL_LOCK_GUARD(m_MutexMessage);
                if (m_EventHolderExit.m_Event.TryWait())
                {
                    pContext->CloseContext();
                    NN_RESULT_THROW(nn::pctl::ResultCanceled());
                }
                // (Signalされていた場合は既に他のAsyncContextが存在する)
                if (!m_EventHolderMessage.m_Event.TryWait())
                {
                    m_pNextAsyncContext = pContext;
                    m_EventHolderMessage.m_Event.Signal();
                    break;
                }
            }
            // 解放されるまで待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
    }
    NN_RESULT_SUCCESS;
}

void NetworkManager::HandleFinishedBackgroundTask(common::AsyncContext* pContext) NN_NOEXCEPT
{
    nn::Result r;
    pContext->IsFinished(&r);
    if (r.IsFailure())
    {
        if (nn::pctl::ResultCanceled::Includes(r) ||
            nn::nifm::ResultCanceled::Includes(r))
        {
            NN_DETAIL_PCTL_INFO("[pctl] Background task canceled.\n");
        }
        else if (nn::nifm::ResultRequestRejected::Includes(r))
        {
            NN_DETAIL_PCTL_INFO("[pctl] Background task has been rejected for connection (result = 0x%08lX).\n",
                r.GetInnerValueForDebug());
        }
        else
        {
            NN_DETAIL_PCTL_WARN("[pctl] Background task failed: 0x%08lX\n",
                r.GetInnerValueForDebug());
        }
    }
    pContext->CloseContext();
}

void NetworkManager::PostponeBackgroundTask() NN_NOEXCEPT
{
    common::AsyncContext* pContext;
    // 実行中の Context を取得し、存在しないかBGではなければ何もしない
    {
        NN_UTIL_LOCK_GUARD(m_MutexBackgroundTask);
        if (m_pRunningBackgroundContext == nullptr)
        {
            return;
        }
        pContext = m_pRunningBackgroundContext;
        if (!pContext->IsBackgroundMode())
        {
            return;
        }
        // BGであればキャンセルを発行
        m_CancelableForBackground.Cancel();
        m_CancelableForIntermittent.Cancel();
        m_IsBackgroundCanceled = true;
    }
    // BGタスクのキャンセル終了待ち
    while (NN_STATIC_CONDITION(true))
    {
        {
            NN_UTIL_LOCK_GUARD(m_MutexBackgroundTask);
            if (m_pRunningBackgroundContext == nullptr)
            {
                break;
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    // 再度実行できるように m_pNextBackgroundAsyncContext に追加
    {
        NN_UTIL_LOCK_GUARD(m_MutexMessage);
        // 既にバックグラウンド処理がセットされていたらそれは完全に破棄する(実行しない)
        if (m_pNextBackgroundAsyncContext != nullptr)
        {
            m_pNextBackgroundAsyncContext->CloseContext();
        }
        m_pNextBackgroundAsyncContext = pContext;
    }
}

void NetworkManager::CheckNotifyNecessaryFlags() NN_NOEXCEPT
{
    NN_STATIC_ASSERT(NotifyNecessaryFlags_None == 0);
    if (m_Config.notifyNecessaryFlags == NotifyNecessaryFlags_None)
    {
        return;
    }
    uint8_t notifyNecessaryFlags;
    nn::ovln::format::PctlSettingTypeFlagSet changedTypeFlags;
    nn::ovln::format::PctlUnlinkReasonFlag unlinkReasonFlags;
    nn::ovln::format::PctlAlarmSettingFlag alarmSettingFlags;
    {
        NN_UTIL_LOCK_GUARD(m_MutexSaveData);
        NN_DETAIL_PCTL_TRACE("[pctl] ** Notify on wake; notifyNecessaryFlags = %d\n", static_cast<int>(m_Config.notifyNecessaryFlags));
        notifyNecessaryFlags = m_Config.notifyNecessaryFlags;
        changedTypeFlags = m_Config.changedTypeFlags;
        unlinkReasonFlags = m_Config.unlinkReasonFlags;
        alarmSettingFlags = m_Config.alarmSettingFlags;
        m_Config.notifyNecessaryFlags = NotifyNecessaryFlags_None;
        m_Config.changedTypeFlags.Reset();
        m_Config.unlinkReasonFlags = static_cast<nn::ovln::format::PctlUnlinkReasonFlag>(0);
        m_Config.alarmSettingFlags = static_cast<nn::ovln::format::PctlAlarmSettingFlag>(0);
        // 一旦保存しておく
        SaveConfigNoLock();
    }

    if ((notifyNecessaryFlags & NotifyNecessaryFlags_SettingChanged) != 0)
    {
        overlay::NotifySettingChanged(changedTypeFlags);
    }
    if ((notifyNecessaryFlags & NotifyNecessaryFlags_Unlinked) != 0)
    {
        overlay::NotifyUnlink(unlinkReasonFlags);
    }
    if ((notifyNecessaryFlags & NotifyNecessaryFlags_AlarmSettingChanged) != 0)
    {
        overlay::NotifyAlarmSettingChanged(alarmSettingFlags);
    }
}

void NetworkManager::ThreadProc() NN_NOEXCEPT
{
    NN_DETAIL_PCTL_TRACE("[pctl] Start network thread...\n");
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::ListenTo(ApplicationIdPctl));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::GetReceiveEvent(m_EventHolderNpns.m_Event));
        m_EventHolderNpns.Initialize();
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_EventHolderNpns.m_WaitHolder);
    }
#endif
    NN_DETAIL_PCTL_TRACE("[pctl] Start network thread done.\n");

    // 連携済みであればスレッド開始時にデバイス状態の更新があるかチェックする
    // (タイムゾーン設定変更はここでは見ていないが、WatcherEventManager で確認するのでそちらに任せる)
    if (IsPairingActive())
    {
        if (RefreshDeviceStatus())
        {
            RequestUpdateDeviceBackground();
        }
    }

    // 間欠起動の処理などによって通知を送るべき状態になっていたら送信を行う
    CheckNotifyNecessaryFlags();

    while (NN_STATIC_CONDITION(true))
    {
        auto holder = nn::os::WaitAny(&m_MultiWait);
        NN_SDK_ASSERT_NOT_NULL(holder);

        common::AsyncContext* pContext = nullptr;
        if (holder == &m_EventHolderExit.m_WaitHolder)
        {
            // Clear しない
            break;
        }
        else if (holder == &m_EventHolderMessage.m_WaitHolder)
        {
            pContext = m_pNextAsyncContext;
            bool isBackground = (pContext == nullptr ? false : pContext->IsBackgroundMode());
            if (isBackground)
            {
                NN_UTIL_LOCK_GUARD(m_MutexBackgroundTask);
                m_pRunningBackgroundContext = pContext;
            }
            // nullptr をセットしてから Clear する
            m_pNextAsyncContext = nullptr;
            m_EventHolderMessage.m_Event.Clear();

            if (pContext != nullptr)
            {
                ExecuteContext(pContext);

                {
                    NN_UTIL_LOCK_GUARD(m_MutexMessage);
                    if (m_pNextAsyncContext == nullptr && m_pNextBackgroundAsyncContext != nullptr)
                    {
                        nn::Result result;
                        if (m_pNextBackgroundAsyncContext->IsFinished(&result))
                        {
                            // ResultCanceled 以外の場合は通常終了しているので
                            // Context の破棄処理を行う
                            if (!nn::pctl::ResultCanceled::Includes(result))
                            {
                                HandleFinishedBackgroundTask(m_pNextBackgroundAsyncContext);
                                m_pNextBackgroundAsyncContext = nullptr;
                            }
                        }
                        if (m_pNextBackgroundAsyncContext != nullptr)
                        {
                            m_pNextBackgroundAsyncContext->ResetFinished();
                            m_pNextAsyncContext = m_pNextBackgroundAsyncContext;
                            m_pNextBackgroundAsyncContext = nullptr;
                            m_EventHolderMessage.m_Event.Signal();
                        }
                    }
                }

                if (isBackground)
                {
                    {
                        NN_UTIL_LOCK_GUARD(m_MutexBackgroundTask);
                        // このタイミングでResultを取得しておく
                        pContext->IsFinished(&m_LastBackgroundTaskResult);
                        m_pRunningBackgroundContext = nullptr;
                        if (m_IsBackgroundCanceled)
                        {
                            // ここに来る場合は CancelBackgroundTask によってキャンセル処理が行われた場合。
                            // 仮に pContext->IsFinished() == true であった場合は上記の
                            // m_pNextBackgroundAsyncContext のチェックで判定するので
                            // このタイミングでの特殊処理は不要
                            m_IsBackgroundCanceled = false;
                            // Finished させない
                            pContext = nullptr;
                        }
                    }
                    if (pContext != nullptr)
                    {
                        HandleFinishedBackgroundTask(pContext);
                    }
                }
            }
        }
        else if (holder == &m_EventHolderNpns.m_WaitHolder)
        {
            m_EventHolderNpns.m_Event.Clear();
            // 終了イベントが発行されていなければ処理を行う
            if (!m_EventHolderExit.m_Event.TryWait())
            {
                m_CancelableForBackground.Reset();
                m_NotificationDataReceiver.ProcessReceiveNotificationData(*m_pBuffer, m_CancelableForBackground);
            }
        }
    }

    NN_DETAIL_PCTL_TRACE("[pctl] End network thread.\n");
}

}}}}}
