﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/pctl_DeviceUser.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_FileSystem.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>
#include <nn/pctl/detail/service/watcher/dispatcher/pctl_DeviceUserDispatcher.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_Result.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher {

namespace
{
    static const char DeviceUserDataFileName[] = "du.dat";

    DeviceUserData g_DeviceUserData;
    nn::os::SdkMutexType g_DeviceUserDataMutex = NN_OS_SDK_MUTEX_INITIALIZER();
}

////////////////////////////////////////////////////////////////////////////////

// メモ:
// - Userの状態確認は以下の3パターン
//   1. Userが追加される
//   2. Userが削除される
//   3. Userのdigest値が変わる
//   (4. Userが変わる: 2と1が行われたパターン)
// -

nn::Result UpdateDeviceUserExecutor::UpdateDeviceUsers(common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable,
    const char* token, ServerDeviceId deviceId) NN_NOEXCEPT
{
    // VS2013までは NN_FUNCTION_LOCAL_STATIC の実装が特殊であり配列を直接宣言できないので
    // 配列型を typedef して対応
    typedef nn::account::Uid UidArray[nn::account::UserCountMax];
    typedef nn::account::ProfileDigest ProfileDigestArray[nn::account::UserCountMax];
    NN_STATIC_ASSERT(std::extent<decltype(g_DeviceUserData.users)>::value == std::extent<UidArray>::value); // decltype(allUsers) を用いるとVS2013では extent が効かない
    NN_FUNCTION_LOCAL_STATIC(UidArray, allUsers);
    NN_FUNCTION_LOCAL_STATIC(ProfileDigestArray, allUserDigests);

    NN_UTIL_LOCK_GUARD(g_DeviceUserDataMutex);
    int count = 0;
    bool dataChanged = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (dataChanged)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] PostSaveDeviceUserData\n");
            g_pWatcher->GetWatcherEventManager().PostSaveDeviceUserData();
        }
        NN_DETAIL_PCTL_TRACE("[pctl] List users done.\n");
    };

    // 情報取得
    NN_RESULT_DO(
        nn::account::ListAllUsers(&count, allUsers, nn::account::UserCountMax)
    );
    NN_DETAIL_PCTL_TRACE("[pctl] List users:\n");
    for (int i = 0; i < count; ++i)
    {
        NN_RESULT_TRY(nn::account::GetProfileDigest(&allUserDigests[i], allUsers[i]))
            NN_RESULT_CATCH(nn::account::ResultUserNotExist)
            {
                // 見つからなくなっているので InvalidUid をセットしておく
                allUsers[i] = nn::account::InvalidUid;
            }
        NN_RESULT_END_TRY
        if (allUsers[i] != nn::account::InvalidUid)
        {
            NN_DETAIL_PCTL_TRACE("[pctl]   %d: %08x_%08x_%08x_%08x, %016llx%016llx\n", i,
                static_cast<uint32_t>(allUsers[i]._data[0] >> 32),
                static_cast<uint32_t>(allUsers[i]._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(allUsers[i]._data[1] >> 32),
                static_cast<uint32_t>(allUsers[i]._data[1] & 0xFFFFFFFFull),
                reinterpret_cast<uint64_t*>(allUserDigests[i].data)[0],
                reinterpret_cast<uint64_t*>(allUserDigests[i].data)[1]);
        }
    }

    // 存在しなくなったユーザーのチェック
    for (auto& u : g_DeviceUserData.users)
    {
        if (u.uid == nn::account::InvalidUid)
        {
            continue;
        }
        bool found = false;
        for (int i = 0; i < count; ++i)
        {
            // (u.uid != nn::account::InvalidUid なので
            //  allUsers[i] == nn::account::InvalidUid のチェックは不要)
            if (allUsers[i] == u.uid)
            {
                found = true;
                break;
            }
        }
        if (!found)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] User %08x_%08x_%08x_%08x is deleted\n",
                static_cast<uint32_t>(u.uid._data[0] >> 32),
                static_cast<uint32_t>(u.uid._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(u.uid._data[1] >> 32),
                static_cast<uint32_t>(u.uid._data[1] & 0xFFFFFFFFull));
            NN_RESULT_DO(
                dispatcher::DeviceUserDispatcher::SendDeleteUser(bufferInfo, pCancelable, token, deviceId, u.uid)
            );
            u.uid = nn::account::InvalidUid;
            dataChanged = true;
        }
    }

    // 追加された/変更されたユーザーのチェック
    for (int i = 0; i < count; ++i)
    {
        if (allUsers[i] == nn::account::InvalidUid)
        {
            continue;
        }
        bool found = false;
        for (auto& u : g_DeviceUserData.users)
        {
            if (allUsers[i] == u.uid)
            {
                found = true;
                // 既にリストにあるユーザーである場合は更新されたかどうかチェック
                if (allUserDigests[i] != u.digest)
                {
                    NN_DETAIL_PCTL_TRACE("[pctl] Update user %08x_%08x_%08x_%08x (index = %d, digest = %016llx%016llx -> %016llx%016llx)\n",
                        static_cast<uint32_t>(u.uid._data[0] >> 32),
                        static_cast<uint32_t>(u.uid._data[0] & 0xFFFFFFFFull),
                        static_cast<uint32_t>(u.uid._data[1] >> 32),
                        static_cast<uint32_t>(u.uid._data[1] & 0xFFFFFFFFull),
                        i,
                        reinterpret_cast<uint64_t*>(allUserDigests[i].data)[0],
                        reinterpret_cast<uint64_t*>(allUserDigests[i].data)[1],
                        reinterpret_cast<uint64_t*>(u.digest.data)[0],
                        reinterpret_cast<uint64_t*>(u.digest.data)[1]);
                    NN_RESULT_DO(
                        dispatcher::DeviceUserDispatcher::SendPutUser(bufferInfo, pCancelable, token, deviceId, allUsers[i])
                    );
                    u.digest = allUserDigests[i];
                    dataChanged = true;
                }
                break;
            }
        }
        if (!found)
        {
            // 見つからない == 追加された
            for (auto& u : g_DeviceUserData.users)
            {
                // 空いている箇所に挿入する
                // (※ 順番を nn::account が返すリストと揃える必要は無い)
                if (u.uid == nn::account::InvalidUid)
                {
                    found = true;
                    NN_DETAIL_PCTL_TRACE("[pctl] Add user %08x_%08x_%08x_%08x to index %d (digest = %016llx%016llx)\n",
                        static_cast<uint32_t>(allUsers[i]._data[0] >> 32),
                        static_cast<uint32_t>(allUsers[i]._data[0] & 0xFFFFFFFFull),
                        static_cast<uint32_t>(allUsers[i]._data[1] >> 32),
                        static_cast<uint32_t>(allUsers[i]._data[1] & 0xFFFFFFFFull),
                        i,
                        reinterpret_cast<uint64_t*>(allUserDigests[i].data)[0],
                        reinterpret_cast<uint64_t*>(allUserDigests[i].data)[1]);
                    NN_RESULT_DO(
                        dispatcher::DeviceUserDispatcher::SendPutUser(bufferInfo, pCancelable, token, deviceId, allUsers[i])
                    );
                    u.uid = allUsers[i];
                    u.digest = allUserDigests[i];
                    dataChanged = true;
                    break;
                }
            }
            // (リストのサイズが同じ＆削除済みユーザー分は事前に消しているので必ず空きがあるはず)
            NN_SDK_ASSERT(found, "New user added but list is full");
        }
    }

    NN_RESULT_SUCCESS;
    // (※ 関数の長さが多いのはTRACEやASSERTが多いため)
} // NOLINT(impl/function_size)

void UpdateDeviceUserExecutor::InitializeAndLoadDeviceUserData() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_DeviceUserDataMutex);
    size_t size = 0;
    {
        common::FileStream stream;
        auto result = common::FileSystem::OpenRead(&stream, DeviceUserDataFileName);
        NN_ABORT_UNLESS(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result),
            "Unexpected result for file open: %08x", result.GetInnerValueForDebug());
        if (result.IsSuccess())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Read(&size, 0, &g_DeviceUserData, sizeof(g_DeviceUserData)));
        }
    }
    // 読み込めなかった場合は初期化しておくことで、全ユーザーの情報をPostできるようになる
    if (size != sizeof(g_DeviceUserData))
    {
        //std::memset(&g_DeviceUserData, 0, sizeof(g_DeviceUserData));
        for (auto& u : g_DeviceUserData.users)
        {
            u.uid = nn::account::InvalidUid;
        }
    }
}

void UpdateDeviceUserExecutor::SaveDeviceUserData() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_DeviceUserDataMutex);
    {
        common::FileStream stream;
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::OpenWrite(&stream, DeviceUserDataFileName, sizeof(g_DeviceUserData)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Write(0, &g_DeviceUserData, sizeof(g_DeviceUserData)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(stream.Flush());
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
}

void UpdateDeviceUserExecutor::ClearDeviceUserData() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(g_DeviceUserDataMutex);
    //std::memset(&g_DeviceUserData, 0, sizeof(g_DeviceUserData));
    for (auto& u : g_DeviceUserData.users)
    {
        u.uid = nn::account::InvalidUid;
    }
    auto resultDelete = common::FileSystem::Delete(DeviceUserDataFileName);
    if (resultDelete.IsSuccess())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(common::FileSystem::Commit());
    }
    else if (!nn::fs::ResultPathNotFound::Includes(resultDelete))
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(resultDelete);
    }
}

}}}}}
