﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/watcher/dispatcher/pctl_DeviceEventDispatcher.h>

#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/service/pctl_PlayState.h>
#include <nn/pctl/detail/service/pctl_ServiceConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMemoryManagement.h>
#include <nn/pctl/detail/service/pctl_ServiceWatcher.h>
#include <nn/pctl/detail/service/common/pctl_HttpRequest.h>
#include <nn/pctl/detail/service/json/pctl_JsonWebApi.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonHttpInputStream.h>
#include <nn/pctl/detail/service/json/pctl_JsonStructuredWriter.h>
#include <nn/pctl/detail/service/watcher/pctl_WatcherErrorHandler.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace watcher { namespace dispatcher {

namespace
{
    ////////////////////////////////////////////////////////////////////////////
    // イベント送信処理

    struct UploadDeviceEventParam
    {
        const EventData* dataArray;
        const json::WriteDataDefinition* arrayPayloadDefinitions;
        PlayState* pTempPlayState; // イベントをもとに計算した結果のデータを保持
        PlayStatePlayedApplicationEnumerator appEnumerator;
        int appDataIndex;
        PlayStatePlayedUserEnumerator userEnumerator;
        int playerDataIndex;
        nn::TimeSpan appElapsedTimeCurrent; // appEnumerator.GetCurrent() で得られるデータから求められる elapsedTime
        int payloadDefinitionsLength;
        int playStateCurrentIndex;
        union
        {
            char applicationIdBuffer[17]; // applicationTimes と payload で使用(競合しないので使いまわす)
            char uidBuffer[36];           // TODO: ユーザー情報取得の機能にある定数を用いる
        };
        uint16_t totalCount;
        uint16_t currentIndex;
        bool isPlayStateUpdated;
    };

    // payload object

    static bool WritePayloadDataApplicationId(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        *outValueLength = 16;
        *outValue = p->applicationIdBuffer;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForPayloadWithApplicationId)
        NN_DETAIL_PCTL_JSON_WRITE_KEY("applicationId")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WritePayloadDataApplicationId)
    NN_DETAIL_PCTL_JSON_END_WRITE()

    // devicePlayer object

    static bool WriteDevicePlayersDataElapsedTime(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& eventData = p->dataArray[p->currentIndex];
        auto& userData = *p->userEnumerator.GetCurrent();
        // イベントの時刻時点での値を返す
        auto time = p->pTempPlayState->GetUserPlayingTime(userData, eventData.timestamp);
        if (time > p->appElapsedTimeCurrent)
        {
            NN_DETAIL_PCTL_INFO("[pctl] devicePlayer.elapsedTime is greater than elapsedTime (%lld > %lld [appId = 0x%016llX, uid = %s])\n",
                time.GetSeconds(), p->appElapsedTimeCurrent.GetSeconds(), p->appEnumerator.GetCurrent()->applicationId.value,
                p->uidBuffer);
            time = p->appElapsedTimeCurrent;
        }
        auto value = time.GetSeconds();
        // elapsedTime を超える場合は調整(計算誤差や地域名変更の過渡期により発生する可能性あり)
        if (value > p->pTempPlayState->lastElapsedTimeSeconds)
        {
            value = p->pTempPlayState->lastElapsedTimeSeconds;
        }
        *outValue = value;
        return true;
    };

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForDevicePlayerObject)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("elapsedTime")    NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteDevicePlayersDataElapsedTime)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    // applicationTime object

    static bool WriteAppTimesDataElapsedTime(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto value = p->appElapsedTimeCurrent.GetSeconds();
        // elapsedTime を超える場合は調整(計算誤差や地域名変更の過渡期により発生する可能性あり)
        if (value > p->pTempPlayState->lastElapsedTimeSeconds)
        {
            value = p->pTempPlayState->lastElapsedTimeSeconds;
        }
        *outValue = value;
        return true;
    };

    static bool WriteAppTimesDataAnonymousPlayTime(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& eventData = p->dataArray[p->currentIndex];
        auto& appData = *p->appEnumerator.GetCurrent();
        // イベントの時刻時点での値を返す
        auto time = p->pTempPlayState->GetApplicationAnonymousPlayingTime(appData, eventData.timestamp);
        if (time > p->appElapsedTimeCurrent)
        {
            NN_DETAIL_PCTL_INFO("[pctl] anonymousTime is greater than elapsedTime (%lld > %lld [appId = 0x%016llX])\n",
                time.GetSeconds(), p->appElapsedTimeCurrent.GetSeconds(), p->appEnumerator.GetCurrent()->applicationId.value);
            time = p->appElapsedTimeCurrent;
        }
        auto value = time.GetSeconds();
        // elapsedTime を超える場合は調整(計算誤差や地域名変更の過渡期により発生する可能性あり)
        if (value > p->pTempPlayState->lastElapsedTimeSeconds)
        {
            value = p->pTempPlayState->lastElapsedTimeSeconds;
        }
        *outValue = value;
        return true;
    };

    static bool WriteAppTimesDataDevicePlayersObject(nn::util::optional<bool>* outHasItem, const char** outKeyName, int* outDefinitionLength, const json::WriteDataDefinition** outDefinition, void* param, int, int itemIndex) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& appData = *p->appEnumerator.GetCurrent();

        if (itemIndex == 0)
        {
            p->userEnumerator = p->pTempPlayState->GetPlayedUserEnumerator(appData.applicationId);
            p->playerDataIndex = 0;
        }
        // 出力先のサイズ不足で2回呼び出される可能性があるのでチェック
        // (itemIndex は単調増加という条件が設けられていないため)
        else if (itemIndex != p->playerDataIndex)
        {
            if (itemIndex < p->playerDataIndex)
            {
                // (実際は巻き戻る方向にはならないはず)
                NN_DETAIL_PCTL_WARN("[pctl] WriteAppTimesDataDevicePlayersObject: itemIndex is smaller than expected.\n");
                p->userEnumerator.MoveToFirst();
                p->playerDataIndex = 0;
            }
            while (p->userEnumerator.HasCurrent() && p->playerDataIndex < itemIndex)
            {
                ++p->playerDataIndex;
                p->userEnumerator.Next();
            }
        }
        bool hasItem = p->userEnumerator.HasCurrent();
        if (hasItem)
        {
            auto& targetItem = *p->userEnumerator.GetCurrent();
            // TODO: UidToString で置き換え
            nn::util::SNPrintf(p->uidBuffer, std::extent<decltype(p->uidBuffer)>::value,
                "%08x_%08x_%08x_%08x",
                static_cast<uint32_t>(targetItem.uid._data[0] >> 32),
                static_cast<uint32_t>(targetItem.uid._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(targetItem.uid._data[1] >> 32),
                static_cast<uint32_t>(targetItem.uid._data[1] & 0xFFFFFFFFull));
            *outKeyName = p->uidBuffer;
        }
        *outHasItem = hasItem;
        *outDefinitionLength = static_cast<int>(std::extent<decltype(WriteJsonFormatForDevicePlayerObject)>::value);
        *outDefinition = WriteJsonFormatForDevicePlayerObject;
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForApplicationTimeObject)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("elapsedTime")        NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteAppTimesDataElapsedTime)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("anonymousPlayTime")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteAppTimesDataAnonymousPlayTime)
            NN_DETAIL_PCTL_JSON_WRITE_KEY("devicePlayers")      NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT(WriteAppTimesDataDevicePlayersObject)
        NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()

    // device event object

    static bool WriteEventsArrayBegin(nn::util::optional<bool>* outHasItem, void* param, int, int itemIndex) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        // 最初の呼び出しでは itemIndex は必ず 0 なので各種データをリセットする
        if (itemIndex == 0)
        {
            p->currentIndex = 0;
            auto holder = PlayStateHolder::Acquire();
            NN_STATIC_ASSERT((std::is_same<
                std::remove_pointer<decltype(p->pTempPlayState)>::type,
                std::remove_const<std::remove_reference<decltype(*holder.Get())>::type>::type
            >::value));
            std::memcpy(p->pTempPlayState, holder.Get(), sizeof(*(p->pTempPlayState)));
            p->playStateCurrentIndex = 0;
            p->isPlayStateUpdated = false;
            p->appDataIndex = 0;
            p->playerDataIndex = 0;
        }
        else
        {
            ++p->currentIndex;
        }
        while (NN_STATIC_CONDITION(true))
        {
            if (p->currentIndex == p->totalCount)
            {
                *outHasItem = false;
                return true;
            }
            NN_SDK_ASSERT_LESS(p->currentIndex, p->totalCount);
            auto& eventData = p->dataArray[p->currentIndex];

            // イベントをプレイ状態に反映させ、その状態をデータ出力時に用いることができるようにする
            // (送信のためにデータをすべて参照するこのタイミングで行う)
            if (p->pTempPlayState->UpdatePlayStateFromEvent(eventData))
            {
                p->isPlayStateUpdated = true;
            }

            // 一時解除している場合、その区間のイベントは IgnorableDeviceEventsOnUnlocked に含まれていれば出力しない
            if (p->pTempPlayState->IsTemporaryUnlocked() &&
                IgnorableDeviceEventsOnUnlocked[static_cast<int>(eventData.eventType)])
            {
                // (次の要素を見る)
            }
            else
            {
                // イベントの名前が定義されていればそれを出力対象にし、
                // 定義されていない(空である)場合は次の要素を対象にするかチェックする
                auto& eventTypeName = DeviceEventTypeName[static_cast<int>(eventData.eventType)];
                if (IsEmptyStringValueDefinitionData(eventTypeName))
                {
                    // (次の要素を見る)
                }
                else
                {
                    // 対象にするのでループを抜ける
                    break;
                }
            }

            ++p->currentIndex;
        }
        *outHasItem = true;
        return true;
    }

    static bool WriteEventDataEventType(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        auto& value = DeviceEventTypeName[static_cast<int>(data.eventType)];
        NN_SDK_ASSERT(!IsEmptyStringValueDefinitionData(value));
        *outValueLength = value.length;
        *outValue = value.value;
        return true;
    };

    static bool WriteEventDataEventTimestamp(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        *outValue = data.timestamp.value;
        return true;
    };

    static bool WriteEventDataTimezone(nn::util::optional<size_t>* outValueLength, const char** outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        size_t length = nn::util::Strnlen(data.timezoneName._value, nn::time::LocationName::Size);
        *outValueLength = length;
        *outValue = data.timezoneName._value;
        return true;
    };

    static bool WriteEventDataTimezoneUtcOffsetSeconds(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        *outValue = static_cast<int64_t>(data.timezoneOffsetSeconds);
        return true;
    };

    static bool WriteEventDataElapsedTime(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        *outValue = p->pTempPlayState->lastElapsedTimeSeconds;
        return true;
    }

    static bool WriteEventDataApplicationTimesKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        // プレイ状態計算の対応がされていないデータでは出力しない
        *outKeyName = (data.dataVersion >= EventDataVersionPlayStateSupported ? "applicationTimes" : nullptr);
        return true;
    };

    static bool WriteEventDataApplicationTimesObject(nn::util::optional<bool>* outHasItem, const char** outKeyName, int* outDefinitionLength, const json::WriteDataDefinition** outDefinition, void* param, int, int itemIndex) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
#if !defined(NN_SDK_BUILD_RELEASE)
        // (EventData は以降の処理で使わない)
        {
            auto& data = p->dataArray[p->currentIndex];
            NN_SDK_ASSERT_GREATER_EQUAL(data.dataVersion, EventDataVersionPlayStateSupported);
            NN_UNUSED(data);
        }
#endif

        if (itemIndex == 0)
        {
            p->appEnumerator = p->pTempPlayState->GetPlayedApplicationEnumerator();
            p->appDataIndex = 0;
        }
        // 出力先のサイズ不足で2回呼び出される可能性があるのでチェック
        // (itemIndex は単調増加という条件が設けられていないため)
        else if (itemIndex != p->appDataIndex)
        {
            if (itemIndex < p->appDataIndex)
            {
                // (実際は巻き戻る方向にはならないはず)
                NN_DETAIL_PCTL_WARN("[pctl] WriteEventDataApplicationTimesObject: itemIndex is smaller than expected.\n");
                p->appEnumerator.MoveToFirst();
                p->appDataIndex = 0;
            }
            while (p->appEnumerator.HasCurrent() && p->appDataIndex < itemIndex)
            {
                p->appEnumerator.Next();
                ++p->appDataIndex;
            }
        }
        bool hasItem = p->appEnumerator.HasCurrent();
        if (hasItem)
        {
            auto& targetItem = *p->appEnumerator.GetCurrent();
            // イベントの時刻時点での値をセット
            p->appElapsedTimeCurrent = p->pTempPlayState->GetApplicationRunningTime(
                targetItem, p->dataArray[p->currentIndex].timestamp
            );
            // MEMO: payload オブジェクトでも applicationIdBuffer を使用
            nn::util::SNPrintf(p->applicationIdBuffer, std::extent<decltype(p->applicationIdBuffer)>::value,
                "%016llX", targetItem.applicationId.value);
            *outKeyName = p->applicationIdBuffer;
        }
        *outHasItem = hasItem;
        *outDefinitionLength = static_cast<int>(std::extent<decltype(WriteJsonFormatForApplicationTimeObject)>::value);
        *outDefinition = WriteJsonFormatForApplicationTimeObject;
        return true;
    }

    static bool WriteEventDataMiscTimeKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        // プレイ状態計算の対応がされていないデータでは出力しない
        *outKeyName = (data.dataVersion >= EventDataVersionPlayStateSupported ? "miscTime" : nullptr);
        return true;
    };

    static bool WriteEventDataMiscTime(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        NN_SDK_ASSERT_GREATER_EQUAL(data.dataVersion, EventDataVersionPlayStateSupported);
        // イベント発生時の時刻で計算
        auto value = p->pTempPlayState->GetMiscTime(data.timestamp).GetSeconds();
        // elapsedTime を超える場合は調整(計算誤差や地域名変更の過渡期により発生する可能性あり)
        if (value > p->pTempPlayState->lastElapsedTimeSeconds)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] misc time is too large (value: %lld)\n", value);
            value = p->pTempPlayState->lastElapsedTimeSeconds;
        }
        else if (value < 0)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] misc time is too small (value: %lld)\n", value);
            value = 0;
        }
        *outValue = value;
        return true;
    };

    static bool WriteEventDataDisabledTimeKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        // プレイ状態計算の対応がされていないデータでは出力しない
        *outKeyName = (data.dataVersion >= EventDataVersionPlayStateSupported ? "disabledTime" : nullptr);
        return true;
    };

    static bool WriteEventDataDisabledTime(nn::util::optional<int64_t>* outValue, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        // イベント発生時の時刻で計算
        auto value = p->pTempPlayState->GetTemporaryUnlockedTime(data.timestamp).GetSeconds();
        // 86400 (= 1日 = 24 * 60 * 60)を超える場合は調整(※ elapsedTime より大きくなることはある)
        if (value > 86400)
        {
            value = 86400;
        }
        else if (value < 0)
        {
            NN_DETAIL_PCTL_TRACE("[pctl] disabled time is too small (value: %lld)\n", value);
            value = 0;
        }
        *outValue = value;
        return true;
    }

    static bool WriteEventDataPayloadKey(const char** outKeyName, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        auto& data = p->dataArray[p->currentIndex];
        nn::ncm::ApplicationId applicationId = nn::ncm::ApplicationId::GetInvalidId();
        switch (data.eventType)
        {
            case EventType::DidApplicationLaunch:
                applicationId = data.payload.applicationLaunch.applicationId;
                break;
            case EventType::DidApplicationTerminate:
                applicationId = data.payload.applicationTerminate.applicationId;
                break;
            case EventType::DidAddNewManagedApplication:
                applicationId = data.payload.addManagedApplication.applicationId;
                break;
            case EventType::DidRemoveManagedApplication:
                applicationId = data.payload.removeManagedApplication.applicationId;
                break;
            case EventType::DidApplicationPlay:
                applicationId = data.payload.playedApplication.applicationId;
                break;
            case EventType::DidApplicationDownloadStart:
                applicationId = data.payload.startedApplicationDownload.applicationId;
                break;
            case EventType::DidLimitedApplicationLaunch:
                applicationId = data.payload.limitedApplicationLaunch.applicationId;
                break;
            default:
                *outKeyName = nullptr;
                break;
        }
        if (applicationId != nn::ncm::ApplicationId::GetInvalidId())
        {
            // MEMO: applicationTimes オブジェクトでも applicationIdBuffer を使用する
            nn::util::SNPrintf(p->applicationIdBuffer, std::extent<decltype(p->applicationIdBuffer)>::value,
                "%016llX", applicationId.value);
            p->arrayPayloadDefinitions = WriteJsonFormatForPayloadWithApplicationId;
            p->payloadDefinitionsLength = static_cast<int>(std::extent<decltype(WriteJsonFormatForPayloadWithApplicationId)>::value);
            *outKeyName = "payload";
        }
        else
        {
            p->arrayPayloadDefinitions = nullptr;
            p->payloadDefinitionsLength = 0;
            *outKeyName = nullptr;
        }
        return true;
    }

    static bool WriteEventDataPayloadObject(nn::util::optional<int>* outDefinitionLength, const json::WriteDataDefinition** outDefinitions, void* param, int) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<UploadDeviceEventParam*>(param);
        if (p->arrayPayloadDefinitions == nullptr)
        {
            *outDefinitionLength = nn::util::nullopt;
            *outDefinitions = nullptr;
        }
        else
        {
            *outDefinitionLength = p->payloadDefinitionsLength;
            *outDefinitions = p->arrayPayloadDefinitions;
        }
        return true;
    }

    NN_DETAIL_PCTL_JSON_BEGIN_WRITE(static const, WriteJsonFormatForDeviceEvent)
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_BEGIN(WriteEventsArrayBegin)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_BEGIN(nullptr)
                // Sys Ver.1
                NN_DETAIL_PCTL_JSON_WRITE_KEY("eventType")         NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteEventDataEventType)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("eventTimestamp")    NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteEventDataEventTimestamp)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("timeZone")          NN_DETAIL_PCTL_JSON_WRITE_VALUE_STRING(WriteEventDataTimezone)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("timeZoneUtcOffsetSeconds")  NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteEventDataTimezoneUtcOffsetSeconds)
                NN_DETAIL_PCTL_JSON_WRITE_KEY("elapsedTime")       NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteEventDataElapsedTime)
                // Sys Ver.3
                NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteEventDataApplicationTimesKey)  NN_DETAIL_PCTL_JSON_WRITE_OBJECT_REPEAT(WriteEventDataApplicationTimesObject)
                NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteEventDataMiscTimeKey)          NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteEventDataMiscTime)
                NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteEventDataDisabledTimeKey)      NN_DETAIL_PCTL_JSON_WRITE_VALUE_INT64(WriteEventDataDisabledTime)
                // Sys Ver.1
                NN_DETAIL_PCTL_JSON_WRITE_KEY_OPTIONAL(WriteEventDataPayloadKey)           NN_DETAIL_PCTL_JSON_WRITE_OBJECT_ANY(WriteEventDataPayloadObject)
            NN_DETAIL_PCTL_JSON_WRITE_OBJECT_END()
        NN_DETAIL_PCTL_JSON_WRITE_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_WRITE()
}

nn::Result UploadDeviceEvents::Execute(uint16_t* outPostCount, common::NetworkBuffer& bufferInfo, common::Cancelable* pCancelable, const char* token,
    ServerDeviceId deviceId, const EventData* arrayData, uint16_t dataCount) NN_NOEXCEPT
{
    WatcherErrorHandler* pErrorHandler = new WatcherErrorHandler();
    NN_RESULT_THROW_UNLESS(pErrorHandler != nullptr, nn::pctl::ResultOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        delete pErrorHandler;
    };

    // tokenヘッダーの作成
    // 22 == strlen("Authorization: Bearer ")
    size_t tokenHeaderLength = std::strlen(token) + 23;
    char* tokenHeader = reinterpret_cast<char*>(AllocateMemoryBlock(sizeof(char) * tokenHeaderLength));
    NN_RESULT_THROW_UNLESS(tokenHeader != nullptr, nn::pctl::ResultHttpErrorOutOfMemory());
    NN_UTIL_SCOPE_EXIT
    {
        FreeMemoryBlock(tokenHeader);
    };
    nn::util::SNPrintf(tokenHeader, tokenHeaderLength, "Authorization: Bearer %s", token);

    // URLの作成
    char url[UrlBufferLength_UploadDeviceEvents];

    nn::util::SNPrintf(url, std::extent<decltype(url)>::value, UrlFormat_UploadDeviceEvents,
        ServerEndpoint, deviceId);

    UploadDeviceEventParam param;
    param.dataArray = arrayData;
    param.totalCount = dataCount;
    param.currentIndex = 0;
    param.appDataIndex = 0;
    param.playerDataIndex = 0;
    param.pTempPlayState = GetTempPlayStateData(); // ※ WriteEventsArrayBegin にて元データを取得
    param.playStateCurrentIndex = 0;
    param.isPlayStateUpdated = false;

    json::JsonStructuredWriter writer(&param, WriteJsonFormatForDeviceEvent);

    nn::pctl::detail::service::json::JsonHttpInputStream stream;
    json::JsonDataHandler handler(nullptr, nullptr, 0);

    NN_RESULT_DO(stream.GetRequest().Open("POST", url));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader("Content-Type: application/json"));
    NN_RESULT_DO(stream.GetRequest().AddRequestHeader(tokenHeader));

    NN_RESULT_DO(
        json::ParseWebStreamWithPostJSON(&writer, &handler, pErrorHandler, &stream,
            bufferInfo.GetBufferForJsonValue(), common::NetworkBuffer::JsonValueBufferMemorySize,
            bufferInfo.GetBufferForStream(), common::NetworkBuffer::StreamBufferMemorySize,
            pCancelable)
    );

    // ここに到達している場合は送信に成功しているので、
    // イベントを元に更新を行ったプレイ状態のデータを更新する
    if (param.isPlayStateUpdated)
    {
        auto holder = PlayStateHolder::Acquire();
        std::memcpy(holder.Get(), param.pTempPlayState, sizeof(*(param.pTempPlayState)));
        g_pWatcher->GetWatcherEventManager().PostSavePlayState(); // 非同期処理
    }

    *outPostCount = param.currentIndex;

    NN_RESULT_SUCCESS;
}

}}}}}}
