﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/pctl_ApplicationStateManager.h>

#include <nn/pctl/detail/pctl_Log.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/service/pctl_ServiceMain.h>

#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/time/time_TimeZoneApi.h>

namespace nn { namespace pctl { namespace detail { namespace service {

void ApplicationStateManager::RecordPlayDataBasedApplicationEvents(uint32_t& eventOffset, bool readWithoutRecord) NN_NOEXCEPT
{
    static const int MaxEvents = 32;
    typedef nn::pdm::ApplicationEvent ApplicationEventArray[MaxEvents]; // VS2013でエラーが出るため一旦 typedef
    NN_FUNCTION_LOCAL_STATIC(ApplicationEventArray, s_AppEvents);

    auto& states = g_pMain->GetStatesRef();

    NN_DETAIL_PCTL_TRACE("[pctl] Check pdm application events...\n");

    while (NN_STATIC_CONDITION(true))
    {
        auto count = nn::pdm::QueryApplicationEvent(
            s_AppEvents, MaxEvents, static_cast<int>(eventOffset)
        );
        if (count == 0)
        {
            break;
        }
        for (int i = 0; i < count; ++i)
        {
            auto& e = s_AppEvents[i];
            // ネットワーク時計が無効なデータは扱わない
            if (e.eventTime.networkClockTime == 0)
            {
                continue;
            }
            nn::time::PosixTime t = nn::time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(static_cast<int64_t>(e.eventTime.networkClockTime));
            if (readWithoutRecord)
            {
                continue;
            }
            // 起動イベントは必ず、それ以外のイベントは直前に起動されているアプリケーションのIDと一致しているもののみ取る
            if (e.eventType == nn::pdm::AppletEventType::Launch || e.applicationId == states.applicationIdFromPdmEvent)
            {
                switch (e.eventType)
                {
                    case nn::pdm::AppletEventType::Launch:
                        NN_DETAIL_PCTL_TRACE("[pctl] ID=0x%016llX: Launch event (ignore).\n",
                            e.applicationId.value);
                        // 起動イベントは独自ハンドリングで対応(特殊用途アプリケーションかどうかのチェックを行うため)
                        break;
                    case nn::pdm::AppletEventType::Exit:
                    case nn::pdm::AppletEventType::Terminate:
                    case nn::pdm::AppletEventType::AbnormalExit:
                        NN_DETAIL_PCTL_TRACE("[pctl] ID=0x%016llX: Exit/Terminate/AbnormalExit (%d) event (ignore).\n",
                            e.applicationId.value, static_cast<int>(e.eventType));
                        // 終了イベントは独自ハンドリングで対応(起動イベントと揃える)
                        break;
                    case nn::pdm::AppletEventType::Background:
                        NN_DETAIL_PCTL_TRACE("[pctl] ID=0x%016llX: Background event.\n",
                            e.applicationId.value);
                        AddApplicationSuspendEvent(e.applicationId, t);
                        break;
                    case nn::pdm::AppletEventType::InFocus:
                    case nn::pdm::AppletEventType::OutOfFocus: // Backgroundでなければ非Suspendとみなす
                        NN_DETAIL_PCTL_TRACE("[pctl] ID=0x%016llX: InFocus/OutOfFocus (%d) event.\n",
                            e.applicationId.value, static_cast<int>(e.eventType));
                        if (states.isApplicationSuspended)
                        {
                            AddApplicationResumeEvent(e.applicationId, t);
                        }
                        break;
                    default:
                        // 上記以外のイベントに対しては何もしない(現状では存在しない)
                        break;
                }
            }
        }
        eventOffset = s_AppEvents[count - 1].eventTime.eventIndex + 1;
    }
}

void ApplicationStateManager::RecordPlayDataBasedAccountEvents(uint32_t& eventOffset, bool readWithoutRecord) NN_NOEXCEPT
{
    static const int MaxEvents = 32;
    typedef nn::pdm::AccountEvent AccountEventDataArray[MaxEvents]; // VS2013でエラーが出るため一旦 typedef
    NN_FUNCTION_LOCAL_STATIC(AccountEventDataArray, s_AccountEvents);

    NN_DETAIL_PCTL_TRACE("[pctl] Check pdm account events...\n");

    while (NN_STATIC_CONDITION(true))
    {
        auto count = nn::pdm::QueryAccountEvent(
            s_AccountEvents, MaxEvents, static_cast<int>(eventOffset)
        );
        if (count == 0)
        {
            break;
        }
        for (int i = 0; i < count; ++i)
        {
            auto& e = s_AccountEvents[i];
            // ネットワーク時計が無効なデータは扱わない(同一時刻は可)
            if (e.eventTime.networkClockTime.value == 0)
            {
                continue;
            }
            if (readWithoutRecord)
            {
                continue;
            }
            switch (e.eventType)
            {
                case nn::pdm::UserAccountEventType::Open:
                    NN_DETAIL_PCTL_TRACE("[pctl] ID=%08x_%08x_%08x_%08x: User open event.\n",
                        static_cast<uint32_t>(e.uid._data[0] >> 32),
                        static_cast<uint32_t>(e.uid._data[0] & 0xFFFFFFFFull),
                        static_cast<uint32_t>(e.uid._data[1] >> 32),
                        static_cast<uint32_t>(e.uid._data[1] & 0xFFFFFFFFull));
                    AddUserOpenEvent(e.uid, e.eventTime.networkClockTime);
                    break;
                case nn::pdm::UserAccountEventType::Close:
                    NN_DETAIL_PCTL_TRACE("[pctl] ID=%08x_%08x_%08x_%08x: User close event.\n",
                        static_cast<uint32_t>(e.uid._data[0] >> 32),
                        static_cast<uint32_t>(e.uid._data[0] & 0xFFFFFFFFull),
                        static_cast<uint32_t>(e.uid._data[1] >> 32),
                        static_cast<uint32_t>(e.uid._data[1] & 0xFFFFFFFFull));
                    AddUserCloseEvent(e.uid, e.eventTime.networkClockTime);
                    break;
                default:
                    // 上記以外のイベントに対しては何もしない(現状では存在しない)
                    break;
            }
        }
        eventOffset = s_AccountEvents[count - 1].eventTime.eventIndex + 1;
    }
}

}}}}
