﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/service/common/pctl_TimeUtil.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_TimeZoneApi.h>

namespace nn { namespace pctl { namespace detail { namespace service { namespace common {

namespace
{
    // @brief hour/minute で表される時刻が calTime の時刻より手前(過去)かどうかを返します。
    // @details hour/minute の時刻は 0 秒ちょうどとみなします。
    inline bool IsPastTime(int hour, int minute, const nn::time::CalendarTime& calTime) NN_NOEXCEPT
    {
        if (hour < calTime.hour)
        {
            return true;
        }
        else if (hour > calTime.hour)
        {
            return false;
        }
        if (minute < calTime.minute)
        {
            return true;
        }
        else if (minute > calTime.minute)
        {
            return false;
        }
        // hour/minute の時刻は 0 秒扱い
        return calTime.second > 0;
    }
    // @brief hour/minute で表される時刻が calTime の時刻と同じまたは手前(過去)かどうかを返します。
    // @details hour/minute の時刻は 0 秒ちょうどとみなします。
    inline bool IsPastOrEqualTime(int hour, int minute, const nn::time::CalendarTime& calTime) NN_NOEXCEPT
    {
        return (hour == calTime.hour && minute == calTime.minute && 0 == calTime.second) ||
            IsPastTime(hour, minute, calTime);
    }

    // @brief 次の指定ローカル時刻になる絶対時刻を返します。
    // @details outCalTime は nullptr を指定することができます。
    nn::Result GetTimeOfNextClockPointImpl(nn::time::PosixTime* outTime, nn::time::CalendarTime* outCalTime,
        const nn::time::PosixTime& currentTime, const nn::time::CalendarTime& currentCalTime, int hourNext, int minuteNext) NN_NOEXCEPT
    {
        nn::time::CalendarTime calTime = currentCalTime;

        // 指定時刻が現在時刻より手前の場合は「次の日」を計算
        // (hour と minute が一致する場合も次の日を得るようにする)
        if (IsPastOrEqualTime(hourNext, minuteNext, calTime))
        {
            auto timeTemp = nn::time::ToPosixTimeFromUtc(calTime);
            calTime = nn::time::ToCalendarTimeInUtc(timeTemp + nn::TimeSpan::FromDays(1));
        }
        calTime.hour = static_cast<int8_t>(hourNext);
        calTime.minute = static_cast<int8_t>(minuteNext);
        calTime.second = 0;

        nn::time::LocationName  namePrevious;
        nn::time::GetDeviceLocationName(&namePrevious);

        while (NN_STATIC_CONDITION(true))
        {
            nn::time::PosixTime times[2];
            int c = 0;
            NN_RESULT_DO(nn::time::ToPosixTime(&c, times, 2, calTime));
            switch (c)
            {
                case 0:
                    // 結果が0個の場合、すなわち指定時刻が存在しない場合は仮想的な時刻を設定
                    // (計算上の指定時刻になる絶対時刻(実際にはその時刻にはならない)を返す)

                    // (CalendarTime の差分は、CalendarTime の日付をUTCに変換することで計算できる)
                    *outTime = currentTime + (nn::time::ToPosixTimeFromUtc(calTime) - nn::time::ToPosixTimeFromUtc(currentCalTime));
                    // 自前で計算していることで *outTime は calTime が指す日付時刻ではなくなっているため、
                    // ToCalendarTime を呼び出して取得する必要がある
                    if (outCalTime != nullptr)
                    {
                        NN_RESULT_DO(nn::time::ToCalendarTime(outCalTime, nullptr, *outTime));
                    }
                    break;
                case 1:
                case 2:
                    // 結果が1つだけの場合はそれを、2つ取得できている場合は後ろの時刻を返す
                    // (2つの場合: 夏時間切り替わりで特定時刻が2回発生するケース)
                    // → c - 1 (== 末尾) のデータを返す
                    *outTime = times[c - 1];
                    if (outCalTime != nullptr)
                    {
                        *outCalTime = calTime;
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }

            // ToPosixTime, ToCalendarTime の呼び出しを行っている間に値が変わっていれば
            // ループして再取得するようにする
            // (本当は共通の TimeZoneRule を使いまわす方が良いが、
            // TimeZoneRule のデータサイズが大きいので LocationName のチェックにとどめておく)
            nn::time::LocationName nameCurrent;
            nn::time::GetDeviceLocationName(&nameCurrent);
            if (nameCurrent != namePrevious)
            {
                namePrevious = nameCurrent;
                continue;
            }
            break;
        }
        NN_RESULT_SUCCESS;
    }
}

// @brief 次の指定ローカル時刻になる絶対時刻を返します。
// @details outCalTime は nullptr を指定することができます。
nn::Result GetTimeOfNextClockPoint(nn::time::PosixTime* outTime, nn::time::CalendarTime* outCalTime,
    const nn::time::PosixTime& currentTime, int hourNext, int minuteNext) NN_NOEXCEPT
{
    nn::time::CalendarTime currentCalTime;
    NN_RESULT_DO(nn::time::ToCalendarTime(&currentCalTime, nullptr, currentTime));
    return GetTimeOfNextClockPointImpl(outTime, outCalTime, currentTime, currentCalTime, hourNext, minuteNext);
}

// @brief 手前の指定ローカル時刻になる絶対時刻を返します。
// @details outCalTime は nullptr を指定することができます。
nn::Result GetTimeOfPreviousClockPoint(nn::time::PosixTime* outTime, nn::time::CalendarTime* outCalTime,
    const nn::time::PosixTime& currentTime, int hourPrevious, int minutePrevious) NN_NOEXCEPT
{
    nn::time::CalendarTime currentCalTime;
    NN_RESULT_DO(nn::time::ToCalendarTime(&currentCalTime, nullptr, currentTime));

    auto timeTemp = nn::time::ToPosixTimeFromUtc(currentCalTime);
    // 「(1日+1秒)前の時刻から見て次の時刻」が「手前の時刻」になる
    // (「1日ちょうど前の時刻から見て次の時刻」ではない)
    currentCalTime = nn::time::ToCalendarTimeInUtc(
        timeTemp - nn::TimeSpan::FromSeconds(24 * 60 * 60 + 1)
    );

    return GetTimeOfNextClockPointImpl(outTime, outCalTime, currentTime, currentCalTime, hourPrevious, minutePrevious);
}

}}}}}
