﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pctl/detail/pctl_ResultHandler.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/err/err_Api.h>
#include <nn/nifm/nifm_ResultMenu.h>
#include <nn/pctl/pctl_ResultPrivate.h>
#include <nn/pctl/detail/pctl_NpnsErrorHandler.h>
#include <nn/pctl/detail/pctl_NpnsErrorHandler.impl.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>

namespace nn { namespace pctl { namespace detail {

// pctl_NpnsErrorHandlerForService.cpp でも定義するため、
// サービス処理がHIPC(別プロセス)の場合のみ有効化する
#if NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_HIPC
bool IsAcceptableNpnsResult(nn::Result result) NN_NOEXCEPT
{
    return IsAcceptableNpnsResultImpl(result);
}

nn::Result ConvertNpnsError(nn::Result result) NN_NOEXCEPT
{
    return ConvertNpnsErrorImpl(result);
}
#endif

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
nn::Result AbortIfUnexpectResultImpl(nn::Result result, const char* functionName) NN_NOEXCEPT
#else
nn::Result AbortIfUnexpectResultImpl(nn::Result result) NN_NOEXCEPT
#endif
{
    // pctlのエラーでも以下のエラー以外は予期しないエラーなのでabort扱いにする
    // - nn::nifm::ResultRequestRejected のエラーはエラービューアアプレットなどで
    //   ハンドリング可能であるためそのまま返す
    // - IsAcceptableNpnsResult が true となるエラーもそのまま返す
    //   (npns 由来のエラーは IsAcceptableNpnsResult が true となるか、
    //   事前に ConvertNpnsError で pctl のResultに変換済みであるため、
    //   以下の if 文の結果が常に true になる)
    if (result.IsSuccess() ||
        nn::pctl::ResultRestricted::Includes(result) ||
        nn::pctl::ResultWrongPinCode::Includes(result) ||
        nn::pctl::ResultConfigurationNotAllowed::Includes(result) ||
        nn::pctl::ResultWatcherError::Includes(result) ||
        nn::nifm::ResultRequestRejected::Includes(result) ||
        IsAcceptableNpnsResult(result))
        return result;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    if (functionName != nullptr)
    {
        NN_SDK_LOG("[pctl] Unexpected error occurred at function '%s'\n", functionName);
    }
#endif
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT("unreachable");
}

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
void HandleResultImpl(nn::Result result, bool isShowError, const char* functionName) NN_NOEXCEPT
#else
void HandleResultImpl(nn::Result result, bool isShowError) NN_NOEXCEPT
#endif
{
    if (result.IsSuccess())
    {
        return;
    }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    AbortIfUnexpectResultImpl(result, functionName);
#else
    AbortIfUnexpectResultImpl(result);
#endif

    if (nn::pctl::ResultCanceled::Includes(result))
    {
        // キャンセルはエラービューアー表示対象とせず簡易的なログ出力のみとする
        NN_SDK_LOG("[pctl] operation canceled\n");
    }
    else if (isShowError)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::err::ShowError(result);
#else
        NN_SDK_LOG("[pctl] *** ErrorViewer 2%03d-%04d ***\n", result.GetModule(), result.GetDescription());
        NN_SDK_LOG("[pctl] An error has occurred.\n");
#endif
    }
    else
    {
        // 簡易的なログ出力を行う
        NN_SDK_LOG("[pctl] Error: result = 0x%08lX\n", result.GetInnerValueForDebug());
    }
}

}}}
