﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <type_traits>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/pcie_LoggedState.h>
#include <nn/pcie/driver/pcie.h>
#include <nn/pcie/server/pcie_HipcServer.h>

#include "../detail/pcie_Memory.h"
#include "../detail/pcie_ServiceName.h"

#include "pcie_ManagerImpl.h"
#include "pcie_LoggedStateManagerImpl.h"

namespace nn { namespace pcie { namespace server {

namespace {

struct MyServerManagerOptions
{
    static const size_t PointerTransferBufferSize = 4096;
};

struct  HipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 4, HipcTag> Allocator;

// HipcSimpleAllInOneServerManager は HIPC のサーバのポートとセッションを一元管理する。
class MyServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<(MaxClientCount * 2) + MaxLoggedStateClientCount, 2, MyServerManagerOptions>
{};

// Allocate storage for server
std::aligned_storage<sizeof(MyServerManager), NN_ALIGNOF(MyServerManager)>::type g_MyServerManagerStorage;
MyServerManager *g_pMyServerManager = nullptr;

// MyManagerImpl for normal class driver use
typedef nn::sf::UnmanagedServiceObject<detail::IManager, ManagerImpl> MyManagerImpl;
std::aligned_storage<sizeof(MyManagerImpl), NN_ALIGNOF(MyManagerImpl)>::type g_MyManagerImplStorage;
MyManagerImpl *g_pMyManagerImpl = nullptr;

// MyLoggedStateManagerImpl for telemetry and testing
typedef nn::sf::UnmanagedServiceObject<detail::ILoggedStateManager, LoggedStateManagerImpl> MyLoggedStateManagerImpl;
std::aligned_storage<sizeof(MyManagerImpl), NN_ALIGNOF(MyLoggedStateManagerImpl)>::type g_MyLoggedStateManagerImplStorage;
MyLoggedStateManagerImpl *g_pMyLoggedStateManagerImpl = nullptr;

// This is the heap for the process, used primarily by the underlying PCIe driver
static uint8_t s_PcieHeapMemory[256 * 1024];

}



void InitializeServer() NN_NOEXCEPT
{
    Result result;

    NN_ABORT_UNLESS( !g_pMyServerManager);

    // Initialize heap used by the driver
    nn::pcie::detail::MemoryInit(s_PcieHeapMemory, sizeof(s_PcieHeapMemory));

    // Construct MyServerManager
    g_pMyServerManager = new(&g_MyServerManagerStorage) MyServerManager;
    NN_ABORT_UNLESS(g_pMyServerManager);

    // Initialize the underlying PCIe root complex driver
    if(!(result=nn::pcie::driver::Initialize()).IsSuccess())
    {
        NN_ABORT("nn::pcie::driver::Initialize() failed %d:%d\n",
                 result.GetModule(), result.GetDescription());
    }

    // Construct ManagerImpl
    g_pMyManagerImpl = new(&g_MyManagerImplStorage) MyManagerImpl();
    NN_ABORT_UNLESS(g_pMyManagerImpl);

    // Construct LoggedStateManagerImpl
    g_pMyLoggedStateManagerImpl = new(&g_MyLoggedStateManagerImplStorage) MyLoggedStateManagerImpl();
    NN_ABORT_UNLESS(g_pMyLoggedStateManagerImpl);

    // Register ports for ManagerImpl and LoggedStateManagerImpl
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_pMyServerManager->RegisterObjectForPort(g_pMyManagerImpl->GetShared(),
                                                  MaxClientCount, detail::ServiceName)
    );
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_pMyServerManager->RegisterObjectForPort(g_pMyLoggedStateManagerImpl->GetShared(),
                                                  MaxLoggedStateClientCount, detail::LoggedStateServiceName)
    );

    // サーバマネージャの開始
    // ただし、実際のサーバ動作は、LoopAuto 関数等を呼び出すことで行う必要がある。
    g_pMyServerManager->Start();
}

void LoopServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pMyServerManager);

    // サーバの処理ループ実行を実行する。
    // - ポートへの接続要求
    // - セッションへの処理要求
    // RequestStop が呼ばれるまで返らない。
    g_pMyServerManager->LoopAuto();
}

void RequestStopServer() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pMyServerManager);

    // 処理ループの停止リクエストを送り、LoopAuto 関数を返す
    g_pMyServerManager->RequestStop();
}

void FinalizeServer() NN_NOEXCEPT
{
    Result result;

    NN_ABORT_UNLESS(g_pMyServerManager);

    // MyServerManager のデストラクト
    // 登録したサービスなどはここで登録が解除される。
    g_pMyServerManager-> ~MyServerManager();
    g_pMyServerManager = nullptr;

    if(!(result=nn::pcie::driver::Finalize()).IsSuccess())
    {
        NN_ABORT("nn::pcie::driver::Finalize() failed %d:%d\n",
                 result.GetModule(), result.GetDescription());
    }

    nn::pcie::detail::MemoryFinalize();
}

}
}
}
