﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {

Function::Function(Driver *pDriver, Device *pDevice, Profile *pProf)
    : m_pDriver(pDriver)
    , m_pBus(nullptr)
    , m_pRootComplex(nullptr)
    , m_pDevice(pDevice)
    , m_HeaderType(pProf->headerType)
    , m_ClassCode(pProf->classCode)
    , m_FuncNum(pProf->funcNum)
    , m_VendorId(pProf->vendorId)
    , m_DeviceId(pProf->deviceId)
    , m_Revision(pProf->revision)
    , m_PmState(PmState_Invalid)
    , m_FunctionType(FunctionType_Base)
    , m_CapableSpeed(BusSpeed_Invalid)
    , m_CurrentSpeed(BusSpeed_Invalid)
    , m_IsPCIe(false)
    , m_ExpCapOffset(0)
    , m_ExpFlags(0)
    , m_DevCapPayload(0)
    , m_NumResources(0)
{
}

Function::~Function()
{

}

Result Function::Initialize()
{
    uint16_t cmdVal;
    DeviceNumber devNum = GetDevNum();
    Result result = ResultSuccess();
    for (int32_t index = 0; index < StandardConfigMaxBaseAddresses; index++)
    {
        m_Resources[index].index = index;
        m_Resources[index].size  = 0;
    }
    m_pDevice->AddFunction(*this);
    m_PmState = PmState_D0;

    // resolve bus and root complex for use later
    m_pBus = GetBus();
    m_pBus->GetRootComplex(&m_pRootComplex);

    // For now, turn off IO, MEM and bus master capability
    m_pBus->ReadConfigWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Command, &cmdVal);
    cmdVal &= ~(StandardConfigValue_CommandIo | StandardConfigValue_CommandMemory | StandardConfigValue_CommandMaster);
    cmdVal |= (StandardConfigValue_CommandIntxDisable);
    m_pBus->WriteConfigWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Command, cmdVal);

    // Detection
    DetectResources();
    DetectAttributes();

    // Although said to be unused on PCIe, always configure cache size
    m_pBus->WriteConfigByte(devNum, m_FuncNum, StandardConfigOffset_CacheLineSize, DataCacheLineSize >> 2);

    // Bus speed
    InitSpeed();

    return result;
}

Result Function::Finalize()
{
    Result result;

    // remove registered resources from bus
    for (int32_t resIndex = 0; resIndex < m_NumResources; resIndex++)
    {
        Resource *pRes = m_Resources + resIndex;
        if (pRes->size != 0)
        {
            m_pBus->UnRegisterResource(pRes);
            pRes->size = 0;
        }
    }

    // remove this function from hosting device
    result = m_pDevice->RemoveFunction(m_FuncNum);
    m_PmState = m_pDevice->GetResetUponResumeEnable() ? PmState_D3Off : PmState_D3Hot;

    return result;
}

void Function::Display()
{
    char classCodeDesc[64];
    GetClassCodeDescription(classCodeDesc, sizeof(classCodeDesc));
    NN_PCIE_LOG_NOPREFIX("BDF-%03x:%02x:%x, Class-%s, VID-0x%04x, DEVID-0x%04x, Speed-%s/%s\n",
                         GetBus()->GetBusNum(), GetDevNum(), GetFuncNum(),
                         classCodeDesc, m_VendorId, m_DeviceId,
                         GetBusSpeedDescription(m_CurrentSpeed), GetBusSpeedDescription(m_CapableSpeed));
}

Result Function::ConfigureResource(Resource *pRes)
{
    Result result = ResultSuccess();
    BusAddress busAddr;
    if ((result = m_pDriver->GetBusAddressess(pRes->address, &busAddr)).IsSuccess())
    {
        uint16_t cmdVal;
        DeviceNumber devNum = GetDevNum();
        int32_t where = StandardConfigOffset_BaseAddress0 + pRes->index * 4;
        NN_PCIE_LOG_VERBOSE("BDF-%03x:%02x:%x BAR %d settings:\n",
                            GetBus()->GetBusNum(), GetDevNum(), GetFuncNum(), pRes->index);
        NN_PCIE_LOG_VERBOSE("  PA    = 0x%llx\n", busAddr);
        NN_PCIE_LOG_VERBOSE("  EA    = 0x%p\n", pRes->address);
        NN_PCIE_LOG_VERBOSE("  size  = 0x%x\n", pRes->size);
        result = m_pBus->WriteConfigDWord(devNum, m_FuncNum, where, busAddr);

        // Make sure command bit is enabled for this type of BAR
        m_pBus->ReadConfigWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Command, &cmdVal);
        if ((pRes->flags & ResourceFlag_Io) && !(cmdVal & StandardConfigValue_CommandIo))
        {
            cmdVal |= StandardConfigValue_CommandIo;
            m_pBus->WriteConfigWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Command, cmdVal);
        }
        else if ((pRes->flags & ResourceFlag_Mem) && !(cmdVal & StandardConfigValue_CommandMemory))
        {
            cmdVal |= StandardConfigValue_CommandMemory;
            m_pBus->WriteConfigWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Command, cmdVal);
        }
    }
    return result;
}

FullClassCode Function::GetClassCode()
{
    return m_ClassCode;
}

ClassCode Function::Get16BitClassCode()
{
    return static_cast<ClassCode>((m_ClassCode & 0x00ffff00) >> 8);
}

BaseClassCode Function::GetBaseClassCode()
{
    return static_cast<BaseClassCode>((m_ClassCode & 0x00ff0000) >> 16);
}

SubClassCode Function::GetSubClassCode()
{
    return static_cast<SubClassCode>((m_ClassCode & 0x0000ff00) >> 8);
}

uint8_t Function::GetProgClassCode()
{
    return static_cast<uint8_t>(m_ClassCode & 0x000000ff);
}

uint16_t Function::GetVendorId()
{
    return m_VendorId;
}

uint16_t Function::GetDeviceId()
{
    return m_DeviceId;
}

uint8_t Function::GetRevision()
{
    return m_Revision;
}

PmState Function::GetPmState()
{
    return m_PmState;
}

BusSpeed Function::GetCurrentSpeed()
{
    return m_CurrentSpeed;
}

BusSpeed Function::GetCapableSpeed()
{
    return m_CapableSpeed;
}

Result Function::ReadSpeed(BusSpeed *pRetCapability, BusSpeed *pRetCurrentSpeed)
{
    uint16_t regVal16;
    Result result = ResultSuccess();
    BusSpeed current = BusSpeed_Invalid;
    BusSpeed capability = BusSpeed_Invalid;

    do
    {
        // Get current speed
        NN_PCIE_BREAK_UPON_ERROR(ReadPCIeCapabilityWord(PcieOffset_LinkStatus, &regVal16));
        switch (regVal16 & PcieValue_LinkStatusCurrentLinkSpeed)
        {
        case PcieValue_LinkStatusCurrentLinkSpeed_2_5GB:
            current = BusSpeed_EGen1;
            break;
        case PcieValue_LinkStatusCurrentLinkSpeed_5_0GB:
            current = BusSpeed_EGen2;
            break;
        case PcieValue_LinkStatusCurrentLinkSpeed_8_0GB:
            current = BusSpeed_EGen3;
            break;
        default:
            break;
        }

        // Get capability
        NN_PCIE_BREAK_UPON_ERROR(ReadPCIeCapabilityWord(PcieOffset_LinkCapability, &regVal16));
        switch (regVal16 & PcieValue_LinkCapabilitySupportedLinkSpeeds)
        {
        case PcieValue_LinkCapabilitySupportedLinkSpeed_2_5GB:
            capability = BusSpeed_EGen1;
            break;
        case PcieValue_LinkCapabilitySupportedLinkSpeed_5_0GB:
            capability = BusSpeed_EGen2;
            break;
        default:
            break;
        }

    }while (false);

    *pRetCapability = capability;
    *pRetCurrentSpeed = current;

    return result;
}

Bus* Function::GetBus()
{
    return m_pDevice->GetBus();
}

bool Function::IsPcie()
{
    return m_IsPCIe;
}

uint8_t Function::GetPCIeFlagsType()
{
    return (uint8_t)((m_ExpFlags & PcieValue_FlagsType) >> 4);
}

DeviceNumber Function::GetDevNum()
{
    return m_pDevice->GetDevNum();
}

Device* Function::GetDevice()
{
    return m_pDevice;
}

void Function::GetClassCodeDescription(char *buffer, size_t bufferSize)
{
    const char *pBaseClass = "unknown";
#define DEFINE_PCI_BASE_CLASS_CASE(BASENAME) case BASENAME: pBaseClass=#BASENAME;break
    switch (GetBaseClassCode())
    {
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeNotDefined);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeStorage);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeNetwork);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeDisplay);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeMultimedia);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeMemory);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeBridge);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeCommunication);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeSystem);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeInput);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeDocking);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeProcessor);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeSerial);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeWireless);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeIntelligent);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeSatellite);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeCrypt);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeSignalprocessing);
        DEFINE_PCI_BASE_CLASS_CASE(BaseClassCodeOthers);
    default:
        break;
    }
    snprintf(buffer, bufferSize, "%s-%04x-%02x",
             pBaseClass, Get16BitClassCode(), GetProgClassCode());
}
FunctionNumber Function::GetFuncNum()
{
    return m_FuncNum;
}

Function::FunctionType Function::GetFunctionType()
{
    return m_FunctionType;
}

uint8_t Function::GetHeaderType()
{
    return m_HeaderType;
}

void Function::GetFunctionProfile(Profile *pProf)
{
    pProf->vendorId   = m_VendorId;
    pProf->deviceId   = m_DeviceId;
    pProf->classCode  = m_ClassCode;
    pProf->headerType = m_HeaderType;
    pProf->revision   = m_Revision;
    pProf->state      = m_PmState;
    pProf->capableSpeed = m_CapableSpeed;
    pProf->speed      = m_CurrentSpeed;
}

Result Function::GetBarProfile(uint8_t bar, DriverBarProfile *pProfile)
{
    Result result = ResultBarDoesNotExist();

    // 1:1 mapping between resources and BARs at this time
    int32_t resIndex = (int32_t)bar;

    if (resIndex < m_NumResources)
    {
        if (m_Resources[resIndex].size != 0)
        {
            BusAddress busAddress = 0;
            m_pDriver->GetBusAddressess(m_Resources[resIndex].address,&busAddress);
            pProfile->flags = m_Resources[resIndex].flags;
            pProfile->pBase = (void *)m_Resources[resIndex].address;
            pProfile->size  = static_cast<uint64_t>(m_Resources[resIndex].size);
            pProfile->base  = static_cast<uint64_t>(busAddress);
            result = ResultSuccess();
        }
    }
    return result;
}

Result Function::GetBarRegion(ResourceAddr *pReturnedBase, uint8_t bar, ResourceSize offset, ResourceSize size)
{
    Result result = ResultSuccess();

    // 1:1 mapping between resources and BARs at this time
    int32_t resIndex = static_cast<int32_t>(bar);

    do
    {
        Resource *pRes = NULL;
        if (resIndex >= m_NumResources)
        {
            result = ResultBarDoesNotExist();
            break;
        }
        pRes = m_Resources + resIndex;
        if (pRes->size == 0)
        {
            result = ResultBarDoesNotExist();
            break;
        }
        if (offset >= pRes->size)
        {
            result = ResultInvalidRegisterOffset();
            break;
        }
        if ((offset + size) >= pRes->size)
        {
            result = ResultInvalidRegisterSize();
            break;
        }
        *pReturnedBase = pRes->address + offset;
    }while(false);

    return result;
}

int32_t Function::FindCapability(CapabilityId cap)
{
    uint8_t startingOffset = FindCapabilityStart();
    int32_t returnedOffset = 0;
    if (startingOffset != 0)
    {
        returnedOffset = FindNextCapability((uint8_t)startingOffset, cap);
    }
    return returnedOffset;
}

int32_t Function::FindExtendedCapability(ExtendedCapabilityId extCap)
{
    return FindNextExtendedCapability(0, extCap);
}

Result Function::DoFunctionLevelResetWait()
{
    Result result = ResultOperationTimeout();
    int iterations = 0;
    const int MaxWaitInMs = 100;
    const int IterationSleepInMs = 10;

    do
    {
        uint32_t cmdVal32 = 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(IterationSleepInMs));
        m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Command, &cmdVal32);
        if(cmdVal32 != 0xffffffff)
        {
            result = ResultSuccess();
            break;
        }
    }while(iterations++ < (MaxWaitInMs / IterationSleepInMs));

    NN_PCIE_LOG_VERBOSE("Function::DoFunctionLevelResetWait() - waited %d ms.\n",
                        iterations*IterationSleepInMs);

    return result;
}

Result Function::ReadPCIeCapabilityWord(int32_t where, uint16_t *pWord)
{
    Result result = ResultNotAvailable();
    if (IsPCIeCapabilityImplemented(where))
    {
        result = m_pBus->ReadConfigWord(GetDevNum(), m_FuncNum, m_ExpCapOffset + where, pWord);
    }
    return result;
}

Result Function::ReadPCIeCapabilityDWord(int32_t where, uint32_t *pDWord)
{
    Result result = ResultNotAvailable();
    if (IsPCIeCapabilityImplemented(where))
    {
        result = m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, m_ExpCapOffset + where, pDWord);
    }
    return result;
}

Result Function::WritePCIeCapabilityWord(int32_t where, uint16_t word)
{
    Result result = ResultNotAvailable();
    if (IsPCIeCapabilityImplemented(where))
    {
        result = m_pBus->WriteConfigWord(GetDevNum(), m_FuncNum, m_ExpCapOffset + where, word);
    }
    return result;
}

Result Function::WritePCIeCapabilityDWord(int32_t where, uint32_t dWord, uint32_t writeMask)
{
    Result result = ResultNotAvailable();
    if (IsPCIeCapabilityImplemented(where))
    {
        uint32_t reg;
        if (writeMask == 0xffffffff)
        {
            reg = dWord;
        }
        else
        {
            result = m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, m_ExpCapOffset + where, &reg);
            if (result.IsSuccess())
            {
                reg &= ~writeMask;
                reg |= (dWord & writeMask);
            }
            else
            {
                return result;
            }
        }
        result = m_pBus->WriteConfigDWord(GetDevNum(), m_FuncNum, m_ExpCapOffset + where, reg);
    }
    return result;
}


// Read a cap/ctrl dword in the extended cap/config space
Result Function::ReadPCIeExtendedCapabilityDWord(int32_t where, uint32_t *dWord)
{
    Result result = ResultNotAvailable();

    if ((uint32_t)where < (StandardConfigSpaceExpressSize - sizeof(uint32_t)))
    {
        uint32_t reg;
        result = m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, where, &reg);
        if (result.IsSuccess()) {
            *dWord = reg;
        }
    }

    return result;
}


// Write to a control dword in the extended cap/config space.  Bits with 1s in
// writeMask will be written.  The default value for writeMask is 0xffff_fffff.
Result Function::WritePCIeExtendedCapabilityDWord(int32_t where, uint32_t dWord, uint32_t writeMask)
{
    Result result = ResultNotAvailable();

    if ((uint32_t)where < (StandardConfigSpaceExpressSize - sizeof(uint32_t)))
    {
        uint32_t reg;
        result = m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, where, &reg);
        if (result.IsSuccess()) {
            reg &= ~writeMask;
            reg |= (dWord & writeMask);
            result = m_pBus->WriteConfigDWord(GetDevNum(), m_FuncNum, where, reg);
        }
    }

    return result;
}


// Get Maximum payload size supported by this device function
Result Function::GetMaxPayloadSize(int32_t *pRetMaxPayloadSize)
{
    Result result;
    uint16_t val16;
    if ((result = ReadPCIeCapabilityWord(PcieOffset_DeviceControl, &val16)).IsSuccess())
    {
        *pRetMaxPayloadSize = 128 << ((val16 & PcieValue_DeviceControlPayload) >> 5);
    }
    else
    {
        *pRetMaxPayloadSize = 0;
    }
    return result;
}

// Set Maximum payload size supported by this device function.
// It's allowed to go to smaller value only, as its default
// is set to maximum supported.
Result Function::SetMaxPayloadSize(int32_t maxPayloadSize)
{
    uint8_t requestedMPS = 0;
    Result result = ResultSuccess();
    switch (maxPayloadSize)
    {
    case 128:
        requestedMPS = 0;
        break;
    case 256:
        requestedMPS = 1;
        break;
    case 512:
        requestedMPS = 2;
        break;
    case 1024:
        requestedMPS = 3;
        break;
    case 2048:
        requestedMPS = 4;
        break;
    case 4096:
        requestedMPS = 5;
        break;
    default:
        result = ResultInvalidSize();
        break;
    }
    if (result.IsSuccess())
    {
        if (requestedMPS <= m_DevCapPayload)
        {
            uint16_t val16;
            if ((result = ReadPCIeCapabilityWord(PcieOffset_DeviceControl, &val16)).IsSuccess())
            {
                val16 &= ~PcieValue_DeviceControlPayload;
                val16 |= (requestedMPS << 5);
                result = WritePCIeCapabilityWord(PcieOffset_DeviceControl, val16);
            }
        }
        else
        {
            result = ResultMaximumExceeded();
        }
    }
    return result;
}

// Get Maximum payload read request size supported by this device function
Result Function::GetMaxReadRequestSize(int32_t *pRetMaxSize)
{
    Result result;
    uint16_t val16;
    if ((result = ReadPCIeCapabilityWord(PcieOffset_DeviceControl, &val16)).IsSuccess())
    {
        *pRetMaxSize = 128 << ((val16 & PcieValue_DeviceControlMaxReadRequestSize) >> 12);
    }
    else
    {
        *pRetMaxSize = 0;
    }
    return result;
}

Result Function::DoPcieFunctionLevelReset()
{
    uint16_t statusWordVal = 0;
    uint32_t capDWordVal = 0;
    Result result;

    if ((result = ReadPCIeCapabilityDWord(PcieOffset_DeviceCapability, &capDWordVal)).IsSuccess() &&
        (result = ReadPCIeCapabilityWord(PcieOffset_DeviceStatus, &statusWordVal)).IsSuccess())
    {
        if(capDWordVal & PcieValue_DeviceCapabilityFunctionLevelReset)
        {
            // Wait for transactions to drain, although we will proceed even if they do not
            if(WaitForClearedWordMask(m_ExpCapOffset + PcieOffset_DeviceStatus, PcieValue_DeviceStatusTransactionPending, 250).IsFailure())
            {
                NN_PCIE_LOG_INFO("Function::DoPcieFunctionLevelReset() - transactions did not quiesce before reset.\n");
            }

            // Issue reset
            uint16_t ctrlWordVal = 0;
            ReadPCIeCapabilityWord(PcieOffset_DeviceControl, &ctrlWordVal);
            capDWordVal |= PcieValue_DeviceControlFunctionLevelReset;
            WritePCIeCapabilityWord(PcieOffset_DeviceControl, ctrlWordVal);

            // Wait for function to come back to life
            result = DoFunctionLevelResetWait();
        }
        else
        {
            result = ResultNotAvailable();
            NN_PCIE_LOG_WARN("Function::DoPcieFunctionLevelReset() - this device is not capable. capability=0x%08x.\n", capDWordVal);
        }
    }

    return result;
}

Result Function::DoAdvancedFeatureFunctionLevelReset()
{
    int32_t advancedFeatureOffset;
    Result result = ResultSuccess();

    if ((advancedFeatureOffset = FindCapability(CapabilityId_AdvancedFeatures)) != 0)
    {
        uint8_t capVal = 0;
        m_pBus->ReadConfigByte(GetDevNum(), GetFuncNum(), advancedFeatureOffset + AdvancedFeaturesOffset_Cap, &capVal);
        if((capVal & (AdvancedFeaturesValue_CapTp | AdvancedFeaturesValue_CapFlr)) == (AdvancedFeaturesValue_CapTp | AdvancedFeaturesValue_CapFlr))
        {
            // Wait for transactions to drain, although we will proceed even if they do not
            if(WaitForClearedWordMask(advancedFeatureOffset + AdvancedFeaturesOffset_Status, AdvancedFeaturesValue_StatusTp, 250).IsFailure())
            {
                NN_PCIE_LOG_INFO("Function::DoAdvancedFeatureFunctionLevelReset() - transactions did not quiesce before reset.\n");
            }

            // Issue reset
            m_pBus->WriteConfigByte(GetDevNum(), GetFuncNum(), advancedFeatureOffset + AdvancedFeaturesOffset_Ctrl, AdvancedFeaturesValue_CtrlFlr);

            // Wait for function to come back to life
            result = DoFunctionLevelResetWait();
        }
        else
        {
            result = ResultNotAvailable();
            NN_PCIE_LOG_WARN("Function::DoAdvancedFeatureFunctionLevelReset() - this device is not capable. capability=0x%02x.\n", capVal);
        }
    }
    else
    {
        result = ResultNotAvailable();
        NN_PCIE_LOG_WARN("Function::DoAdvancedFeatureFunctionLevelReset() - Advanced Features not available.\n");
    }

    return result;
}

Result Function::DoFunctionLevelReset()
{
    Result result = ResultNotAvailable();

    // Try PCIe specific FLR first
    if(IsPcie())
    {
        NN_PCIE_LOG_INFO("Trying PCIe FLR...\n");
        result = DoPcieFunctionLevelReset();
    }

    // Failing that, try legacy FLR
    if(result.IsFailure())
    {
        NN_PCIE_LOG_INFO("Trying PCI AF FLR...\n");
        result = DoAdvancedFeatureFunctionLevelReset();
    }

    return result;
}


Result Function::WaitForClearedWordMask(int32_t where, uint16_t mask, uint32_t maxWaitTimeInMilliseconds)
{
    Result result = ResultOperationTimeout();
    const int IterationSleepInMs = 10;
    int iterationCount = 0;

    do
    {
        uint16_t regVal16 = 0;
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), where, &regVal16);
        if(!(regVal16 & mask))
        {
            result = ResultSuccess();
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(IterationSleepInMs));
    }while(static_cast<uint32_t>(iterationCount++ * IterationSleepInMs) < maxWaitTimeInMilliseconds);

    return result;
}

void* Function::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "Function");
}
void Function::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "Function");
}

// Static method
ClassCode Function::Get16BitClassCode(FullClassCode classCode)
{
    return static_cast<ClassCode>((classCode & 0x00ffff00) >> 8);
}

// Static method
Result Function::ReadProfile(Bus *pBus, DeviceNumber devNum, FunctionNumber funcNum,
                             Profile *pProf, uint32_t timeoutInMilliseconds)
{
    Result result;
    uint32_t vidDWord = 0;
    bool giveUp = false;
    memset(pProf, 0, sizeof(Profile));
    do
    {
        uint32_t waitingTime = 0;
        if ((result = pBus->ReadConfigDWord(devNum, funcNum, StandardConfigOffset_VendorId, &vidDWord)).IsSuccess())
        {
            result = ResultNoDevice();
            switch (vidDWord)
            {
            case 0xffff0000:
            case 0x00000000:
            case 0xffffffff:
            case 0x0000ffff:
                /* clearly no device */
                giveUp = true;
                break;
            case 0xffff0001:
                /* keep trying */
                if (++waitingTime < timeoutInMilliseconds)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                }
                else
                {
                    giveUp = true;
                }
                break;
            default:
                /* OK */
                result = ResultSuccess();
                break;
            }
        }
        else
        {
            giveUp = true;
            break;
        }
    }while (!giveUp && !result.IsSuccess());

    if (result.IsSuccess())
    {
        uint32_t classRev = 0;
        uint8_t headerType;
        if ((result = pBus->ReadConfigDWord(devNum, funcNum, StandardConfigOffset_ClassRevision, &classRev)).IsSuccess() &&
            (result = pBus->ReadConfigByte(devNum, funcNum, StandardConfigOffset_HeaderType, &headerType)).IsSuccess())
        {
            pProf->vendorId   = vidDWord & 0xffff;
            pProf->deviceId   = vidDWord >> 16;
            pProf->classCode  = static_cast<FullClassCode>(classRev >> 8);
            pProf->headerType = headerType;
            pProf->revision   = classRev & 0xff;
            pProf->state      = PmState_Invalid;
        }
    }

    return result;
}




void Function::SetFunctionType(FunctionType functionType)
{
    m_FunctionType = functionType;

}

RegionFlagMask Function::DecodeFlags(uint32_t rawBarVal)
{
    RegionFlagMask regionFlags;
    if ((rawBarVal & StandardConfigValue_BaseAddressSpace) == StandardConfigValue_BaseAddressSpaceIo)
    {
        regionFlags = (rawBarVal & ~StandardConfigValue_IoMasked) | ResourceFlag_Io;
        return regionFlags;
    }
    else
    {
        regionFlags = (rawBarVal & ~StandardConfigValue_MemMasked) | ResourceFlag_Mem;

        if (rawBarVal & StandardConfigValue_BaseAddressMemPrefetch)
        {
            regionFlags |= ResourceFlag_Prefetch;
        }

        switch (StandardConfigValue_BaseAddressMemTypeMask & rawBarVal)
        {
        default:
        case StandardConfigValue_BaseAddressMemType1M:
        case StandardConfigValue_BaseAddressMemType32:
            // assumed to be 32-bit MEM BAR
            break;
        case StandardConfigValue_BaseAddressMemType64:
            regionFlags |= ResourceFlag_Mem64;
            break;
        }
    }
    return regionFlags;
}

void Function::DetectResources()
{
    DeviceNumber devNum = GetDevNum();
    m_NumResources = (m_HeaderType == StandardConfigValue_HeaderTypeNormal) ?
        Type0ConfigMaxBaseAddresses : Type1ConfigMaxBaseAddresses;

    // for all BARs
    for (int32_t resIndex = 0; resIndex < m_NumResources; resIndex++)
    {
        uint32_t barVal = 0;
        int32_t where = StandardConfigOffset_BaseAddress0 + resIndex * 4;
        Resource *pRes = m_Resources + resIndex;
        pRes->pFunction = this;
        m_pBus->WriteConfigDWord(devNum, m_FuncNum, where, 0xffffffff);
        m_pBus->ReadConfigDWord(devNum, m_FuncNum, where, &barVal);
        m_pBus->WriteConfigDWord(devNum, m_FuncNum, where, 0);
        pRes->flags = DecodeFlags(barVal);
        if (pRes->flags & ResourceFlag_Mem64)
        {
            m_pBus->WriteConfigDWord(devNum, m_FuncNum, where + 4, 0);
            resIndex++;
        }
        if (pRes->flags & ResourceFlag_Io)
        {
            barVal &= StandardConfigValue_IoMasked;
        }
        else
        {
            barVal &= StandardConfigValue_MemMasked;
        }
        pRes->size = (barVal & ~(barVal - 1));

        // register resource with bus
        if (pRes->size != 0)
        {
            m_pBus->RegisterResource(pRes);
        }
    }
}

void Function::DetectAttributes()
{
    int32_t offset;

    // This this PCIe?
    m_IsPCIe = false;
    if ((offset = FindCapability(CapabilityId_Pcie)) != 0)
    {
        uint16_t regVal16;
        m_IsPCIe = true;
        m_ExpCapOffset = offset;
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), offset + PcieOffset_Flags, &m_ExpFlags);
        m_pBus->ReadConfigWord(GetDevNum(), GetFuncNum(), offset + PcieOffset_DeviceCapability, &regVal16);
        m_DevCapPayload = regVal16 & PcieValue_DeviceCapabilityPayload;
    }

    // Resolve config space size
    m_ConfigSpaceSize = ReadConfigSpaceSize();

}

Result Function::InitSpeed()
{
    Result result = ResultSuccess();

    do
    {
        BusSpeed devSpeedCapability, devSpeedCurrent;
        BusSpeed bridgeSpeedCapability, bridgeSpeedCurrent, speedLimitPolicy;
        char classCodeDesc[64];
        BridgeFunction *pBridgeFunction = m_pBus->GetUpStreamBridgeFunction();

        if (pBridgeFunction == NULL)
        {
            // Not applicable, exit quietly, this is root bridge function
            break;
        }

        GetClassCodeDescription(classCodeDesc, sizeof(classCodeDesc));

        // Get upstream bridge capability and speed
        NN_PCIE_BREAK_UPON_ERROR(pBridgeFunction->ReadSpeed(&bridgeSpeedCapability, &bridgeSpeedCurrent));

        // Get policy
        speedLimitPolicy = m_pRootComplex->GetBusSpeedLimitPolicy();

        // Get device speed parameters
        NN_PCIE_BREAK_UPON_ERROR(ReadSpeed(&devSpeedCapability, &devSpeedCurrent));

        // Speed up the link?
        if (((bridgeSpeedCapability >= BusSpeed_EGen2) && (devSpeedCapability >= BusSpeed_EGen2)) &&
            ((bridgeSpeedCurrent < BusSpeed_EGen2) || (devSpeedCurrent < BusSpeed_EGen2)) &&
            (speedLimitPolicy >= BusSpeed_EGen2))
        {
            NN_PCIE_LOG_INFO("Function::InitSpeed() BDF-%03x:%02x:%x Class-%s, Increasing to GEN2 bus speed...\n",
                             GetBus()->GetBusNum(), GetDevNum(), GetFuncNum(), classCodeDesc);
            pBridgeFunction->SetLinkSpeed(BusSpeed_EGen2);
        }
        // Slow down the link?
        else if (((bridgeSpeedCurrent >= BusSpeed_EGen2) || (devSpeedCurrent >= BusSpeed_EGen2)) &&
                ((speedLimitPolicy < BusSpeed_EGen2)))
        {
            NN_PCIE_LOG_INFO("Function::InitSpeed() BDF-%03x:%02x:%x Class-%s, Decreasing to GEN1 bus speed...\n",
                             GetBus()->GetBusNum(), GetDevNum(), GetFuncNum(), classCodeDesc);
            pBridgeFunction->SetLinkSpeed(BusSpeed_EGen1);
        }

    }while (false);

    if (result.IsSuccess())
    {
        result = ReadSpeed(&m_CapableSpeed, &m_CurrentSpeed);
    }

    return result;
}

int32_t Function::ReadExtConfigSpaceSize()
{
    int32_t configSpaceSize = StandardConfigSpaceSize;
    uint32_t val32;
    if (m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, StandardConfigSpaceSize, &val32).IsSuccess())
    {
        if (val32 != 0xffffffff)
        {
            configSpaceSize = StandardConfigSpaceExpressSize;
        }
    }
    return configSpaceSize;
}

int32_t Function::ReadConfigSpaceSize()
{
    int32_t configSpaceSize = StandardConfigSpaceSize;
    if ((Get16BitClassCode() == BridgeHostClassCode) || m_IsPCIe)
    {
        uint32_t val32;
        if (m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, StandardConfigSpaceSize, &val32).IsSuccess())
        {
            if (val32 != 0xffffffff)
            {
                configSpaceSize = StandardConfigSpaceExpressSize;
            }
        }
    }
    else
    {
        int32_t offset;
        if ((offset = FindCapability(CapabilityId_Pcix)) != 0)
        {
            uint32_t status;
            if (m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, offset + PciXType0Offset_Status, &status).IsSuccess())
            {
                if (status & (PciXType0Value_Status266Mhz | PciXType0Value_Status533Mhz))
                {
                    configSpaceSize = ReadExtConfigSpaceSize();
                }
            }
        }
    }
    return configSpaceSize;
}

bool Function::IsPCIeCapabilityImplemented(int32_t where)
{
    bool bImpl = false;
    uint8_t fltype = GetPCIeFlagsType();
    if (m_IsPCIe)
    {
        switch (where)
        {
        case PcieOffset_Flags:
            bImpl = true;
            break;
        case PcieOffset_DeviceCapability:
        case PcieOffset_DeviceControl:
        case PcieOffset_DeviceStatus:
            bImpl = true;
            break;
        case PcieOffset_LinkCapability:
        case PcieOffset_LinkControl:
        case PcieOffset_LinkStatus:
            bImpl = (fltype == PcieValue_TypeEndpoint       || fltype == PcieValue_TypeLegacyEndpoint ||
                     fltype == PcieValue_TypeRootPort       || fltype == PcieValue_TypeUpstreamPort   ||
                     fltype == PcieValue_TypeDownstreamPort || fltype == PcieValue_TypePciBridge      ||
                     fltype == PcieValue_TypePcieBridge);
            break;
        case PcieOffset_SlotCapability:
        case PcieOffset_SlotControl:
        case PcieOffset_SlotStatus:
            bImpl = ((fltype == PcieValue_TypeRootPort || fltype == PcieValue_TypeDownstreamPort) &&
                     (m_ExpFlags & PcieValue_FlagsSlot));
            break;
        case PcieOffset_RootControl:
        case PcieOffset_RootCapabilities:
        case PcieOffset_RootStatus:
            bImpl = (fltype == PcieValue_TypeRootPort) || (fltype == PcieValue_TypeRootComplexEventCollector);
            break;
        case PcieOffset_DeviceCapability2:
        case PcieOffset_DeviceControl2:
        case PcieOffset_LinkCapability2:
        case PcieOffset_LinkControl2:
        case PcieOffset_LinkStatus2:
            bImpl = ((m_ExpFlags & PcieValue_FlagsVersion) > 1) ? true : false;
            break;
        default:
            bImpl = false;
            break;
        }
    }
    return bImpl;
}

uint8_t Function::FindCapabilityStart()
{
    uint16_t status;
    uint8_t offset = 0;
    if (m_pBus->ReadConfigWord(GetDevNum(), m_FuncNum, StandardConfigOffset_Status, &status).IsSuccess())
    {
        if (status & StandardConfigValue_StatusCapabilityList)
        {
            switch (GetHeaderType())
            {
            case StandardConfigValue_HeaderTypeNormal:
            case StandardConfigValue_HeaderTypeBridge:
                offset = Type0ConfigOffset_CapabilityList;
                break;
            case StandardConfigValue_HeaderTypeCardbus:
                offset = Type2ConfigOffset_CapabilityList;
                break;
            default:
                offset = 0;
                break;
            }
        }
    }
    return offset;
}

int32_t Function::FindNextCapability(uint8_t startingOffset, CapabilityId cap)
{
    uint8_t iteratedOffset = startingOffset;
    int32_t iterationLimit = CapabilityIdLimit;
    while (iterationLimit--)
    {
        uint8_t capRead = 0;
        m_pBus->ReadConfigByte(GetDevNum(), m_FuncNum, iteratedOffset, &iteratedOffset);
        if (iteratedOffset < 0x40)
        {
            break;
        }
        iteratedOffset &= ~3;
        m_pBus->ReadConfigByte(GetDevNum(), m_FuncNum, iteratedOffset + CapabilityId_List, &capRead);
        if (capRead == 0xff)
        {
            iteratedOffset = 0;
            break;
        }
        if (capRead == static_cast<uint8_t>(cap))
        {
            return (int32_t)iteratedOffset;
        }

        // continue search
        iteratedOffset++;
    }
    return 0;
}

int32_t Function::FindNextExtendedCapability(int32_t start, ExtendedCapabilityId extCap)
{
#define EXT_CAP_ID(header)      (header& 0x0000ffff)
#define EXT_CAP_NEXT(header)    ((header >> 20)& 0xffc)
    int32_t offset = StandardConfigSpaceSize;
    uint32_t header;
    int32_t iterationLimit = (StandardConfigSpaceExpressSize - StandardConfigSpaceSize) / 8;
    if (m_ConfigSpaceSize <= static_cast<int32_t>(StandardConfigSpaceSize))
    {
        return 0;
    }
    if (start != 0)
    {
        offset = start;
    }
    if (m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, offset, &header).IsSuccess())
    {
        if (header == 0)
        {
            return 0;
        }
    }
    else
    {
        return 0;
    }
    while (iterationLimit-- > 0)
    {
        if ((EXT_CAP_ID(header) == static_cast<uint32_t>(extCap)) && (offset != start))
        {
            return offset;
        }
        offset = EXT_CAP_NEXT(header);
        if (offset < static_cast<int32_t>(StandardConfigSpaceSize))
        {
            break;
        }
        if (!m_pBus->ReadConfigDWord(GetDevNum(), m_FuncNum, offset, &header).IsSuccess())
        {
            break;
        }
    }
    return 0;
}


} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
