﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>

#include "pcie_PrivateIncludes.h"

#if defined(NN_PCIE_TEGRA_21X_SOC)
#include "tegra/pcie_Tegra21xRootComplex.h"
#endif

#if defined(NN_PCIE_TEGRA_124_SOC)
#include "tegra/pcie_Tegra124RootComplex.h"
#endif

namespace nn { namespace pcie { namespace driver { namespace detail {

Driver::Driver()
    : m_AccessMutex(true)
    , m_IntThreadRun(false)
    , m_pRootComplex(nullptr)
    , m_AllocatedBusNum(0)
    , m_AllocatedIoAddr(0)
    , m_IoAddrLimit(0)
    , m_AllocatedNoPrefMemAddr(0)
    , m_NoPrefMemAddrLimit(0)
    , m_AllocatedPrefMemAddr(0)
    , m_PrefMemAddrLimit(0)
    , m_IsWifiResetManaged(true)
    , m_IsLoggedStateStatisticsRateLimitActive(false)
    , m_IsLoggedStateErrorRateLimitActive(false)
    , m_LoggedStateStatisticsPendingCount(0)
    , m_LoggedStateErrorPendingCount(0)
    , m_LoggedStateCallback(nullptr)
    , m_LoggedStateContext(0)
{
    memset(m_BusTable, 0, sizeof(m_BusTable));
}

Driver::~Driver()
{

}

Result Driver::Initialize()
{
    Result result = ResultSuccess();

    // PM dependency configuration
    const nn::psc::PmModuleId pcieDependencies[] = {
        nn::psc::PmModuleId_PcvClock,
        nn::psc::PmModuleId_PcvVoltage,
        nn::psc::PmModuleId_Gpio,
    };

    NN_PCIE_ABORT_UPON_ERROR(
        m_PmModule.Initialize(nn::psc::PmModuleId_Pcie, pcieDependencies,
                              sizeof(pcieDependencies) / sizeof(pcieDependencies[0]),
                              nn::os::EventClearMode_ManualClear)
    );

    // Init multiwait which synchronizes interrupt thread
    nn::os::InitializeMultiWait(&m_MultiWait);

    // Init re-evaluation event and holder
    nn::os::CreateSystemEvent(&m_ReEvaluateEvent, nn::os::EventClearMode_ManualClear, false);
    nn::os::InitializeMultiWaitHolder(&m_ReEvaluateHolder, &m_ReEvaluateEvent);
    nn::os::SetMultiWaitHolderUserData(&m_ReEvaluateHolder, static_cast<uintptr_t>(IntThreadMultiWait_ReEvaluate));
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_ReEvaluateHolder);

    // Init PM event and holder
    nn::os::InitializeMultiWaitHolder(&m_PmModuleHolder, m_PmModule.GetEventPointer()->GetBase());
    nn::os::SetMultiWaitHolderUserData(&m_PmModuleHolder, static_cast<uintptr_t>(IntThreadMultiWait_Pm));
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_PmModuleHolder);

    // Init local event manager timer
    m_LocalEventManager.Initialize(&m_LocalEventManagerTimerHolder);
    nn::os::SetMultiWaitHolderUserData(&m_LocalEventManagerTimerHolder, static_cast<uintptr_t>(IntThreadMultiWait_LocalEventTimer));
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_LocalEventManagerTimerHolder);


    NN_PCIE_ABORT_UPON_ERROR(nn::os::CreateThread(&m_IntThread,
                                                  InterruptThreadEntry,
                                                  this, m_IntThreadStack,
                                                  sizeof(m_IntThreadStack),
                                                  NN_SYSTEM_THREAD_PRIORITY(pcie, InterruptHandler)));
    nn::os::SetThreadNamePointer(&m_IntThread, NN_SYSTEM_THREAD_NAME(pcie, InterruptHandler));

    NN_PCIE_RETURN_UPON_ERROR(
        InitializeRootComplex()
    );

    // Resume root complex by default, since PSC does not issue "awake" command upon startup
    NN_PCIE_RETURN_UPON_ERROR(
        Resume()
    );

    // Start interrupt thread
    m_IntThreadRun = true;
    nn::os::StartThread(&m_IntThread);

    return result;
}

Result Driver::Finalize()
{
    Result result = ResultSuccess();

    // Stop and destroy root complex
    NN_PCIE_RETURN_UPON_ERROR(
        Suspend()
    );

    NN_PCIE_RETURN_UPON_ERROR(
        FinalizeRootComplex()
    );

    // Initiate interrupt thread exit
    m_IntThreadRun = false;
    nn::os::SignalSystemEvent(&m_ReEvaluateEvent);

    // Take atomic control over driver
    m_AccessMutex.Lock();

    // Synchronize with interrupt thread exit
    nn::os::WaitThread(&m_IntThread);
    nn::os::DestroyThread(&m_IntThread);

    m_LoggedStateCallback = nullptr;
    m_LoggedStateContext = 0;

    return result;
}

Result Driver::RegisterInterruptEvent(nn::os::InterruptEventType *event, IntVector intVector)
{
    Result result = ResultSuccess();
    if ((intVector < IntVector_MaxTypes) && (intVector >= 0))
    {
        m_InterruptEvents[intVector] = event;
        nn::os::InitializeMultiWaitHolder(&m_InterruptHolders[intVector], event);
        nn::os::SetMultiWaitHolderUserData(&m_InterruptHolders[intVector], intVector);
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_InterruptHolders[intVector]);
    }
    else
    {
        result = ResultInvalidHandle();
    }
    return result;
}

Result Driver::UnregisterInterruptEvent(IntVector intVector)
{
    Result result = ResultSuccess();
    if ((intVector < IntVector_MaxTypes) && (intVector >= 0))
    {
        m_InterruptEvents[intVector] = NULL;
        nn::os::SetMultiWaitHolderUserData(&m_InterruptHolders[intVector], static_cast<uintptr_t>(-1));
        nn::os::UnlinkMultiWaitHolder(&m_InterruptHolders[intVector]);
    }
    else
    {
        result = ResultInvalidHandle();
    }
    return result;
}

Result Driver::GetBusAddressess(ResourceAddr resourceAddr, BusAddress *pRetBusAddress)
{
    size_t offset = 0;
    BusAddress base = 0;
    Result result = ResultInvalidResourceAddress();

    // Find region
    if ((resourceAddr >= m_IoRegion.resourceAddr) &&
        (resourceAddr < (m_IoRegion.resourceAddr + m_IoRegion.size)))
    {
        offset = (size_t)(resourceAddr - m_IoRegion.resourceAddr);
        base = m_IoRegion.busAddr;
        result = ResultSuccess();
    }
    else if ((resourceAddr >= m_PrefMemRegion.resourceAddr) &&
             (resourceAddr < (m_PrefMemRegion.resourceAddr + m_PrefMemRegion.size)))
    {
        offset = (size_t)(resourceAddr - m_PrefMemRegion.resourceAddr);
        base = m_PrefMemRegion.busAddr;
        result = ResultSuccess();
    }
    else if ((resourceAddr >= m_NoPrefMemRegion.resourceAddr) &&
             (resourceAddr < (m_NoPrefMemRegion.resourceAddr + m_NoPrefMemRegion.size)))
    {
        offset = (size_t)(resourceAddr - m_NoPrefMemRegion.resourceAddr);
        base = m_NoPrefMemRegion.busAddr;
        result = ResultSuccess();
    }

    *pRetBusAddress = (result.IsSuccess()) ? (base + offset) : 0;

    return result;
}

Result Driver::GetRootComplex(RootComplex** ppRootComplex)
{
    Result result = ResultSuccess();

    if(m_pRootComplex != NULL)
    {
        *ppRootComplex = m_pRootComplex;
    }
    else
    {
        result = ResultNoController();
    }

    return result;
}

void Driver::RequestLoggedStateUpdate(bool isDueToError)
{
    do
    {
        nn:: TimeSpan rateLimitPeriod;

        // Record the report
        if(isDueToError)
        {
            m_LoggedStateErrorPendingCount++;
        }
        else
        {
            m_LoggedStateStatisticsPendingCount++;
        }

        // If the shorter rate limit, errors, is active, stop
        if(m_IsLoggedStateErrorRateLimitActive)
        {
            break;
        }

        // If this is a statistics update and the statistics update is active, stop
        if(m_IsLoggedStateStatisticsRateLimitActive && !isDueToError)
        {
            break;
        }

        // Determine which rate limit we will engage
        if(isDueToError)
        {
            rateLimitPeriod = nn::TimeSpan::FromMilliSeconds((1000 * 1000) / LoggedStateReportingErrorUpdateRateInMilliHertz);
            m_IsLoggedStateErrorRateLimitActive = true;
        }
        else
        {
            rateLimitPeriod = nn::TimeSpan::FromMilliSeconds((1000 * 1000) / LoggedStateReportingStatisticsUpdateRateInMilliHertz);
            m_IsLoggedStateStatisticsRateLimitActive = true;
        }

        // Do the callback
        DoLoggedStateCallback(isDueToError);

        // Start the rate limit timer
        m_LocalEventManager.RestartTimedEvent(&m_LoggedStateRateLimitTimer, rateLimitPeriod);

    }while NN_STATIC_CONDITION(false);
}

ResourceAddr Driver::AlignResourceAddressWithBus(ResourceAddr inputAddress, size_t powerOfTwoRoundedSize)
{
    Result result = ResultSuccess();
    BusAddress inputBusAddressess, alignment, alignedBusAddressess;
    ResourceAddr alignedAddress;
    size_t sizeDelta;

    NN_PCIE_ABORT_UPON_ERROR(GetBusAddressess(inputAddress, &inputBusAddressess));
    alignment = static_cast<BusAddress>(powerOfTwoRoundedSize);
    alignedBusAddressess = (inputBusAddressess + alignment - 1) & ~(alignment - 1);
    sizeDelta = static_cast<size_t>(alignedBusAddressess - inputBusAddressess);
    alignedAddress = inputAddress + sizeDelta;

    return alignedAddress;
}

Result Driver::InitializeRootComplex()
{
    Result result = ResultSuccess();

    m_pRootComplex = nullptr;

    // create root complex
#ifdef NN_PCIE_TEGRA_124_SOC
    RootComplex::Config config = NN_PCI_TEGRA124_ROOTCOMPLEX_CFG;
    m_pRootComplex = new Tegra124RootComplex(this, &config);
#endif
#ifdef NN_PCIE_TEGRA_21X_SOC
    RootComplex::Config config = Tegra21xRootComplexConfiguration;
    m_pRootComplex = new Tegra21xRootComplex(this, &config);
#endif

    if (m_pRootComplex != NULL)
    {
        result = m_pRootComplex->Initialize();

        if(result.IsFailure())
        {
            delete m_pRootComplex;
            m_pRootComplex = nullptr;
        }
    }
    else
    {
        result = ResultNoController();
    }

    return result;
}

Result Driver::FinalizeRootComplex()
{
    Result result = ResultSuccess();
    if(m_pRootComplex != nullptr)
    {
        result = m_pRootComplex->Finalize();
        if(result.IsSuccess())
        {
            delete m_pRootComplex;
            m_pRootComplex = nullptr;
        }

    }
    return result;
}

Result Driver::Resume()
{
    Result result = ResultSuccess();

    if(m_pRootComplex == nullptr)
    {
        NN_PCIE_LOG_INFO("InitializeRootComplex() cannot proceed, no controller.\n");
        result = ResultNoController();
        return result;
    }

    if(!m_pRootComplex->IsResumed())
    {
        // (re)Initialize data
        memset(m_BusTable, 0, sizeof(m_BusTable));

        // Register local event pool
        m_LocalEventManager.RegisterPool(this, &m_LocalEventPool, HandleLocalEventStatic,
                                         &m_LocalEventStorage[0], sizeof(m_LocalEventStorage[0]),
                                         NN_PCIE_ARRAY_SIZE(m_LocalEventStorage));

        // Initialize enumeration timer
        m_LocalEventManager.InitStaticEvent(&m_LocalEventPool, &m_EnumerationTimer);
        m_EnumerationTimer.data.eventId = LocalEventId_DoEnumeration;

        // Initialize periodic monitoring timer
        m_LocalEventManager.InitStaticEvent(&m_LocalEventPool, &m_LoggedStateRateLimitTimer);
        m_LoggedStateRateLimitTimer.data.eventId = LocalEventId_LoggedStateRateLimit;

        // reset WiFi
        if(m_IsWifiResetManaged)
        {
            ResetWiFiDevice(false);
        }

        // initialize root complex
        if ((result = m_pRootComplex->Resume()).IsSuccess())
        {
            Bus* pRootBus = NULL;

            // Get IO, prefetchable MEM and non-prefetchable regions
            m_pRootComplex->GetRegions(&m_IoRegion, &m_PrefMemRegion, &m_NoPrefMemRegion);
            m_AllocatedIoAddr         = m_IoRegion.resourceAddr;
            m_IoAddrLimit             = m_IoRegion.resourceAddr + m_IoRegion.size - 1;
            m_AllocatedPrefMemAddr    = m_PrefMemRegion.resourceAddr;
            m_PrefMemAddrLimit        = m_PrefMemRegion.resourceAddr + m_PrefMemRegion.size - 1;
            m_AllocatedNoPrefMemAddr  = m_NoPrefMemRegion.resourceAddr;
            m_NoPrefMemAddrLimit      = m_NoPrefMemRegion.resourceAddr + m_NoPrefMemRegion.size - 1;
            pRootBus                  = m_pRootComplex->GetBus();
            NN_PCIE_ABORT_IF_NULL(pRootBus);
            m_AllocatedBusNum         = pRootBus->GetBusNum() + 1;

            // Section 6.6.1 of the PCIe Base Specification 2.1 calls for a minimum of 100ms after reset is released before
            // any configuration space accesses are made.
            m_LocalEventManager.StartTimedEvent(&m_EnumerationTimer, nn::TimeSpan::FromMilliSeconds(100));
        }
        else
        {
            NN_PCIE_LOG_ERROR("RootComplex.Resume() failed\n");
        }
    }

    return result;
}

Result Driver::Suspend()
{
    Result result = ResultSuccess();

    // Disable and destroy root complex
    if (m_pRootComplex != NULL)
    {
        if(m_pRootComplex->IsResumed())
        {
            result = m_pRootComplex->Suspend();

            // Make sure enumeration timer is not running
            m_LocalEventManager.StopTimedEvent(&m_EnumerationTimer);

            // Make sure logged state timer is not running
            m_LocalEventManager.StopTimedEvent(&m_LoggedStateRateLimitTimer);

            // Unregister local event pool
            m_LocalEventManager.UnRegisterPool(&m_LocalEventPool);

            // Reset logged state rate limit mechanism
            m_IsLoggedStateStatisticsRateLimitActive = false;
            m_IsLoggedStateErrorRateLimitActive = false;
            m_LoggedStateStatisticsPendingCount = 0;
            m_LoggedStateErrorPendingCount = 0;
        }
    }
    else
    {
        result = ResultNoController();
    }

    if(m_IsWifiResetManaged)
    {
        ShutdownWiFiDevice();
    }

    return result;
}

void Driver::EnumerateDepthFirst(Bus *pBus, BusNumber *pAllocBusNum, uint64_t deviceMask)
{
    Result result = ResultSuccess();


    // For all potential devices on this bus...
    for (DeviceNumber devNum = 0; devNum < MaxDeviceCount; devNum++)
    {
        // Has this device been selected for enumeration? If not, skip it.
        if(!(deviceMask & (1ULL << devNum))) continue;

        // Maybe this device has already been enumerated?
        Device *pDevice = pBus->GetAttachedDevice(devNum);

        // For all potential functions in this device...
        for (FunctionNumber funcNum = 0; funcNum < MaxDeviceFunctionCount; funcNum++)
        {
            Function::Profile profile;

            //NN_PCIE_LOG_INFO("Scanning BDF %d-%d-%d\n",pBus->GetBusNum(),devNum,funcNum);
            if ((result = Function::ReadProfile(pBus, devNum, funcNum, &profile, 5000)).IsSuccess())
            {
                if (profile.headerType == StandardConfigValue_HeaderTypeBridge &&
                    Function::Get16BitClassCode(profile.classCode) == PciBridgeClassCode)
                {
                    // Create new bridge device if not already available
                    BridgeDevice *pBridgeDevice = NULL;
                    if (pDevice == NULL)
                    {
                        pBridgeDevice = new BridgeDevice(this, devNum, pBus);
                        NN_PCIE_ABORT_IF_NULL(pBridgeDevice);
                        NN_PCIE_ABORT_UPON_ERROR(pBridgeDevice->Initialize());
                        pDevice = pBridgeDevice;
                    }
                    else
                    {
                        pBridgeDevice = static_cast<BridgeDevice *>(pDevice);
                    }

                    // create new bridge function if not already available
                    BridgeFunction* pBridgeFunction = static_cast<BridgeFunction*>(pBridgeDevice->GetFunction(funcNum));
                    if(pBridgeFunction == NULL)
                    {
                        pBridgeFunction = new BridgeFunction(this, pBridgeDevice, &profile, pBus, *pAllocBusNum);
                        NN_PCIE_ABORT_IF_NULL(pBridgeFunction);
                        NN_PCIE_ABORT_UPON_ERROR(pBridgeFunction->Initialize());

                        // the creation of the bridge consumed a bus number
                        *pAllocBusNum = *pAllocBusNum + 1;
                    }

                    // compute address space base
                    int spaceShift = FindFirstSetBit(MaxSwitchPortCount) - 1;
                    BusNumber busNumber      = pBridgeFunction->GetSecondaryBusNumber();
                    m_AllocatedIoAddr        = m_IoRegion.resourceAddr + ((m_IoRegion.size >> spaceShift) * (busNumber - 1));
                    m_AllocatedNoPrefMemAddr = m_NoPrefMemRegion.resourceAddr + ((m_NoPrefMemRegion.size >> spaceShift) * (busNumber - 1));
                    m_AllocatedPrefMemAddr   = m_PrefMemRegion.resourceAddr + ((m_PrefMemRegion.size >> spaceShift) * (busNumber - 1));

                    // enumerate downward
                    EnumerateDepthFirst(pBridgeFunction->GetDownStreamBus(), pAllocBusNum, 0xffffffffffffffff);
                    pBridgeFunction->ConfigureSubordinate(busNumber,
                                                          &m_AllocatedIoAddr, m_IoAddrLimit,
                                                          &m_AllocatedNoPrefMemAddr, m_NoPrefMemAddrLimit,
                                                          &m_AllocatedPrefMemAddr, m_PrefMemAddrLimit);
                    pBridgeFunction->SetDeviceEnable(true);
                }
                else
                {
                    // Create new endpoint device if not already available
                    if (pDevice == NULL)
                    {
                        pDevice = new Device(this, devNum, pBus);
                        NN_PCIE_ABORT_IF_NULL(pDevice);
                        NN_PCIE_ABORT_UPON_ERROR(pDevice->Initialize());
                    }

                    // Create new endpoint function if not already available
                    EndpointFunction* pEndpoint = static_cast<EndpointFunction*>(pDevice->GetFunction(funcNum));
                    if(pEndpoint == NULL)
                    {
                        EndpointFunction *pEndpoint = new EndpointFunction(this, pDevice, &profile);
                        NN_PCIE_ABORT_IF_NULL(pEndpoint);
                        NN_PCIE_ABORT_UPON_ERROR(pEndpoint->Initialize());

                        // Enable ASPM by default
                        if(IsAspmEnabledByDefault)
                        {
                            RootComplex *pRC = nullptr;
                            pEndpoint->GetBus()->GetRootComplex(&pRC);
                            NN_PCIE_ABORT_IF_NULL(pRC);
                            pRC->SetAspmEnable(pEndpoint, true);
                        }
                    }
                }
            }
        }
    }
}

void Driver::HandlePowerManagementEvent()
{
    Result result = ResultSuccess();
    nn::psc::PmState    state;
    nn::psc::PmFlagSet  flags;

    NN_PCIE_ABORT_UPON_ERROR(
        m_PmModule.GetRequest(&state, &flags)
    );

    NN_PCIE_LOG_INFO("Driver::HandlePowerManagementEvent() nn::psc::PmState=%d.\n", state);

    switch (state)
    {
    case nn::psc::PmState_FullAwake:
    case nn::psc::PmState_MinimumAwake:
        result = Resume();
        break;
    case nn::psc::PmState_SleepReady:
        result = Suspend();
        break;
    default:
        break;
    }

    NN_PCIE_ABORT_UPON_ERROR(
        m_PmModule.Acknowledge(state, result)
    );
}

void Driver::InterruptThread()
{
    while (m_IntThreadRun)
    {
        nn::os::MultiWaitHolderType *holder = nn::os::WaitAny(&m_MultiWait);
        IntThreadMultiWait intThreadMultiWait = static_cast<IntThreadMultiWait>(GetMultiWaitHolderUserData(holder));
        std::lock_guard<nn::os::Mutex> lock(m_AccessMutex);
        switch(intThreadMultiWait)
        {
        case IntThreadMultiWait_Pm:
            HandlePowerManagementEvent();
            break;
        case IntThreadMultiWait_ReEvaluate:
            ClearSystemEvent(&m_ReEvaluateEvent);
            break;
        case IntThreadMultiWait_LocalEventTimer:
            m_LocalEventManager.DispatchTimedEvents();

            // move out to always run if local event use is expanded
            m_LocalEventManager.DispatchEvents();
            break;
        default:
            if ((intThreadMultiWait >= IntThreadMultiWait_First) &&
                (intThreadMultiWait < IntThreadMultiWait_LastIntVector))
            {
                IntVector intVector = static_cast<IntVector>(intThreadMultiWait);
                m_pRootComplex->HandleInterrupt(intVector);
                nn::os::ClearInterruptEvent(m_InterruptEvents[intVector]);
            }
            break;
        }
    }
}

void Driver::HandleLocalEvent(LocalEventType *pEvent)
{
    switch(pEvent->data.eventId)
    {
        case LocalEventId_DoEnumeration:
            if(m_pRootComplex->IsResumed())
            {
                NN_PCIE_LOG_INFO("Performing enumeration...\n");

                // Enumerate tree of devices
                EnumerateDepthFirst(m_pRootComplex->GetBus(), &m_AllocatedBusNum, 0xffffffffffffffff);

                // Now that probing is done, enable error reporting
                m_pRootComplex->SetErrorReportingEnable(true);

                NN_PCIE_LOG_INFO("Enumeration done.\n");
            }
            break;
        case LocalEventId_LoggedStateRateLimit:
            {
                if(m_LoggedStateStatisticsPendingCount > 0 || m_LoggedStateErrorPendingCount > 0)
                {
                    DoLoggedStateCallback(m_LoggedStateErrorPendingCount > 0);
                    m_LoggedStateStatisticsPendingCount = m_LoggedStateErrorPendingCount = 0;
                }
                m_IsLoggedStateStatisticsRateLimitActive = m_IsLoggedStateErrorRateLimitActive = false;
            }
            break;
        default:
            break;
    }
}

void Driver::DoLoggedStateCallback(bool isDueToError)
{
    if(m_LoggedStateCallback != nullptr)
    {
        (m_LoggedStateCallback)(m_LoggedStateContext, isDueToError);
    }
}

void Driver::PmStateCallout(const FunctionState* pFs)
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // Is this the BCM device on the second root port?
    if((pFs->busNum == WiFiBusNumber) && (pFs->deviceNum == WiFiDeviceNumber) && (pFs->funcNum == 0))
    {
        switch(pFs->pmState)
        {
            case PmState_D3Hot:
                m_IsWifiResetManaged = false;
                break;
            case PmState_D3Off:
                m_IsWifiResetManaged = true;
                break;
            default:
                break;
        }
    }
#else
    NN_UNUSED(pFs);
#endif
}

Result Driver::ResetDevice(BusNumber busNumber, DeviceNumber deviceNumber, FunctionResetOptionMask resetOptionMask)
{
    Result result = ResultSuccess();

    do
    {
        NN_PCIE_LOG_INFO("Resetting Bus %03x Device %02x...\n", busNumber, deviceNumber);

        // Get the bus
        Bus* pBus = Bus::GetBus(this, busNumber);
        if(!pBus)
        {
            NN_PCIE_BREAK_UPON_ERROR(ResultNoDevice());
        }

        // Get the device that we want to reset
        Device* pDevice = pBus->GetAttachedDevice(deviceNumber);
        if(!pDevice)
        {
            NN_PCIE_BREAK_UPON_ERROR(ResultNoDevice());
        }

        // Get the upstream bridge device
        BridgeFunction* pUpstreamBridgeFunction = pBus->GetUpStreamBridgeFunction();
        if(!pUpstreamBridgeFunction)
        {
            NN_PCIE_BREAK_UPON_ERROR(ResultNoDevice());
        }
        BusNumber baseBusNumber = pUpstreamBridgeFunction->GetSecondaryBusNumber();
        BridgeDevice* pUpStreamBridgeDevice = pUpstreamBridgeFunction->GetDevice();
        if(!pUpStreamBridgeDevice)
        {
            NN_PCIE_BREAK_UPON_ERROR(ResultNoDevice());
        }
        DeviceNumber upStreamBridgeDeviceNumber = pUpStreamBridgeDevice->GetDevNum();

        // Get upstream bus
        Bus* pUpStreamBus = pUpstreamBridgeFunction->GetUpStreamBus();
        if(!pUpStreamBus)
        {
            NN_PCIE_BREAK_UPON_ERROR(ResultNoDevice());
        }

        // Destroy the upstream bridge device and its children (including this device being reset)
        NN_PCIE_WARN_UPON_ERROR(pUpStreamBridgeDevice->Finalize());
        delete pUpStreamBridgeDevice;

        // Stop now if re-enumeration not requested
        if(FunctionResetOption_DoNotReEnumerate & resetOptionMask)
        {
            break;
        }

        // For finite number of attempts, try to re-enumerate this device.
        for(int attemptCount = 0; attemptCount < 3; attemptCount++)
        {
            bool isReEnumSuccessful = false;

            // Assert port reset prior to custom reset option
            m_pRootComplex->PortControl(upStreamBridgeDeviceNumber, RootComplex::PortCommand_HoldReset);

            // Handle custom reset option
            if(FunctionResetOption_DoExternalReset & resetOptionMask)
            {
                if ((busNumber == WiFiBusNumber) && (deviceNumber == WiFiDeviceNumber))
                {
                    ResetWiFiDevice(true);
                }
                else
                {
                    NN_PCIE_LOG_WARN("External reset not supported for bus %d device %d.\n", busNumber, deviceNumber);
                }
            }

            // Release port reset
            if((result = m_pRootComplex->PortControl(upStreamBridgeDeviceNumber, RootComplex::PortCommand_CycleReset)).IsSuccess())
            {
                nn::os::Tick startTick = nn::os::GetSystemTick();

                // Section 6.6.1 of the PCIe Base Specification 2.1 calls for a minimum of 100ms after reset is released before
                // any configuration space accesses are made.
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

                // Scan bus for enumeration
                int iterations = 0;
                nn::os::Tick timeoutTick = nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(500)) + startTick;
                do
                {
                    iterations++;

                    // re-enumerate
                    EnumerateDepthFirst(pUpStreamBus, &baseBusNumber, 1ULL << upStreamBridgeDeviceNumber);

                    // determine if successful
                    isReEnumSuccessful = ((pBus = Bus::GetBus(this, busNumber)) != NULL) &&
                        ((pDevice = pBus->GetAttachedDevice(deviceNumber)) != NULL);

                    if(!isReEnumSuccessful)
                    {
                        // backoff, wait for device
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));
                    }
                }while(!isReEnumSuccessful && (nn::os::GetSystemTick() < timeoutTick));

                // Did it re-enumerate?
                if(isReEnumSuccessful)
                {
                    NN_PCIE_LOG_INFO("Bus %03x Device %02x successfully re-enumerated in %dus, %d iterations.\n",
                                     busNumber, deviceNumber, nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick).GetMicroSeconds(),
                                     iterations);
                    break;
                }
            } // if((result = m_pRootComplex->PortControl(...

            if(!isReEnumSuccessful)
            {
                NN_PCIE_LOG_INFO("Attempt #%d did not succeed. Retrying reset sequence for Bus %03x Device %02x...\n",
                                 attemptCount + 1, busNumber, deviceNumber);
            }

        } // for(int attemptCount = 0; ...

    } while NN_STATIC_CONDITION(false);

    return result;
}

void Driver::SetLoggedStateCallback(LoggedStateCallback callback, uintptr_t context)
{
    m_LoggedStateCallback = callback;
    m_LoggedStateContext  = context;
}

void Driver::ResetWiFiDevice(bool isLongResetDuration)
{

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_PCIE_LOG_INFO("Resetting WLAN.\n");
    nn::gpio::GpioPadSession session;
    nn::gpio::Initialize();
    nn::gpio::OpenSession(&session, nn::gpio::GpioPadName_WifiReset);
    nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);
    nn::gpio::SetValue(&session, nn::gpio::GpioValue_Low);  // make the device "reset state"
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds((isLongResetDuration) ? 100 : 10));
    nn::gpio::SetValue(&session, nn::gpio::GpioValue_High);  // make the device "active state"
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    nn::gpio::CloseSession(&session);
    nn::gpio::Finalize();
    NN_PCIE_LOG_INFO("WLAN Reset Done.\n");
#else
    NN_UNUSED(isLongResetDuration);
#endif
}

void Driver::ShutdownWiFiDevice()
{

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_PCIE_LOG_INFO("Shutdowning WLAN.\n");
    nn::gpio::GpioPadSession session;
    nn::gpio::Initialize();
    nn::gpio::OpenSession(&session, nn::gpio::GpioPadName_WifiReset);
    nn::gpio::SetDirection(&session, nn::gpio::Direction_Output);
    nn::gpio::SetValue(&session, nn::gpio::GpioValue_Low);  // make the device "reset state"
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    nn::gpio::CloseSession(&session);
    nn::gpio::Finalize();
    NN_PCIE_LOG_INFO("WLAN Shutdown Done.\n");
#endif
}

} // end of namespace detail
} // end of namespace driver
} // end of namespace pcie
} // end of namespace nn
