﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcie_PrivateIncludes.h"

namespace nn { namespace pcie { namespace driver { namespace detail {


const Client::EndpointFunctionProfile Client::InvalidEndpointFunctionProfile =
{
    {
        InvalidFunctionHandle,      // FunctionHandle
        255,                        // BusNumber
        255,                        // DeviceNumber
        255,                        // FunctionNumber
        0,                          // vendorId
        0,                          // deviceId
        0,                          // headerType
        0,                          // FullClassCode
        0,                          // revision
        BusSpeed_Invalid,           // BusSpeed
        0,                          // maxNumMsiVectors
        PmState_Invalid,            // PmState
        false,                      // isAcquired
    },
    {NULL, 0, IrqType_Invalid, 0},  // IrqParameters
    false,                          // isResetUponResumeEnable
    0,                              // clientContext
};

const Client::IrqParameters Client::InvalidIrqParameters =
{
    nullptr,                        // callback
    0,                              // context
    IrqType_Invalid,                // type
    0,                              // numberOfVectors
};

Client::Client(Driver *pDriver, const ClassDriverConfig *pConfig,
               RegistrationCallback regIndCallback,
               PmStateCallback pmStateCallback)
    : m_pDriver(pDriver)
    , m_Config(*pConfig)
    , m_RegistrationCallback(regIndCallback)
    , m_PmStateCallback(pmStateCallback)
    , m_Handle(nn::pcie::InvalidClassDriverHandle)
{
    for(int32_t i = 0; i < NN_PCIE_ARRAY_COUNT32(m_Functions); i++)
    {
        m_Functions[i] = InvalidEndpointFunctionProfile;
    }
}

Client::~Client() { }

Result Client::Initialize(ClassDriverHandle *pReturnedHandle)
{
    Result result;

    // Assign client handle
    if ((result = m_pDriver->m_ClientHandleManager.Assign(this, &m_Handle)).IsSuccess())
    {
        // Pass back client handle now
        *pReturnedHandle = m_Handle;

        // Add client to global list
        m_pDriver->m_ClientList.push_back(*this);

        // Evaluate client's registration against previously enumerated endpoint functions
        EvaluateRegistration();
    }

    return result;
}

void Client::Finalize()
{
    Result result = ResultSuccess();

    // Release any endpoint functions acquired by this client
    for(int32_t i=0; i < NN_PCIE_ARRAY_COUNT32(m_Functions); i++)
    {
        EndpointFunction* pEpFunc = detail::EndpointFunction::GetEndpointFunction(m_pDriver, m_Functions[i].state.functionHandle);
        if(pEpFunc != nullptr)
        {
            NN_PCIE_ABORT_UPON_ERROR(ReleaseFunction(pEpFunc->GetHandle()));
        }
    }

    // Release client's handle
    result = m_pDriver->m_ClientHandleManager.Release(m_Handle);

    // Remove this client from global list
    m_pDriver->m_ClientList.erase(m_pDriver->m_ClientList.iterator_to(*this));
}

ClassDriverHandle Client::GetHandle()
{
    return m_Handle;
}

Result Client::AcquireFunction(FunctionHandle funcHandle,  uintptr_t clientContext)
{
    Result result = ResultSuccess();
    EndpointFunction *pEpFunc = EndpointFunction::GetEndpointFunction(m_pDriver, funcHandle);
    if (pEpFunc != NULL)
    {
        if (pEpFunc->GetAssignedClient() == nn::pcie::InvalidClassDriverHandle)
        {
            result = ResultMaximumExceeded();

            // Release any endpoint functions acquired by this client
            for(int32_t i=0; i < NN_PCIE_ARRAY_COUNT32(m_Functions); i++)
            {
                // Find an unused slot
                if(m_Functions[i].state.functionHandle == InvalidFunctionHandle)
                {
                    // Push the client profile to the endpoint
                    if((result = pEpFunc->SetAssignedClient(m_Handle)).IsSuccess())
                    {
                        // Use this slot for newly acquired
                        pEpFunc->GetState(&m_Functions[i].state);
                        m_Functions[i].isResetUponResumeEnable = pEpFunc->GetResetUponResumeEnable();
                        m_Functions[i].clientContext           = clientContext;
                        NN_PCIE_LOG_INFO("Client::AcquireFunction() client 0x%x has acquired functionHandle 0x%x.\n",
                                         GetHandle(), funcHandle);
                    }
                    break;
                }
            }
        }
        else
        {
            result = ResultAlreadyAcquired();
        }
    }
    else
    {
        result = ResultInvalidFunctionHandle();
    }
    return result;
}

Result Client::ReleaseFunction(FunctionHandle funcHandle)
{
    Result result = ResultSuccess();
    EndpointFunction *pEpFunc = EndpointFunction::GetEndpointFunction(m_pDriver, funcHandle);
    Client::EndpointFunctionProfile* pProfile = GetEndpointFunctionProfile(funcHandle);
    if (pEpFunc != NULL)
    {
        if (pEpFunc->GetAssignedClient() == m_Handle)
        {
            // Push the cleared client profile to the endpoint
            pEpFunc->SetAssignedClient(nn::pcie::InvalidClassDriverHandle);
        }
        else
        {
            result = ResultFunctionNotAcquired();
        }
    }

    // Unconditional success if found in client profile, for case where function went to D3
    if((pProfile != NULL) && (pProfile->state.functionHandle == funcHandle))
    {
        *pProfile = InvalidEndpointFunctionProfile;
        result = ResultSuccess();
    }

    if(result.IsSuccess())
    {
        NN_PCIE_LOG_INFO("Client::ReleaseFunction() client 0x%x has released functionHandle 0x%x.\n", GetHandle(), funcHandle);
    }

    return result;
}

Result Client::GetFunctionState(FunctionState* pOutput, FunctionHandle functionHandle)
{
    Result result = ResultSuccess();
    Client::EndpointFunctionProfile* pProfile = Client::GetEndpointFunctionProfile(functionHandle);
    if(pProfile != nullptr)
    {
        *pOutput = pProfile->state;
    }
    else
    {
         result = ResultInvalidFunctionHandle();
    }
    return result;
}

Result Client::GetEndpointFunction(detail::EndpointFunction** ppEpFunc, detail::Client** ppClient,
                                   detail::Driver* pDriver, ClassDriverHandle classDriverHandle,
                                   FunctionHandle functionHandle, SecurityPolicy policy)
{
    Result result = ResultSuccess();
    if(ppEpFunc !=NULL) {*ppEpFunc = NULL;}
    if(ppClient !=NULL) {*ppClient = NULL;}
    detail::Client *pClient =  detail::Client::GetClient(pDriver, classDriverHandle);
    if (pClient != NULL)
    {
        detail::EndpointFunction *pEpFunc = detail::EndpointFunction::GetEndpointFunction(pDriver, functionHandle);
        if(pEpFunc != NULL)
        {
            bool isFunctionOwnedByCaller = (pEpFunc->GetAssignedClient() == pClient->GetHandle()) ? true : false;
            bool isFunctionOwned = pEpFunc->IsClientAssigned();
            if(isFunctionOwned && !isFunctionOwnedByCaller && (policy != SecurityPolicy_Open))
            {
                // Caller trying to access function owned by another client
                result = ResultFunctionNotAcquired();
            }
            else if(!isFunctionOwned && (policy == SecurityPolicy_MustBeAcquired))
            {
                // caller trying access unacquired function for privileged operation
                result = ResultFunctionNotAcquired();
            }

        }
        else
        {
            // check if client is trying to access an endpoint function which is not currently available
            Client::EndpointFunctionProfile* pProfile = pClient->GetEndpointFunctionProfile(functionHandle);
            if(pProfile != NULL)
            {
                result = ResultFunctionIsInD3State();
            }
            else
            {
                result = ResultInvalidFunctionHandle();
            }
        }

        if(result.IsSuccess())
        {
            if(ppClient!=NULL) {*ppClient = pClient;}
            if(ppEpFunc!=NULL) {*ppEpFunc = pEpFunc;}
        }
    }
    else
    {
        result = ResultInvalidClassDriverHandle();
    }

    return result;
}

int32_t Client::QueryFunctions(FunctionState* pOutput, size_t queryBufferSize)
{
    FunctionHandle handles[MaxEnumeratedDeviceFunctions];
    int32_t maxReturned = queryBufferSize / sizeof(FunctionState);
    maxReturned = (maxReturned < MaxEnumeratedDeviceFunctions) ? maxReturned : MaxEnumeratedDeviceFunctions;
    EndpointFunction::SearchCriteria searchCriteria;
    searchCriteria.baseClassCode = m_Config.baseClassCode;
    searchCriteria.subClassCode  = m_Config.subClassCode;
    searchCriteria.vendorId      = m_Config.vendorId;
    searchCriteria.deviceId      = m_Config.deviceId;
    searchCriteria.selectMask    = m_Config.selectMask;
    int32_t handleCount = EndpointFunction::FindEndpointFunctions(handles, m_pDriver, maxReturned, &searchCriteria, false);
    // For all endpoint functions that matched search criteria
    int32_t outputIndex = 0;
    for (int32_t handleIndex = 0; (handleIndex < handleCount) && (outputIndex < maxReturned); handleIndex++)
    {
        EndpointFunction *pEpFunc;
        if ((pEpFunc = EndpointFunction::GetEndpointFunction(m_pDriver, handles[handleIndex])) != NULL)
        {
            // important point - don't return any functions which are already assigned to a client
            if(!pEpFunc->IsClientAssigned())
            {
                pEpFunc->GetState(pOutput + outputIndex);
                outputIndex++;
            }
        }
    }
    return outputIndex;
}

void* Client::operator new(size_t size) NN_NOEXCEPT
{
    return nn::pcie::detail::MemoryAllocAligned(size, MinDataAlignmentSize, "Client");
}
void Client::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::pcie::detail::MemoryFree(p, "Client");
}

int32_t Client::GetNormalizedHandleIndex(Driver *pDriver, ClassDriverHandle handle)
{
    return pDriver->m_ClientHandleManager.GetNormalizedIndex(handle);
}

Client* Client::GetClient(Driver *pDriver, ClassDriverHandle handle)
{
    Client *pClient = pDriver->m_ClientHandleManager.Get(handle);
    return pClient;
}

// static method
void Client::UpdateClientFunctionState(Driver *pDriver,
                                       EndpointFunction *pEpFunc,
                                       ClassDriverHandle handle)
{
    Client *pClient = GetClient(pDriver, handle);
    FunctionState functionState = InvalidFunctionState;

    pEpFunc->GetState(&functionState);
    NN_PCIE_LOG_INFO("Function 0x%x has transitioned to state %s\n",
                     pEpFunc->GetHandle(), GetPmStateDescription(pEpFunc->GetPmState()));

    pDriver->PmStateCallout(&functionState);

    // Is there a bound client?
    if (pClient != NULL)
    {
        // Update bound client
        pClient->DoPmStateCallback(pEpFunc);

        // Endpoint function is going away
        if (pEpFunc->GetPmState() == PmState_D3Off)
        {
            pClient->ReleaseFunction(pEpFunc->GetHandle());
        }
    }
    else
    {
        // Is non-acquired device ready to be used?
        if (pEpFunc->GetPmState() == PmState_D0)
        {
            // For all clients
            for (Driver::ClientListType::iterator it = pDriver->m_ClientList.begin();
                 it != pDriver->m_ClientList.end(); it++)
            {
                Client *pClient = &(*it);
                EndpointFunction::SearchCriteria searchCriteria;
                searchCriteria.baseClassCode = pClient->m_Config.baseClassCode;
                searchCriteria.subClassCode  = pClient->m_Config.subClassCode;
                searchCriteria.vendorId      = pClient->m_Config.vendorId;
                searchCriteria.deviceId      = pClient->m_Config.deviceId;
                searchCriteria.selectMask    = pClient->m_Config.selectMask;
                if (pEpFunc->EvaluateSearchCriteria(&searchCriteria, false))
                {
                    pClient->DoRegistrationCallback();
                }
            }
        }
    }
}

Result Client::LookupReservedEndpointFunction(ClassDriverHandle* pOutClassDriverHandle,
                                              FunctionHandle* pOutFunctionHandle,
                                              IrqParameters* pOutIrqParameters,
                                              Driver *pDriver, BusNumber busNumber, DeviceNumber deviceNumber,
                                              FunctionNumber functionNumber, FullClassCode fullClassCode,
                                              uint16_t vendorId, uint16_t deviceId)
{
    Result result = ResultInvalidFunctionHandle();

    // For all clients
    for (Driver::ClientListType::iterator it = pDriver->m_ClientList.begin();
         it != pDriver->m_ClientList.end(); it++)
    {
        Client *pClient = &(*it);
        result = pClient->LookupReservedEndpointFunction(pOutClassDriverHandle, pOutFunctionHandle,
                                                         pOutIrqParameters,
                                                         busNumber, deviceNumber, functionNumber,
                                                         fullClassCode, vendorId, deviceId);
        if(result.IsSuccess())
        {
            break;
        }
    }

    return result;
}

Result Client::LookupReservedEndpointFunction(ClassDriverHandle* pOutClassDriverHandle,
                                              FunctionHandle* pOutFunctionHandle,
                                              IrqParameters* pOutIrqParameters,
                                              BusNumber busNumber, DeviceNumber deviceNumber,
                                              FunctionNumber functionNumber, FullClassCode fullClassCode,
                                              uint16_t vendorId, uint16_t deviceId)
{
    Result result = ResultInvalidFunctionHandle();

    // Client may have ownership of a device, but root complex is still asleep
    for(int32_t i=0; i < NN_PCIE_ARRAY_COUNT32(m_Functions); i++)
    {
        EndpointFunctionProfile* pF = m_Functions + i;
        if(pF->state.functionHandle != InvalidFunctionHandle)
        {
            if((pF->state.busNum == busNumber) && (pF->state.deviceNum == deviceNumber) &&
               (pF->state.funcNum == functionNumber) && (pF->state.classCode == fullClassCode) &&
               (pF->state.vendorId == vendorId) && (pF->state.deviceId == deviceId))
            {
                if(pOutClassDriverHandle != NULL) {*pOutClassDriverHandle = GetHandle();}
                if(pOutFunctionHandle != NULL) {*pOutFunctionHandle = pF->state.functionHandle;}
                if(pOutIrqParameters != NULL){*pOutIrqParameters = pF->irqParameters;}
                result = ResultSuccess();
                break;
            }
        }
    }

    return result;
}

void Client::DoRegistrationCallback()
{
    if(m_RegistrationCallback != NULL)
    {
        (*m_RegistrationCallback)(m_Config.classDriverContext);
    }
}

void Client::DoPmStateCallback(EndpointFunction *pEpFunc)
{
    if(m_PmStateCallback != NULL)
    {
        uintptr_t clientContext = 0;
        PmState pmState = pEpFunc->GetPmState();
        EndpointFunctionProfile* pProfile = GetEndpointFunctionProfile(pEpFunc->GetHandle());
        if(pProfile != NULL)
        {
            pProfile->state.pmState = pmState;
            clientContext = pProfile->clientContext;
        }
        (*m_PmStateCallback)(m_Config.classDriverContext, clientContext, pmState);
    }
}

void Client::EvaluateRegistration()
{
    FunctionHandle handles[MaxEnumeratedDeviceFunctions];
    EndpointFunction::SearchCriteria searchCriteria;
    searchCriteria.baseClassCode = m_Config.baseClassCode;
    searchCriteria.subClassCode  = m_Config.subClassCode;
    searchCriteria.vendorId      = m_Config.vendorId;
    searchCriteria.deviceId      = m_Config.deviceId;
    searchCriteria.selectMask    = m_Config.selectMask;
    int32_t funcCount = EndpointFunction::FindEndpointFunctions(handles, m_pDriver,
                                                                sizeof(handles) / sizeof(handles[0]),
                                                                &searchCriteria, false);
    if(funcCount > 0)
    {
        DoRegistrationCallback();
    }
}

Client::EndpointFunctionProfile* Client::GetEndpointFunctionProfile(FunctionHandle functionHandle)
{
    Client::EndpointFunctionProfile* pEpFuncProfile = NULL;

    // Client may have ownership of a device, but root complex is still asleep
    for(int32_t i=0; i < NN_PCIE_ARRAY_COUNT32(m_Functions); i++)
    {
        if(m_Functions[i].state.functionHandle == functionHandle)
        {
            pEpFuncProfile = m_Functions + i;
            break;
        }
    }

    return pEpFuncProfile;
}

Result Client::SetResetUponResumeEnable(FunctionHandle functionHandle, bool isEnabled)
{
    Result result = ResultSuccess();
    EndpointFunction* pEpFunc;
    if((result = GetEndpointFunction(&pEpFunc, NULL, m_pDriver, GetHandle(), functionHandle, SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        Client::EndpointFunctionProfile* pProfile = GetEndpointFunctionProfile(functionHandle);
        if(pProfile != NULL)
        {
            pProfile->isResetUponResumeEnable = isEnabled;
        }
        pEpFunc->SetResetUponResumeEnable(isEnabled);
    }
    return result;
}

Result Client::AcquireIrq(FunctionHandle functionHandle, IrqCallback callback,
                          uintptr_t context, IrqType irqType, int32_t numberOfVectors)
{
    Result result = ResultSuccess();
    EndpointFunction* pEpFunc;
    if((result = GetEndpointFunction(&pEpFunc, NULL, m_pDriver, GetHandle(), functionHandle, SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        Client::EndpointFunctionProfile* pProfile = GetEndpointFunctionProfile(functionHandle);
        if((result=pEpFunc->AcquireIrq(callback, context, irqType, numberOfVectors)).IsSuccess())
        {
            pProfile->irqParameters.callback        = callback;
            pProfile->irqParameters.context         = context;
            pProfile->irqParameters.type            = irqType;
            pProfile->irqParameters.numberOfVectors = numberOfVectors;
        }
    }
    return result;
}

Result Client::ReleaseIrq(FunctionHandle functionHandle)
{
    Result result = ResultSuccess();
    EndpointFunction* pEpFunc;
    Client::EndpointFunctionProfile* pProfile = GetEndpointFunctionProfile(functionHandle);

    // Is this a valid function?
    if((result = GetEndpointFunction(&pEpFunc, NULL, m_pDriver, GetHandle(), functionHandle, SecurityPolicy_MustBeAcquired)).IsSuccess())
    {
        result = pEpFunc->ReleaseIrq();
    }

    // Unconditional success if found in client profile, for case where function went to D3
    if((pProfile != NULL) && (pProfile->irqParameters.type != IrqType_Invalid))
    {
        pProfile->irqParameters = InvalidEndpointFunctionProfile.irqParameters;
        result = ResultSuccess();
    }

    return result;
}


} // namespace detail
} // namespace driver
} // namespace pcie
} // namespace nn
