﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/omm/detail/omm_Log.h>
#include <nn/omm/srv/omm_AwakeModeManager.h>
#include <nn/os/os_Thread.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/psm/psm_SystemApi.h>
#include <nn/bpc/bpc.h>
#include <nn/settings/system/settings_SystemApplication.h>

#include "detail/omm_Utility.h"
#include "omm_ImageBinary.h"

namespace nn { namespace omm { namespace srv {

    void AwakeModeManager::Initialize(DisplayManager* displayManager, CradleObserver* cradleObserver) NN_NOEXCEPT
    {
        int chargingSignMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&chargingSignMilliseconds, sizeof(chargingSignMilliseconds), "omm", "charging_sign_ms");
        int lowBatterySignMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&lowBatterySignMilliseconds, sizeof(lowBatterySignMilliseconds), "omm", "low_battery_sign_ms");
        int signFadeInMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&signFadeInMilliseconds, sizeof(signFadeInMilliseconds), "omm", "sign_fade_in_ms");
        int signFadeOutMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&signFadeOutMilliseconds, sizeof(signFadeOutMilliseconds), "omm", "sign_fade_out_ms");
        int signWaitLayerVisibleMilliseconds;
        settings::fwdbg::GetSettingsItemValue(&signWaitLayerVisibleMilliseconds, sizeof(signWaitLayerVisibleMilliseconds), "omm", "sign_wait_layer_visible_ms");

        m_ChargingSignageSpan = TimeSpan::FromMilliSeconds(chargingSignMilliseconds);
        m_LowBatterySignageSpan = TimeSpan::FromMilliSeconds(lowBatterySignMilliseconds);
        m_SignageFadeInSpan = TimeSpan::FromMilliSeconds(signFadeInMilliseconds);
        m_SignageFadeOutSpan = TimeSpan::FromMilliSeconds(signFadeOutMilliseconds);
        m_WaitLayerVisibleSpan = TimeSpan::FromMilliSeconds(signWaitLayerVisibleMilliseconds);

        m_DisplayManager = displayManager;
        m_CradleObserver = cradleObserver;
    }

    void AwakeModeManager::EnterSleepAndWait(os::SystemEvent* pSleepCancelEvent) NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(spsm::SleepSystemAndWaitAwake());

            auto powerState = spsm::GetCurrentState();
            NN_DETAIL_OMM_TRACE("[AwakeModeManager] powerState 0x%08x\n", powerState);
            if (powerState == spsm::PowerState_MinimumAwakeForLowBatterySign)
            {
                NN_DETAIL_OMM_TRACE("[AwakeModeManager] Power state low battery. Sign low battery %lld ms\n", m_LowBatterySignageSpan.GetMilliSeconds());
                bool isCharging;
                auto result = bpc::GetAcOk( &isCharging );
                if (result.IsFailure())
                {
                    NN_DETAIL_OMM_TRACE("[AwakeModeManager] bpc::GetAcOk() failed(0x%08x)\n", result.GetInnerValueForDebug());
                    isCharging = false;
                }

                const int pointX = (isCharging) ? LowBatteryChargingPointX : LowBatteryPointX;
                const int pointY = (isCharging) ? LowBatteryChargingPointY : LowBatteryPointY;
                const int width = (isCharging) ? LowBatteryChargingWidth : LowBatteryWidth;
                const int height = (isCharging) ? LowBatteryChargingHeight : LowBatteryHeight;
                const size_t dataSize = (isCharging) ? LowBatteryChargingDataSize : LowBatteryDataSize;
                const Bit8* pData = (isCharging) ? LowBatteryChargingData : LowBatteryData;

                Sign(pointX, pointY, width, height, pData, dataSize, m_LowBatterySignageSpan, NULL);

                NN_DETAIL_OMM_TRACE("[AwakeModeManager] Enter sleep again\n");
                continue;
            }

            auto wakeReason = spsm::GetLastWakeReason();
            NN_DETAIL_OMM_TRACE("[AwakeModeManager] Wake reason 0x%08x\n", wakeReason);
            if (wakeReason.Test<spsm::WakeReasonFlag::AcOk>() && m_CradleObserver->IsAnyCradleMounted())
            {
                NN_DETAIL_OMM_TRACE("[AwakeModeManager] Wake reason AcOk. Cradle is mounted\n");

                if ( settings::system::GetQuestFlag() )
                {
                    NN_DETAIL_OMM_TRACE("[AwakeModeManager] QuestFlag is true. Sleep canceled\n");
                    break;
                }

                NN_DETAIL_OMM_TRACE("[AwakeModeManager] Enable bluetooth and sign charging %lld ms\n", m_ChargingSignageSpan.GetMilliSeconds());
                detail::EnableBtmRadio();

                ShowChargingBatteryImage(pSleepCancelEvent);

                if ( pSleepCancelEvent != NULL && pSleepCancelEvent->TryWait() )
                {
                    NN_DETAIL_OMM_TRACE("[AwakeModeManager] Sleep canceled\n");
                    break;
                }

                NN_DETAIL_OMM_TRACE("[AwakeModeManager] Enter sleep again\n");
                continue;
            }

            break;
        }
    }

    void AwakeModeManager::ShowChargingBatteryImage(os::SystemEvent* pSleepCancelEvent) NN_NOEXCEPT
    {
        const int BatteryStartX = 6;
        const int BatteryStartY = 6;
        const int BatteryWidth = 36;
        const int BatteryFillHeight = 16;

        auto batteryPercentage = psm::GetBatteryChargePercentage();
        auto isChargingRed = batteryPercentage <= 15;

        const int pointX = (isChargingRed) ? ChargingRedPointX : ChargingPointX;
        const int pointY = (isChargingRed) ? ChargingRedPointY : ChargingPointY;
        const int width = (isChargingRed) ? ChargingRedWidth : ChargingWidth;
        const int height = (isChargingRed) ? ChargingRedHeight : ChargingHeight;
        const size_t dataSize = (isChargingRed) ? ChargingRedDataSize : ChargingDataSize;
        const Bit8* pData = (isChargingRed) ? ChargingRedData : ChargingData;

        // am プロセスはヒープに 512KB 用意しており、最大でも使用量は 200KB 程度なため、ここでのメモリ確保(8KB 強)は必ず成功する
        std::unique_ptr<Bit32[]> buffer(new Bit32[dataSize / sizeof(Bit32)]);
        NN_ABORT_UNLESS( buffer.get() != NULL );
        std::memcpy(buffer.get(), pData, dataSize);

        // OverDispTest におけるバッテリー残量表示スケールの近似式 (100% のとき 1.00、1% のとき 0.05)
        auto scale = 0.0096 * batteryPercentage + 0.0404;
        auto batteryFillWidth = static_cast<int>(BatteryWidth * (1 - scale) + 0.5);
        auto offsetX = BatteryStartX + BatteryWidth - batteryFillWidth;

        NN_DETAIL_OMM_TRACE("[AwakeModeManager] fillWidth=%d, offsetX=%d\n", batteryFillWidth, offsetX);

        FillInBlack(buffer.get(), width, height, batteryFillWidth, BatteryFillHeight, offsetX, BatteryStartY);
        Sign(pointX, pointY, width, height, buffer.get(), dataSize, m_ChargingSignageSpan, pSleepCancelEvent);
    }

    void AwakeModeManager::FillInBlack(Bit32 *pBuffer, int bufferWidth, int bufferHeight, int width, int height, int offsetX, int offsetY) NN_NOEXCEPT
    {
        if (width + offsetX > bufferWidth)
        {
            return;
        }
        if (height + offsetY > bufferHeight)
        {
            return;
        }

        Bit32 *p = pBuffer + offsetY * bufferWidth + offsetX;
        for (int h = 0; h < height; h++)
        {
            for (int w = 0; w < width; w++)
            {
                *(p + w) = 0xff000000;
            }
            p += bufferWidth;
        }
    }

    void AwakeModeManager::Sign(int x, int y, int width, int height, const void* buffer, size_t size, TimeSpan span, os::SystemEvent* pSleepCancelEvent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(width * height * sizeof(Bit32) <= size);
        NN_DETAIL_OMM_TRACE("[AwakeModeManager] Sign fade in %lld ms out %lld ms\n", m_SignageFadeInSpan.GetMilliSeconds(), m_SignageFadeOutSpan.GetMilliSeconds());

        m_DisplayManager->StartDrawingLayer();
        m_DisplayManager->SetDrawingLayerTexture(x, y, width, height, buffer, size);
        m_DisplayManager->SetDrawingLayerVisible(true);

        m_DisplayManager->EnableHandheldDisplay();

        // workaround for SIGLO-38206
        // BG レイヤを有効にしても、その絵が出るまで数フレームの時間がかかる恐れ
        os::SleepThread(m_WaitLayerVisibleSpan);

        // workaround for SIGLO-36938
        // フェード時に下のレイヤが透けないように、バックライトだけでフェードする
        // スリープ復帰時の表示は LCD のみなので OK
        m_DisplayManager->FadeLcd(true, 0);
        m_DisplayManager->FadeBacklight(true, m_SignageFadeInSpan);

        if ( pSleepCancelEvent != NULL )
        {
            pSleepCancelEvent->TimedWait( span );
        }
        else
        {
            os::SleepThread( span );
        }

        m_DisplayManager->FadeBacklight(false, m_SignageFadeOutSpan);
        m_DisplayManager->FadeLcd(false, 0);

        m_DisplayManager->DisableAllDisplay();

        m_DisplayManager->SetDrawingLayerVisible(false);
        m_DisplayManager->StopDrawingLayer();
    }

}}}
