﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Context.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/olsc/srv/util/olsc_SeriesPosition.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace util {
namespace {
Result FindAndGetSaveDataCommitId(nn::util::optional<nn::fs::SaveDataCommitId>* pOut, const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    auto pSaveDataInfo = util::FindSaveData(uid, appId);
    if (pSaveDataInfo)
    {
        fs::SaveDataCommitId commitId;
        fs::ScopedAutoAbortDisabler antiAbort;
        auto r = fs::GetSaveDataCommitId(&commitId, pSaveDataInfo->saveDataSpaceId, pSaveDataInfo->saveDataId);
        NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultPathNotFound::Includes(r), r);
        if (r.IsSuccess())
        {
            *pOut = commitId;
            NN_RESULT_SUCCESS;
        }
    }
    *pOut = nn::util::nullopt;
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::olsc::<anonymous>

DataNewness ConvertToDataNewness(SeriesPosition seriesPosition) NN_NOEXCEPT
{
    switch (seriesPosition)
    {
    case SeriesPosition::Equal:
        return DataNewness::Equal;
    case SeriesPosition::ServerDataIsNewer:
        return DataNewness::Newer;
    case SeriesPosition::LocalDataIsNewer:
        return DataNewness::Older;
    case SeriesPosition::Discrete:
        return DataNewness::Discrete;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

SeriesPosition CalculateSeriesPosition(
    const SeriesInfo& lastSi,
    const nn::util::optional<fs::SaveDataCommitId>& pCurrentCommitId,
    const nn::util::optional<SaveDataArchiveInfo>& pCachedSda) NN_NOEXCEPT
{
    if (!pCachedSda)
    {
        if (pCurrentCommitId)
        {
            // サーバにデータが無くローカルにはセーブがあるので、ローカルが新しいことになる
            return SeriesPosition::LocalDataIsNewer;
        }
        // サーバにもローカルにもデータがない場合は、系列違い扱いとする
        return SeriesPosition::Discrete;
    }

    // ----- 以下 サーバにデータがあるパターン

    if (!pCurrentCommitId)
    {
        // サーバにデータがあってローカルにセーブが無い場合は、サーバが新しいことになる
        return SeriesPosition::ServerDataIsNewer;
    }

    // ----- 以下 サーバにもローカルにもデータがあるパターン

    auto& cachedSi = pCachedSda->seriesInfo;
    if (cachedSi == lastSi)
    {
        if (*pCurrentCommitId != lastSi.commitId)
        {
            // 最後に同期した状態からローカルのセーブデータが変わっている
            return SeriesPosition::LocalDataIsNewer;
        }
        // 最後に同期した状態からローカルのセーブデータが変わっていない
        return SeriesPosition::Equal;
    }

    // ----- 以下 サーバSI と LastSI が異なるパターン

    // LastSiとサーバで ID が一致している
    // LastSiとサーバでコミットID が異なっている（LastSi != cachedSi && lastSi.id == cachedSi.id から自明なので下記 if 文の条件から省略）
    // LastSi とローカル最新のコミット ID が一致している
    // このケースでは同系列でサーバ側だけ更新されていると言える
    if (cachedSi.seriesId == lastSi.seriesId && lastSi.commitId == *pCurrentCommitId)
    {
        return SeriesPosition::ServerDataIsNewer;
    }

    // その他のケースでは系列が変わっている
    return SeriesPosition::Discrete;
}

Result EvaluateSaveDataInfoCacheSeriesInfo(
    SeriesPosition* pOut,
    const account::Uid& uid,
    const SaveDataArchiveInfo& cachedSda,
    const database::SeriesInfoDatabase& seriesInfoDatabase) NN_NOEXCEPT
{
    nn::util::optional<fs::SaveDataCommitId> pCommitId;
    NN_RESULT_DO(FindAndGetSaveDataCommitId(&pCommitId, uid, cachedSda.applicationId));

    *pOut = CalculateSeriesPosition(
        seriesInfoDatabase.Get(cachedSda.applicationId),
        pCommitId,
        nn::util::optional<SaveDataArchiveInfo>(cachedSda));
    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::olsc::srv::util
