﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_SaveDataTransferForCloudBackUp.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/util/olsc_SaveDataAccessLockManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace olsc { namespace srv { namespace util {

void SaveDataAccessLockManager::AccessLock::Append(ApplicationId appId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_AccessLockCount < MaxAccessLockCount, "Too many access lock.\n");

    for (int i = 0; i < m_AccessLockCount; ++i)
    {
        if (m_AccessLockInfos[i].first == appId)
        {
            return;
        }
    }

    m_AccessLockInfos[m_AccessLockCount].first = appId;
    m_AccessLockCount++;
}

void SaveDataAccessLockManager::AccessLock::LockAccess() NN_NOEXCEPT
{
    for (int i = 0; i < m_AccessLockCount; ++i)
    {
        auto& t = m_AccessLockInfos[i];
        if (!t.second)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenSaveDataTransferProhibiterForCloudBackUp(&t.second, { t.first.value }));
        }
    }
}

void SaveDataAccessLockManager::AccessLock::UnlockAccess() NN_NOEXCEPT
{
    for (int i = 0; i < m_AccessLockCount; ++i)
    {
        m_AccessLockInfos[i].second.reset();
    }
}

ApplicationId SaveDataAccessLockManager::AccessLock::GetMainApplicationId() const NN_NOEXCEPT
{
    return m_MainApp;
}

ApplicationId SaveDataAccessLockManager::AccessLock::GetLockedApplicationId(int index) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(index >= 0 && index < m_AccessLockCount);
    return m_AccessLockInfos[index].first;
}

int SaveDataAccessLockManager::AccessLock::GetLockedApplicationIdCount() const NN_NOEXCEPT
{
    return m_AccessLockCount;
}

// --------------------------------------------------------------------------------------

void SaveDataAccessLockManager::Enable(ApplicationId appId) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_ABORT_UNLESS(!m_IsSuppressed);

    NN_ABORT_UNLESS(!FindAccessLock(appId), "Access lock has already enabled. AppId = %016llx\n", appId.value);

    for (auto& accessLock : m_AccessLocks)
    {
        if (!accessLock)
        {
            accessLock.emplace(appId);
            accessLock->Append(appId);
            accessLock->LockAccess();
            return;
        }
    }
    NN_ABORT("Access lock is full.\n");
}

void SaveDataAccessLockManager::Disable(ApplicationId appId) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_ABORT_UNLESS(!m_IsSuppressed);

    for (auto& accessLock : m_AccessLocks)
    {
        if (accessLock && accessLock->GetMainApplicationId() == appId)
        {
            accessLock = nn::util::nullopt;
            return;
        }
    }
    NN_ABORT("Not found access lock. AppId = %016llx.\n", appId.value);
}

void SaveDataAccessLockManager::Update(ApplicationId appId, const ApplicationId targets[], int targetCount) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_ABORT_UNLESS(!m_IsSuppressed);

    auto info = FindAccessLock(appId);
    NN_ABORT_UNLESS(info, "Not found access lock. AppId = %016llx.\n", appId.value);

    for (int i = 0; i < targetCount; ++i)
    {
        info->Append(targets[i]);
    }

    info->LockAccess();
}

void SaveDataAccessLockManager::Suppress() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_ABORT_UNLESS(!m_IsSuppressed);

    for (auto& accessLock : m_AccessLocks)
    {
        if (accessLock)
        {
            accessLock->UnlockAccess();
        }
    }

    m_IsSuppressed = true;
}

void SaveDataAccessLockManager::Unsuppress() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_ABORT_UNLESS(m_IsSuppressed);

    for (auto& accessLock : m_AccessLocks)
    {
        if (accessLock)
        {
            accessLock->LockAccess();
        }
    }

    m_IsSuppressed = false;
}

int SaveDataAccessLockManager::ListLockedApplicationIds(ApplicationId out[], int maxOutCount) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    int listedCount = 0;
    for (auto& accessLock : m_AccessLocks)
    {
        if (accessLock)
        {
            for (int i = 0; i < accessLock->GetLockedApplicationIdCount() && listedCount < maxOutCount; ++i)
            {
                out[listedCount] = accessLock->GetLockedApplicationId(i);
                listedCount++;
            }
        }
    }
    return listedCount;
}

SaveDataAccessLockManager::AccessLock* SaveDataAccessLockManager::FindAccessLock(ApplicationId appId) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    for (auto& accessLock : m_AccessLocks)
    {
        if (accessLock && accessLock->GetMainApplicationId() == appId)
        {
            return &accessLock.value();
        }
    }
    return nullptr;
}

}}}} //namespace nn::olsc::srv::util
