﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/nn_Abort.h>
#include <nn/olsc/srv/util/olsc_ActiveUserList.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace olsc { namespace srv { namespace util {

namespace {
    TimeSpan GetCurrentTime() NN_NOEXCEPT
    {
        return os::GetSystemTick().ToTimeSpan();
    }
} // namespace

// ---------------------------------------------------------------------------------

ActiveUserList::ActiveUserList(TimeSpan updateInterval) NN_NOEXCEPT
    : ActiveUserListBase(updateInterval)
{}

int ActiveUserList::ListActiveUsers(account::Uid out[], int maxListCount) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(maxListCount >= account::UserCountMax);
    NN_UNUSED(maxListCount);

    account::Uid users[account::UserCountMax];
    int listedCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&listedCount, users, account::UserCountMax));

    int availablesCount = 0;
    for (int i = 0; i < listedCount; ++i)
    {
        auto& user = users[i];
        account::NetworkServiceAccountManager m;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetNetworkServiceAccountManager(&m, user));

        bool isAvailable;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m.IsNetworkServiceAccountAvailable(&isAvailable));

        if (isAvailable)
        {
            out[availablesCount] = user;
            availablesCount++;
        }
    }

    return availablesCount;
}

// ---------------------------------------------------------------------------------

ActiveUserListBase::ActiveUserListBase(TimeSpan updateInterval) NN_NOEXCEPT
    : m_UpdateInterval(updateInterval)
{}

int ActiveUserListBase::GetCount() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    return m_Count;
}

void ActiveUserListBase::Refresh() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    // NSA が有効なアカウントを列挙
    account::Uid activeUsers[account::UserCountMax];
    int activeUserCount = ListActiveUsers(activeUsers, account::UserCountMax);

    UserInfo refreshed[account::UserCountMax] = {};
    for (int i = 0; i < activeUserCount; ++i)
    {
        auto& activeUser = activeUsers[i];
        auto index = FindIndex(activeUser);
        if (index)
        {
            refreshed[i] = m_UserInfos[*index];
        }
        else
        {
            // 新しく追加されたものは即時実行するため次の実行時刻を 0 にしておく
            refreshed[i] = UserInfo::Make(activeUser, TimeSpan::FromSeconds(0));
        }
    }

    std::copy_n(&refreshed[0], activeUserCount, &m_UserInfos[0]);
    m_Count = activeUserCount;
}

bool ActiveUserListBase::IsActive(const account::Uid& uid) const NN_NOEXCEPT
{
    return FindIndex(uid);
}

void ActiveUserListBase::ScheduleNextUpdate(const account::Uid& uid) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    auto index = FindIndex(uid);
    NN_ABORT_UNLESS(index);
    // Refresh と ScheduleNextUpdate を別スレッドで動かすようなケースでは
    // 予期しないところで Refresh が呼ばれていて、アクティブユーザが消えると
    // ↑のアサートに引っかかる可能性がある

    m_UserInfos[*index].nextUpdateTime = GetCurrentTime() + m_UpdateInterval;
}

int ActiveUserListBase::ListUsersToUpdate(account::Uid out[], const int maxOutCount) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_SDK_ASSERT(maxOutCount >= m_UserInfos.size());
    NN_UNUSED(maxOutCount);

    auto currentTime = GetCurrentTime();

    int listedCount = 0;
    for (int i = 0; i < m_Count; ++i)
    {
        auto& user = m_UserInfos[i];
        if (currentTime >= user.nextUpdateTime)
        {
            out[listedCount] = user.uid;
            listedCount++;
        }
    }

    return listedCount;
}

TimeSpan ActiveUserListBase::GetTimeSpanToNextUpdate() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);
    NN_SDK_ASSERT(m_Count > 0);

    auto currentTime = GetCurrentTime();

    TimeSpan returnValue = m_UpdateInterval;
    for (int i = 0; i < m_Count; ++i)
    {
        auto& user = m_UserInfos[i];
        auto waitTime = std::max(user.nextUpdateTime - currentTime, TimeSpan::FromSeconds(0));
        returnValue = std::min(returnValue, waitTime);
    }

    return returnValue;
}

// -------------------------------------------------------------------------------

nn::util::optional<int> ActiveUserListBase::FindIndex(const account::Uid& uid) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Lock);

    for (int i = 0; i < m_Count; ++i)
    {
        if (m_UserInfos[i].uid == uid)
        {
            return i;
        }
    }
    return nn::util::nullopt;
}

}}}} //namespace nn::olsc::srv::util
