﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <random>
#include <string>

#include <nn/nn_Common.h>
#include <curl/curl.h>

#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_TimeStampForDebug.h>
#include <nn/fs/fs_Directory.h>
#include <nn/ns/ns_ApplicationVersionSystemApi.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/srv/database/olsc_DataArray.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileUpload.h>
#include <nn/olsc/srv/transfer/olsc_ComponentFileDownload.h>
#include <nn/olsc/srv/transfer/olsc_PolicyInfoDownload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/olsc/srv/transfer/olsc_TransferUtil.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/os/os_Random.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/util/util_TFormatString.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

namespace
{
    // TODO : 設定できるようにしたい。せめて NINTENDO_SDK_ROOT に絡めたい
    static const std::string Root("C:/Windows/Temp/NxBackup/");
    static const std::string SdaTable("sdaTable");
    static const std::string CfTable("cfTable");
    static const std::string ObjExtension(".s3obj");

    static const int MaxApplicationCount = 2048;
    static const int MaxDivisionCount = 64;
    static const int MaxComponentFileCount = 2048 * MaxDivisionCount;
    static const int MaxListCacheCount = 32;
    static const int SdaInfoListCacheSize = MaxListCacheCount * sizeof(SaveDataArchiveInfo);
    static const int CfInfoListCacheSize = MaxListCacheCount * sizeof(ComponentFileInfo);

    class HostMountManager : public util::MountManagerBase
    {
    public:
        HostMountManager() = default;
        virtual void Release(util::MounterId mounter, util::ReferenceMode referenceMode) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(mounter);
            NN_UNUSED(referenceMode);

            std::lock_guard<decltype(m_MountLock)> lock(m_MountLock);
            if (m_MountCount == 1)
            {
                nn::fs::UnmountHostRoot();
            }
            m_MountCount--;

            NN_SDK_ASSERT(m_MountCount >= 0);
        }

        virtual const char* GetMountName(util::MounterId mounter) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(mounter);
            return "";
        }
        virtual const char* GetRootPath(util::MounterId mounter) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(mounter);
            return Root.c_str();
        }


        util::MounterId Acquire() NN_NOEXCEPT
        {
            std::lock_guard<decltype(m_MountLock)> lock(m_MountLock);
            if (m_MountCount == 0)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
            }
            m_MountCount++;

            return 0;
        }
    private:
        os::SdkRecursiveMutex m_MountLock;
        int m_MountCount = 0;
    };

    HostMountManager g_HostMountManager;

    class SdaInfoList : public database::DataArray<HostMountManager, SaveDataArchiveInfo, MaxApplicationCount, 0, MaxListCacheCount>
    {
    public:
        using Base = database::DataArray<HostMountManager, SaveDataArchiveInfo, MaxApplicationCount, 0, MaxListCacheCount>;
        SdaInfoList(ReadBuffer* readBuffer, HostMountManager& mountManager) : Base(&m_CacheBuffer, readBuffer, mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sdaInfodb_meta";
        }
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sdatdb_el_entry";
        }
        virtual util::ReadMount AcquireReadMount(HostMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE
        {
            auto mounterId = mountManager.Acquire();

            return mountManager.GetMountContext<util::ReadMount>(mounterId);
        }
        virtual util::WriteMount AcquireWriteMount(HostMountManager& mountManager) NN_NOEXCEPT NN_OVERRIDE
        {
            auto mounterId = mountManager.Acquire();

            return mountManager.GetMountContext<util::WriteMount>(mounterId);
        }
    private:
        CacheBuffer m_CacheBuffer;
    };

    class CfInfoList : public database::DataArray<HostMountManager, ComponentFileInfo, MaxComponentFileCount, 0, MaxListCacheCount>
    {
    public:
        using Base = database::DataArray<HostMountManager, ComponentFileInfo, MaxComponentFileCount, 0, MaxListCacheCount>;
        CfInfoList(ReadBuffer* readBuffer, HostMountManager& mountManager) : Base(&m_CacheBuffer, readBuffer, mountManager)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sdaInfodb_meta";
        }
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "sdatdb_el_entry";
        }
        virtual util::ReadMount AcquireReadMount(HostMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE
        {
            auto mounterId = mountManager.Acquire();

            return mountManager.GetMountContext<util::ReadMount>(mounterId);
        }
        virtual util::WriteMount AcquireWriteMount(HostMountManager& mountManager) NN_NOEXCEPT NN_OVERRIDE
        {
            auto mounterId = mountManager.Acquire();

            return mountManager.GetMountContext<util::WriteMount>(mounterId);
        }
    private:
        CacheBuffer m_CacheBuffer;
    };

    Result ReadFileAtOnce(size_t* outValue, const char* path, void* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        fs::FileHandle handle;
        NN_RESULT_DO(fs::OpenFile(&handle, path, fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(handle);
        };

        NN_RESULT_DO(fs::ReadFile(outValue, handle, 0, pBuffer, bufferSize, fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    std::string GetSaveDataArchiveInfoListTablePath() NN_NOEXCEPT
    {
        return Root + SdaTable;
    }
    std::string GetComponentFileInfoListTablePath() NN_NOEXCEPT
    {
        return Root + CfTable;
    }
    std::string GetSaveDataArchivePath(SaveDataArchiveId sdaId) NN_NOEXCEPT
    {
        return Root + std::to_string(sdaId);
    }
    std::string GetObjectFilePath(SaveDataArchiveId sdaId, ComponentFileId chunkId) NN_NOEXCEPT
    {
        return Root + std::to_string(sdaId) + "/" + std::to_string(chunkId) + ObjExtension;
    }

    // SaveDataArchive を偽造
    SaveDataArchiveInfo CreateSaveDataArchive(const account::Uid& uid, ApplicationId appId, const SeriesInfo& seriesInfo, size_t dataSize, time::PosixTime updatedTime) NN_NOEXCEPT
    {
        SaveDataArchiveInfo ret;
        nn::os::GenerateRandomBytes(&ret.id, sizeof(ret.id));
        ret.nsaId = {0};
        ret.applicationId = appId;
        ret.userId = uid;
        ret.deviceId = static_cast<DeviceId>(0x0);
        ret.dataSize = dataSize;
        ret.seriesInfo = seriesInfo;
        ret.status = SaveDataArchiveStatus::Uploading;
        ret.autoBackup = false;
        // TODO : API 提供してもらう予定
        ret.hasThumbnail = false;
        // nn::ncm::ApplicationId ncmAppId;
        // ncmAppId.value = appId.value;
        // ret.launchRequiredVersion = nn::ns::GetLaunchRequiredVersion(ncmAppId);
        ret.launchRequiredVersion = 0;
        ret.numOfPartitions = 8;
        ret.savedAt    = updatedTime;
        ret.timeoutAt  = {0};
        ret.finishedAt = {0};
        ret.createdAt  = updatedTime;
        ret.createdAt  = updatedTime;
        return ret;
    }

    // ComponentFileInfo を偽造
    ComponentFileInfo CreateComponentFile(SaveDataArchiveId sdaId, ClientArgument clientArgument, ComponentFileType type) NN_NOEXCEPT
    {
        ComponentFileInfo ret;
        nn::os::GenerateRandomBytes(&ret.id, sizeof(ret.id));
        ret.sdaId = sdaId;
        ret.clientArgument = clientArgument;
        ret.type = type;
        ret.status = ComponentFileStatus::Uploading;
        // TODO : 現状はサーバーで必須になっているため入れるが使用予定はないため固定にする
        // 将来的に削除予定。
        ret.componentFileSize = 1024;
        memset(ret.componentFileDigest.data, 0x00, ComponentFileDigest::Size);

        ret.saveDataChunkSize = 0;
        memset(ret.saveDataChunkDigest.data, 0x00, SaveDataChunkDigest::Size);
        // obj パスを作成
        std::string objPath = GetObjectFilePath(sdaId, ret.id);
        nn::util::Strlcpy(ret.url, objPath.c_str(), ComponentFileInfo::MaxUrlLength);
        ret.createdAt = {0};
        ret.updatedAt = {0};
        return ret;
    }

    Result AddNewSaveDataArchive(SaveDataArchiveInfo* pOut, const account::Uid& uid, ApplicationId appId, const SeriesInfo& seriesInfo, size_t dataSize, time::PosixTime updatedTime, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        // TODO : フォルダは uid ではなく NsaId で行う。NsaId は NsaIdTokenCache を Base64 でデコード後 Json パースすれば取得可能
        SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
        std::string table = GetSaveDataArchiveInfoListTablePath();
        SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
        std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

        // TODO : 同一 NSA ID , ApplicationID でアップロード中の SaveDataArchiveInfo が存在した場合の挙動をサーバ側とすり合わせ
        // 一旦削除する仕様にする(真面目にやるなら排他)
        NN_RESULT_DO(sdaInfoList.RemoveAll([&](const SaveDataArchiveInfo& s) {
            return (uid == s.userId && appId == s.applicationId && s.status == SaveDataArchiveStatus::Uploading);
        }));

        SaveDataArchiveInfo targetSda = CreateSaveDataArchive(uid, appId, seriesInfo, dataSize, updatedTime);
        sdaInfoList.PushBack(targetSda);
        std::string filePath = GetSaveDataArchivePath(targetSda.id);
        NN_RESULT_DO(nn::fs::CreateDirectory(filePath.c_str()));
        *pOut = targetSda;
        NN_RESULT_SUCCESS;
    }

    Result AddNewComponentFile(ComponentFileInfo* pOut, SaveDataArchiveId sdaId, ClientArgument clientArgument, ComponentFileType type, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
    {
        CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(workBuffer);
        std::string table = GetComponentFileInfoListTablePath();
        CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
        std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);
        ComponentFileInfo targetComponentFileInfo = CreateComponentFile(sdaId, clientArgument, type);
        cfInfoList.PushBack(targetComponentFileInfo);
        *pOut = targetComponentFileInfo;
        NN_RESULT_SUCCESS;
    }
}

Result StartSaveDataArchiveUpload(SaveDataArchiveInfo* out, const account::Uid& uid, ApplicationId appId, const SeriesInfo& info, size_t dataSize, time::PosixTime updatedTime, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);
    NN_RESULT_THROW_UNLESS(workBufferSize >= sizeof(SdaInfoList::CacheBuffer), ResultInvalidArgument());

    auto writeMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(writeMount, util::ReferenceMode::ReadAndWrite);
    };

    NN_RESULT_TRY(nn::fs::CreateDirectory(Root.c_str()))
        NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        {
        }
    NN_RESULT_END_TRY
    NN_RESULT_DO(AddNewSaveDataArchive(out, uid, appId, info, dataSize, updatedTime, workBuffer, workBufferSize));
    NN_RESULT_SUCCESS;
}

Result CreateComponentFile(ComponentFileInfo* out, SaveDataArchiveId sdaId, const ClientArgument& clientArgument, ComponentFileType type, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto writeMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(writeMount, util::ReferenceMode::ReadAndWrite);
    };

    // CF をテーブルに追加
    NN_RESULT_DO(AddNewComponentFile(out, sdaId, clientArgument, type, workBuffer, workBufferSize));
    NN_RESULT_SUCCESS;
}


Result ExportComponentFile(size_t* pOutSize, SaveDataChunkDigest* pOutDigest, const char* path, fs::ISaveDataDivisionExporter* exporter, fs::SaveDataChunkId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);

    auto writeMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(writeMount, util::ReferenceMode::ReadAndWrite);
    };

    // 空回ししてサイズとハッシュを取得
    size_t dataSize;
    SaveDataChunkDigest digest;
    NN_RESULT_DO(GetExportSizeByDryRun(&dataSize, &digest, exporter, id, workBuffer, workBufferSize));

    std::unique_ptr<fs::ISaveDataChunkExporter> chunkExporter;
    NN_RESULT_DO(exporter->OpenSaveDataChunkExporter(&chunkExporter, id));

    // 既にあったら削除
    fs::DeleteFile(path);
    NN_RESULT_DO(nn::fs::CreateFile(path, static_cast<int64_t>(dataSize)));
    fs::FileHandle handle;
    NN_RESULT_DO(OpenFile(&handle, path, fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(handle);
    };

    size_t exportTotalSize = 0;
    size_t offset = 0;
    while(NN_STATIC_CONDITION(true))
    {
        if(pCancelable)
        {
            NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), ResultCanceledDuringDataAccess());
        }
        size_t pulledSize = 0;
        NN_RESULT_DO(chunkExporter->Pull(&pulledSize, workBuffer, workBufferSize));
        if (pulledSize > 0)
        {
            NN_RESULT_DO(fs::WriteFile(handle, offset, workBuffer, pulledSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
            offset += pulledSize;
            exportTotalSize += pulledSize;
        }
        else
        {
            break;
        }
    }

    NN_RESULT_THROW_UNLESS(exportTotalSize == dataSize, ResultInvalidArgument());

    // アップロードに成功後出力値を設定
    *pOutSize = dataSize;
    *pOutDigest = digest;

    NN_RESULT_SUCCESS;
}

Result CompleteComponentFile(ComponentFileId id, size_t archiveDataSize, const SaveDataChunkDigest& SaveDataChunkDigest, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto writeMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(writeMount, util::ReferenceMode::ReadAndWrite);
    };

    CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(workBuffer);
    std::string table = GetComponentFileInfoListTablePath();
    CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
    std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);

    auto targetCf = cfInfoList.FindIf([&id](const ComponentFileInfo& s) {
        return s.id == id;
        });
    NN_ABORT_UNLESS(targetCf);

    // ComponentFileInfo の status が Uploading であることを検証
    NN_RESULT_THROW_UNLESS(targetCf->status == ComponentFileStatus::Uploading, ResultComponentFileUnacceptableContent());

    targetCf->saveDataChunkSize = archiveDataSize;
    targetCf->saveDataChunkDigest = SaveDataChunkDigest;

    // s3obj との Hash 検証はパス
    targetCf->status = ComponentFileStatus::Fixed;

    NN_RESULT_DO(
        cfInfoList.ReplaceIf([&targetCf](const ComponentFileInfo& s) {
           return s.id == targetCf->id;
        }, *targetCf));
    NN_RESULT_SUCCESS;
}

Result FinishSaveDataArchiveUpload(SaveDataArchiveId id, const srv::KeySeed& keySeed, const srv::InitialDataMac& mac, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // TODO : KeySeed と mac を SdaInfoList に覚えておく
    NN_UNUSED(keySeed);
    NN_UNUSED(mac);
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    NN_SDK_ASSERT(workBufferSize >= SdaInfoListCacheSize + CfInfoListCacheSize);
    NN_UNUSED(workBufferSize);

    auto writeMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(writeMount, util::ReferenceMode::ReadAndWrite);
    };

    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(reinterpret_cast<char*>(workBuffer) + SdaInfoListCacheSize);
    CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
    std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);

    // テーブルに SDA が登録されているかを調べる。
    auto newSda = sdaInfoList.FindIf([&id](const SaveDataArchiveInfo& r) {
        return r.id == id && r.status == SaveDataArchiveStatus::Uploading;
        });
    NN_RESULT_THROW_UNLESS(newSda, ResultInvalidArgument());
    // SDA の status が uploading であることを検証
    NN_RESULT_THROW_UNLESS(newSda->status == SaveDataArchiveStatus::Uploading, ResultSaveDataArchiveUnacceptableContent());

    // 全ての ComponentFileInfo が Uploading でないことを確認
    auto notFixedCf = cfInfoList.FindIf([&](const ComponentFileInfo& s) {
        return s.sdaId == id && s.status == ComponentFileStatus::Uploading;
        });
    NN_RESULT_THROW_UNLESS(!notFixedCf, ResultComponentFileUnacceptableContent());

    // 同一 NSA ID, Application ID で Fixed 状態の SDA を取得
    while(NN_STATIC_CONDITION(true))
    {
        auto oldSda = sdaInfoList.FindIf([&newSda](const SaveDataArchiveInfo& r) {
            return (newSda->userId == r.userId && newSda->applicationId == r.applicationId && r.status == SaveDataArchiveStatus::Fixed);
        });
        if(oldSda)
        {
            // 古い SDA が見つかった
            NN_RESULT_DO(sdaInfoList.Remove(*oldSda));
            // フォルダごと削除
            std::string dirPath = GetSaveDataArchivePath(oldSda->id);
            nn::fs::DeleteDirectoryRecursively(dirPath.c_str());
        }
        else
        {
            break;
        }
    }

    newSda->status = SaveDataArchiveStatus::Fixed;
    nn::time::PosixTime time;
    NN_RESULT_DO(nn::time::StandardUserSystemClock::GetCurrentTime(&time));
    newSda->finishedAt = time;

    NN_RESULT_DO(
        sdaInfoList.ReplaceIf([&newSda](const SaveDataArchiveInfo& r) {
            return r.id == newSda->id;
        }, *newSda));
    NN_RESULT_SUCCESS;
}

Result RequestSaveDataArchiveInfoList(int* pOutCount, SaveDataArchiveInfo saveDataArchiveList[], size_t listCount, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetSaveDataArchiveInfoListTablePath();
    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    size_t sdaNum = 0;
    nn::Result result = ResultSuccess();
    sdaInfoList.ForEach([&](const SaveDataArchiveInfo& r) {
        if (sdaNum < listCount)
        {
            memcpy(&saveDataArchiveList[sdaNum++], &r, sizeof(SaveDataArchiveInfo));
            return true;
        }
        result = ResultInvalidArgument();
        return false;
    });
    NN_RESULT_DO(result);
    *pOutCount = static_cast<int>(sdaNum);
    NN_RESULT_SUCCESS;
}

Result RequestSaveDataArchiveInfoList(int* pOutCount, SaveDataArchiveInfo saveDataArchiveList[], size_t listCount, const NsaIdToken& nsaIdToken, ApplicationId appId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetSaveDataArchiveInfoListTablePath();
    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    // TODO : NsaId でも検索する
    int outCount = 0;
    sdaInfoList.ForEach([&](const SaveDataArchiveInfo& r) {
        if(r.applicationId == appId)
        {
            if(outCount < static_cast<int>(listCount))
            {
                saveDataArchiveList[outCount] = r;
            }
            outCount++;
        }
        return true;
    });
    NN_SDK_ASSERT(outCount < 3);
    *pOutCount = outCount;
    NN_RESULT_SUCCESS;
}

Result RequestSaveDataArchiveInfo(nn::util::optional<SaveDataArchiveInfo>* pOutValue, const NsaIdToken& nsaIdToken, SaveDataArchiveId sdaId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetSaveDataArchiveInfoListTablePath();
    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    *pOutValue = sdaInfoList.FindIf([&sdaId](const SaveDataArchiveInfo& r) {
        if(r.status == SaveDataArchiveStatus::Fixed && r.id == sdaId)
        {
            return true;
        }
        return false;
    });
    NN_RESULT_SUCCESS;
}

Result RequestSaveDataArchiveInfoList(database::SaveDataArchiveInfoCache& sdaInfoCache, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetSaveDataArchiveInfoListTablePath();
    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    nn::Result result = ResultSuccess();
    sdaInfoList.ForEach([&](const SaveDataArchiveInfo& r) {
        if(r.status == SaveDataArchiveStatus::Uploading || r.status == SaveDataArchiveStatus::Fixed)
        {
            result = sdaInfoCache.Add(r);
        }
        return result.IsSuccess();
    });
    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

Result RequestFixedSaveDataArchiveInfo(nn::util::optional<SaveDataArchiveInfo>* pOutValue, const NsaIdToken& nsaIdToken, ApplicationId appId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetSaveDataArchiveInfoListTablePath();
    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    // TODO : NsaId でも検索する
    *pOutValue = sdaInfoList.FindIf([&appId](const SaveDataArchiveInfo& r) {
        if(r.status == SaveDataArchiveStatus::Fixed && r.applicationId == appId)
        {
            return true;
        }
        return false;
    });
    NN_RESULT_SUCCESS;
}


Result RequestComponentFileInfoList(int* pOutCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetComponentFileInfoListTablePath();
    CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(workBuffer);
    CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
    std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);

    size_t cfNum = 0;
    nn::Result result = ResultSuccess();
    cfInfoList.ForEach([&](const ComponentFileInfo& s) {
        if (cfNum < listCount)
        {
            if(s.sdaId == id)
            {
                memcpy(&componentFileList[cfNum++], &s, sizeof(ComponentFileInfo));
            }
            return true;
        }
        result = ResultInvalidArgument();
        return false;
    });
    NN_RESULT_DO(result);
    *pOutCount = static_cast<int>(cfNum);
    NN_RESULT_SUCCESS;
}

Result RequestStartDownloadSaveDataArchive(int* pOutCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    return RequestComponentFileInfoList(pOutCount, componentFileList, listCount, sdaId, nsaIdToken, curlHandle, workBuffer, workBufferSize, pCancelable);
}

Result RequestDownloadComponentFile(size_t* pOutSize, void* outBuffer, size_t bufferSize, const char* path, size_t downloadSize, CURL* curlHandle, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);
    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    NN_RESULT_DO(ReadFileAtOnce(pOutSize, path, outBuffer, downloadSize));
    NN_RESULT_SUCCESS;
}

Result RequestImportComponentFile(fs::ISaveDataChunkImporter* importer, const char* path, size_t importSize, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(importSize);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    fs::FileHandle handle;
    NN_RESULT_DO(fs::OpenFile(&handle, path, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(handle);
    };

    int64_t offset = 0;
    while(NN_STATIC_CONDITION(true))
    {
        if(pCancelable)
        {
            NN_RESULT_THROW_UNLESS(!pCancelable->IsCanceled(), ResultCanceledDuringDataAccess());
        }
        size_t readSize;
        NN_RESULT_DO(fs::ReadFile(&readSize, handle, offset, workBuffer, workBufferSize, fs::ReadOption()));

        if (readSize > 0)
        {
            NN_RESULT_DO(importer->Push(workBuffer, readSize));
        }
        else
        {
            break;
        }

        offset += readSize;
    }
    NN_RESULT_SUCCESS;
}

Result RequestFinishDownloadSaveDataArchive(SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 特に何もしない
    NN_RESULT_SUCCESS;
}

Result DeleteSaveDataArchive(SaveDataArchiveId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(workBufferSize);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    // TODO : フォルダは uid ではなく NsaId で行う。NsaId は NsaIdTokenCache を Base64 でデコード後 Json パースすれば取得可能
    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    std::string table = GetSaveDataArchiveInfoListTablePath();
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    // テーブルから削除
    // サーバー版と挙動を揃える必要があれば、見つからない場合 http::ResultHttpStatusNotFound
    NN_RESULT_DO(sdaInfoList.RemoveIf([&id](const SaveDataArchiveInfo& s) {
            return s.id == id;
        }));

    // 実体削除
    std::string filePath = GetSaveDataArchivePath(id);
    NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(filePath.c_str()));
    NN_RESULT_SUCCESS;
}

Result GetKeySeedPackage(fs::SaveDataTransferManagerVersion2::KeySeedPackage* pOutKsp, SaveDataArchiveId sdaId, const fs::SaveDataTransferManagerForCloudBackUp::Challenge& challenge, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // 一旦でっち上げる
    // TODO : fs の ksp 権限チェック対応までに、ごまかし方を検討する
    NN_UNUSED(sdaId);
    NN_UNUSED(challenge);
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(workBufferSize);
    NN_UNUSED(pCancelable);
    memset(pOutKsp, 0x00, fs::SaveDataTransferManagerVersion2::KeySeedPackage::Size);
    NN_RESULT_SUCCESS;
}

Result GetComponentFileSignedUrlForDownload(ComponentFileInfo* pOut, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    std::string tablePath = GetComponentFileInfoListTablePath();
    CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(workBuffer);
    CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
    std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);
    auto targetCf = cfInfoList.FindIf([&id](const ComponentFileInfo& s) {
        return s.id == id;
        });
    // 真面目にやるなら 404 を返す
    NN_ABORT_UNLESS(targetCf);
    *pOut = *targetCf;
    NN_RESULT_SUCCESS;
}

Result UpdateComponentFile(ComponentFileInfo* out, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(workBuffer);
    CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
    std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);

    auto cf = cfInfoList.FindIf([&id](const ComponentFileInfo& c) {
        if(c.status == ComponentFileStatus::HandOver && c.id == id)
        {
            return true;
        }
        return false;
    });
    NN_ABORT_UNLESS(cf);

    cf->status = ComponentFileStatus::Uploading;
    NN_RESULT_DO(
        cfInfoList.ReplaceIf([&cf](const ComponentFileInfo& c) {
        return c.id == cf->id;
        }, *cf));

    NN_RESULT_SUCCESS;
}

// モノは使いまわしつつステートだけ差分用
Result StartSaveDataArchiveDifferentialUpload(SaveDataArchiveInfo* pOutSda, int* pOutCfCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId id, const SeriesInfo& info, size_t dataSize, time::PosixTime updatedTime, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_UNUSED(nsaIdToken);
    NN_UNUSED(curlHandle);
    NN_UNUSED(pCancelable);

    auto readMount = g_HostMountManager.Acquire();
    NN_UTIL_SCOPE_EXIT
    {
        g_HostMountManager.Release(readMount, util::ReferenceMode::Read);
    };

    SdaInfoList::ReadBuffer* pSdaInfoListBuffer = reinterpret_cast<SdaInfoList::ReadBuffer*>(workBuffer);
    SdaInfoList sdaInfoList(pSdaInfoListBuffer, g_HostMountManager);
    std::lock_guard<SdaInfoList> sdaInfoListLock(sdaInfoList);

    // Sda を検索
    auto sda = sdaInfoList.FindIf([&id](const SaveDataArchiveInfo& s) {
        if(s.status == SaveDataArchiveStatus::Fixed && s.id == id)
        {
            return true;
        }
        return false;
    });
    NN_ABORT_UNLESS(sda);

    // fixed -> uploading
    sda->status = SaveDataArchiveStatus::Uploading;
    NN_RESULT_DO(
        sdaInfoList.ReplaceIf([&sda](const SaveDataArchiveInfo& s) {
           return s.id == sda->id;
        }, *sda));

    *pOutSda = *sda;

    NN_RESULT_DO(RequestComponentFileInfoList(pOutCfCount, componentFileList, listCount, id, nsaIdToken, curlHandle, workBuffer, workBufferSize, pCancelable));

    CfInfoList::ReadBuffer* pCfInfoListBuffer = reinterpret_cast<CfInfoList::ReadBuffer*>(workBuffer);
    CfInfoList cfInfoList(pCfInfoListBuffer, g_HostMountManager);
    std::lock_guard<CfInfoList> cfInfoListLock(cfInfoList);

    for(int i = 0; i < *pOutCfCount; i++)
    {
        // CF 検索
        auto cfId = componentFileList[i].id;
        auto cf = cfInfoList.FindIf([&cfId](const ComponentFileInfo& c) {
            if(c.status == ComponentFileStatus::Fixed && c.id == cfId)
            {
                return true;
            }
            return false;
        });
        NN_ABORT_UNLESS(cf);

        // fixed -> hand_over
        cf->status = ComponentFileStatus::HandOver;
        NN_RESULT_DO(
            cfInfoList.ReplaceIf([&cf](const ComponentFileInfo& c) {
            return c.id == cf->id;
            }, *cf));
    }
    NN_RESULT_SUCCESS;
}

Result FinishSaveDataArchiveDifferentialUpload(SaveDataArchiveId id, const srv::InitialDataMac& mac, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    srv::KeySeed ks = {};
    return FinishSaveDataArchiveUpload(id, ks, mac, nsaIdToken, curlHandle, workBuffer, workBufferSize, pCancelable);
}

Result GetPolicyInfo(PolicyInfo* pOut, ApplicationId appId, uint32_t version, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    pOut->type = PolicyType::AllOk;
    NN_RESULT_SUCCESS;
}

Result ExtendSaveDataArchiveUploadTimeout(SaveDataArchiveInfo* pOutSda, int* pOutCfCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    nn::util::optional<SaveDataArchiveInfo> targetSda;
    NN_RESULT_DO(RequestSaveDataArchiveInfo(&targetSda, nsaIdToken, sdaId, curlHandle, workBuffer, workBufferSize, pCancelable));
    // 真面目にやるなら 404
    NN_ABORT_UNLESS(targetSda);
    *pOutSda = *targetSda;
    NN_RESULT_DO(RequestComponentFileInfoList(pOutCfCount, componentFileList, listCount, sdaId, nsaIdToken, curlHandle, workBuffer, workBufferSize, pCancelable));
    NN_RESULT_SUCCESS;
}

Result GetComponentFileSignedUrlForUpload(ComponentFileInfo* pOut, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    return GetComponentFileSignedUrlForDownload(pOut, id, nsaIdToken, curlHandle, workBuffer, workBufferSize, pCancelable);
}


}}}} //namespace nn::olsc::srv::transfer

