﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/olsc/srv/transfer/olsc_TransferUtil.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>

#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_JsonErrorMap.h>

#include <nn/http/stream/http_CurlInputStream.h>
#include <nn/http/stream/http_ClientCertificate.h>

#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Base64.h>

#include "../adaptor/olsc_OutputStream.h"
#include "../adaptor/olsc_SaveDataArchiveInfoAdaptor.h"
#include "../adaptor/olsc_ComponentFileInfoAdaptor.h"
#include "../adaptor/olsc_KeySeedPackageAdaptor.h"

namespace nn { namespace olsc { namespace srv { namespace transfer {

namespace
{
    static const size_t MaxSameAppIdSdaCount = 2;
    static const size_t JsonStringLength = 1024u;

    Result SetupStreamForGetImpl(http::stream::CurlInputStream& stream, curl_slist** headers, const char* url, const NsaIdToken& nsaIdToken, void* workBuffer, size_t workBufferSize)
    {
        struct Buffer
        {
            union
            {
                char workBuffer[RequiredBufferSizeForAuthorizationHeader];
                struct
                {
                    char clientCertBuffer[DeviceCertBufferSizeMax];
                    char stringBuffer[IoBufferSizeMin];
                    char inputBuffer[IoBufferSizeMin];
                } io;
            } u;
        };

        NN_SDK_REQUIRES(workBufferSize >= sizeof(Buffer));
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<Buffer>::value == 0);

        NN_RESULT_DO(stream.Initialize());

        // リクエストの作成
        auto* buffer = reinterpret_cast<Buffer*>(workBuffer);
        stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
        stream.SetInputBuffer(buffer->u.io.inputBuffer, workBufferSize - offsetof(Buffer, u.io.inputBuffer));

        // Authorization ヘッダの準備
        CreateAuthorizationHeader(buffer->u.workBuffer, sizeof(buffer->u.workBuffer), nsaIdToken);
        *headers = curl_slist_append(*headers, buffer->u.workBuffer);

        // リクエスト生成
        NN_RESULT_DO(stream.SetHeaders(*headers));

        // SSL / URL設定
        http::stream::CertIoBuffer certBuffer = {buffer->u.io.clientCertBuffer, sizeof(buffer->u.io.clientCertBuffer)};
        NN_RESULT_DO(stream.SetSslContextHandler(http::stream::SslCtxHandlerWithClientCert, &certBuffer));

        NN_RESULT_DO(stream.SetUrl(url));
        NN_RESULT_SUCCESS;
    }

    Result SetupStreamForPostImpl(http::stream::CurlInputStream& stream, curl_slist** headers, const NsaIdToken& nsaIdToken, const char* url, const char* postData, void* workBuffer, size_t workBufferSize)
    {
        struct Buffer
        {
            union
            {
                char workBuffer[RequiredBufferSizeForAuthorizationHeader];
                char postDataBuffer[JsonStringLength];
                struct
                {
                    char clientCertBuffer[DeviceCertBufferSizeMax];
                    char stringBuffer[IoBufferSizeMin * 2];
                    char inputBuffer[IoBufferSizeMin * 2];
                } io;
            } u;
        };

        NN_SDK_REQUIRES(workBufferSize >= sizeof(Buffer));
        NN_SDK_REQUIRES_NOT_NULL(workBuffer);
        NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(workBuffer) % std::alignment_of<Buffer>::value == 0);

        NN_RESULT_DO(stream.Initialize());
        auto* buffer = reinterpret_cast<Buffer*>(workBuffer);
        stream.SetStringBuffer(buffer->u.io.stringBuffer, sizeof(buffer->u.io.stringBuffer));
        stream.SetInputBuffer(buffer->u.io.inputBuffer, workBufferSize - offsetof(Buffer, u.io.inputBuffer));

        // Authorization ヘッダの準備
        CreateAuthorizationHeader(buffer->u.workBuffer, sizeof(buffer->u.workBuffer), nsaIdToken);
        *headers = curl_slist_append(*headers, buffer->u.workBuffer);
        NN_SDK_ASSERT(*headers != nullptr);
        *headers = curl_slist_append(*headers, "Content-Type: application/json");
        NN_SDK_ASSERT(*headers != nullptr);
        // リクエスト生成
        NN_RESULT_DO(stream.SetHeaders(*headers));

        // POST データ設定
        auto l = nn::util::Strlcpy(buffer->u.postDataBuffer, postData, sizeof(buffer->u.postDataBuffer));
        NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(buffer->u.postDataBuffer));
        NN_UNUSED(l);
        NN_RESULT_DO(stream.SetHttpPost(buffer->u.postDataBuffer, false));

        // SSL / URL設定
        http::stream::CertIoBuffer certBuffer = {buffer->u.io.clientCertBuffer, sizeof(buffer->u.io.clientCertBuffer)};
        NN_RESULT_DO(stream.SetSslContextHandler(http::stream::SslCtxHandlerWithClientCert, &certBuffer));

        NN_RESULT_DO(stream.SetUrl(url));
        NN_RESULT_SUCCESS;
    }
}

Result GetImpl(nn::olsc::srv::adaptor::AdaptorBase<8,128>& adaptor, const NsaIdToken& nsaIdToken, CURL* curlHandle, const char* url, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };
    http::stream::CurlInputStream stream(curlHandle, pCancelable);

    NN_RESULT_DO(SetupStreamForGetImpl(stream, &headers, url, nsaIdToken, workBuffer, workBufferSize));
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    NN_RESULT_DO(adaptor.GetResult());
    NN_RESULT_DO(stream.GetResult());

    NN_RESULT_SUCCESS;
}

Result PostImpl(nn::olsc::srv::adaptor::AdaptorBase<8,128>& adaptor, const NsaIdToken& nsaIdToken, const char* url, const char* postData, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    struct curl_slist *headers = nullptr;
    NN_UTIL_SCOPE_EXIT
    {
        curl_slist_free_all(headers);
    };
    http::stream::CurlInputStream stream(curlHandle, pCancelable);

    NN_RESULT_DO(SetupStreamForPostImpl(stream, &headers, nsaIdToken, url, postData, workBuffer, workBufferSize));
    NN_RESULT_DO(stream.Open());
    NN_RESULT_DO(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, stream, pCancelable));
    NN_RESULT_DO(adaptor.GetResult());
    NN_RESULT_DO(stream.GetResult());

    NN_RESULT_SUCCESS;
}

Result RequestSaveDataArchiveInfoList(int* pOutCount, SaveDataArchiveInfo saveDataArchiveList[], size_t listCount, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestSaveDataArchiveInfoList(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(saveDataArchiveList, static_cast<int>(listCount));
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(GetImpl(adaptor, nsaIdToken, curlHandle, url, workBuffer, workBufferSize, pCancelable));
    *pOutCount = outputStream.GetCount();
    NN_RESULT_SUCCESS;
}

// NSA, SDA Info Cache 指定 ... fixed のみ N 個 (N >= 0)
Result RequestSaveDataArchiveInfoList(database::SaveDataArchiveInfoCache& sdaInfoCache, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestSaveDataArchiveInfoList(url, sizeof(url));
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);
    adaptor::SaveDataArchiveInfoStorageOutputStream outputStream(sdaInfoCache);
    outputStream.SetExtractStatus(SaveDataArchiveStatus::Fixed);
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(GetImpl(adaptor, nsaIdToken, curlHandle, url, workBuffer, workBufferSize, pCancelable));
    NN_RESULT_SUCCESS;
}

// NSA, App ID, ... 0 ~ 2 個の結果を返す
Result RequestSaveDataArchiveInfoList(int* pOutCount, SaveDataArchiveInfo saveDataArchiveList[], size_t listCount, const NsaIdToken& nsaIdToken, ApplicationId appId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(listCount >= MaxSameAppIdSdaCount);

    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestSaveDataArchiveInfoListWithApplicationId(url, sizeof(url), appId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    SaveDataArchiveInfo sdaInfoList[MaxSameAppIdSdaCount];

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(sdaInfoList, MaxSameAppIdSdaCount); // 最大 2 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(GetImpl(adaptor, nsaIdToken, curlHandle, url, workBuffer, workBufferSize, pCancelable));

    std::memcpy(saveDataArchiveList, sdaInfoList, sizeof(SaveDataArchiveInfo) * outputStream.GetCount());
    *pOutCount = outputStream.GetCount();
    NN_RESULT_SUCCESS;
}

// NSA, SDA ID 指定 ...1 個 or エラー (HTTP 404)
Result RequestSaveDataArchiveInfo(nn::util::optional<SaveDataArchiveInfo>* pOutValue, const NsaIdToken& nsaIdToken, SaveDataArchiveId sdaId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestSaveDataArchiveInfoListWithId(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    SaveDataArchiveInfo sda;

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(&sda, 1); // 最大 1 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);

    bool notFound = false;

    NN_RESULT_TRY(GetImpl(adaptor, nsaIdToken, curlHandle, url, workBuffer, workBufferSize, pCancelable));
        NN_RESULT_CATCH(http::ResultHttpStatusNotFound)
        {
            notFound = true;
        }
    NN_RESULT_END_TRY

    if(notFound || outputStream.GetCount() == 0)
    {
        *pOutValue = nn::util::nullopt;
    }
    else
    {
        NN_SDK_ASSERT(outputStream.GetCount() == 1);
        *pOutValue = sda;
    }
    NN_RESULT_SUCCESS;
}

Result RequestFixedSaveDataArchiveInfo(nn::util::optional<SaveDataArchiveInfo>* pOutValue, const NsaIdToken& nsaIdToken, ApplicationId appId, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestSaveDataArchiveInfoListWithApplicationId(url, sizeof(url), appId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    SaveDataArchiveInfo sdaInfoList[MaxSameAppIdSdaCount];

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(sdaInfoList, MaxSameAppIdSdaCount); // 最大 2 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(GetImpl(adaptor, nsaIdToken, curlHandle, url, workBuffer, workBufferSize, pCancelable));

    // サーバー上の  Fixed の Sda を探す
    for(int i = 0; i < outputStream.GetCount(); i++)
    {
        // Fixed が見つかったら終了
        if(sdaInfoList[i].status == SaveDataArchiveStatus::Fixed)
        {
            *pOutValue = sdaInfoList[i];
            NN_RESULT_SUCCESS;
        }
    }
    *pOutValue = nn::util::nullopt;
    NN_RESULT_SUCCESS;
}

Result RequestComponentFileInfoList(int* pOutCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestSaveDataArchiveInfoListWithId(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::ComponentFileInfoMemoryOutputStream outputStream(componentFileList, static_cast<int>(listCount));
    adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(GetImpl(adaptor, nsaIdToken, curlHandle, url, workBuffer, workBufferSize, pCancelable));

    *pOutCount = outputStream.GetCount();
    NN_RESULT_SUCCESS;
}

Result RequestStartDownloadSaveDataArchive(int* pOutCount, ComponentFileInfo componentFileList[], size_t listCount, SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestStartDownloadSaveDataArchive(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::ComponentFileInfoMemoryOutputStream outputStream(componentFileList, static_cast<int>(listCount));
    adaptor::ComponentFileInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, "", curlHandle, workBuffer, workBufferSize, pCancelable));

    *pOutCount = outputStream.GetCount();
    NN_RESULT_SUCCESS;
}

Result RequestFinishDownloadSaveDataArchive(SaveDataArchiveId sdaId, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForRequestFinishDownloadSaveDataArchive(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    SaveDataArchiveInfo sda;

    adaptor::SaveDataArchiveInfoMemoryOutputStream outputStream(&sda, 1); // 最大 1 つ
    adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);
    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, "", curlHandle, workBuffer, workBufferSize, pCancelable));

    // finish_download でも sda がサーバーから返ってくるがこの sda は使用予定なしなので一応 sdaId の一致確認だけして終わり
    NN_SDK_ASSERT(sda.id == sdaId);

    NN_RESULT_SUCCESS;
}

Result GetKeySeedPackage(fs::SaveDataTransferManagerVersion2::KeySeedPackage* pOutKsp, SaveDataArchiveId sdaId, const fs::SaveDataTransferManagerForCloudBackUp::Challenge& challenge, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT
{
    // URL の作成
    char url[256];
    auto l = CreateUrlForGetKeySeedPackage(url, sizeof(url), sdaId);
    NN_SDK_ASSERT(static_cast<uint32_t>(l) < sizeof(url));
    NN_UNUSED(l);

    adaptor::KeySeedPackageAdaptor adaptor;

    static const size_t Base64Size = ((fs::SaveDataTransferManagerForCloudBackUp::Challenge::Size * 4 + (3 - 1) ) / 3 ) + 2;
    char b64Value[Base64Size];
    auto b64Error = nn::util::Base64::ToBase64String(b64Value, sizeof(b64Value), challenge.data, sizeof(challenge.data), nn::util::Base64::Mode_UrlSafe);
    NN_ABORT_UNLESS(b64Error == nn::util::Base64::Status_Success);

    char jsonString[512];
    l = nn::util::TSNPrintf(jsonString, sizeof(jsonString),
        "{"
             "\"encoded_challenge\": \"%s\""
        "}"
        , b64Value
    );

    NN_RESULT_DO(PostImpl(adaptor, nsaIdToken, url, jsonString, curlHandle, workBuffer, workBufferSize, pCancelable));
    fs::SaveDataTransferManagerVersion2::KeySeedPackage keySeedPackage;
    NN_RESULT_DO(adaptor.GetKeySeedPackage(&keySeedPackage));

    *pOutKsp = keySeedPackage;

    NN_RESULT_SUCCESS;
}

}}}} //namespace nn::olsc::srv::transfer

