﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mem/mem_NumberLineAllocator.h>
#include <nn/sf/sf_DefaultAllocationPolicy.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/sfdl/olsc_IOlscService.sfdl.h>
#include <nn/olsc/srv/database/olsc_PolicyInfoCache.h>
#include <nn/olsc/srv/olsc_AsyncResultImpl.h>
#include <nn/olsc/srv/olsc_DaemonControllerImpl.h>
#include <nn/olsc/srv/olsc_PolicyInfoTask.h>
#include <nn/olsc/srv/olsc_RemoteStorageControllerImpl.h>
#include <nn/olsc/srv/olsc_SaveDataArchiveInfoCacheManager.h>
#include <nn/olsc/srv/olsc_Service.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/olsc_TransferTaskListControllerImpl.h>
#include <nn/olsc/srv/olsc_SeriesInfoDatabaseManager.h>
#include <nn/olsc/srv/transfer/olsc_PolicyInfoDownload.h>
#include <nn/olsc/srv/util/olsc_ApplicationVersion.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace olsc { namespace srv {

namespace {

    DataTransferPolicy ConvertToDataTransferPolicy(const PolicyInfo& policyInfo) NN_NOEXCEPT
    {
        switch (policyInfo.type)
        {
            case PolicyType::AllOk:
                {
                    DataTransferPolicy dp;
                    dp.downloadRestriction = DataDownloadRestriction::None;
                    dp.uploadRestriction = DataUploadRestriction::None;
                    return dp;
                }
            case PolicyType::AllNg:
                {
                    DataTransferPolicy dp;
                    dp.downloadRestriction = DataDownloadRestriction::Restricted;
                    dp.uploadRestriction = DataUploadRestriction::Restricted;
                    return dp;
                }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
}

Service::Service(
    MemoryResource& objectMemoryResource,
    RemoteStorageControllerImpl& remoteStorageController,
    TransferTaskListControllerImpl& transferTaskListController,
    DaemonControllerImpl& daemonController,
    SeriesInfoDatabaseManager& seriesInfoDatabaseManager,
    SaveDataArchiveInfoCacheManager& sdaInfoCacheManager,
    database::ErrorHistoryDatabase& errorHistoryDatabase,
    database::PolicyInfoCache& policyInfoCache,
    TransferTaskAgent& transferTaskAgent,
    Executor& executor) NN_NOEXCEPT
    : m_ObjectMemoryResource(objectMemoryResource)
    , m_RemoteStorageController(remoteStorageController)
    , m_TransferTaskListController(transferTaskListController)
    , m_DaemonController(daemonController)
    , m_TransferTaskAgent(transferTaskAgent)
    , m_Executor(executor)
    , m_SeriesInfoDatabaseManager(seriesInfoDatabaseManager)
    , m_SdaInfoCacheManager(sdaInfoCacheManager)
    , m_ErrorHistoryDatabase(errorHistoryDatabase)
    , m_PolicyInfoCache(policyInfoCache)
{
}

Result Service::GetTransferTaskListController(sf::Out<sf::SharedPointer<ITransferTaskListController>> outValue) NN_NOEXCEPT
{
    *outValue = sf::CreateSharedObjectWithoutManagement<ITransferTaskListController, TransferTaskListControllerImpl>(&m_ObjectMemoryResource, &m_TransferTaskListController);
    NN_RESULT_SUCCESS;
}

Result Service::GetRemoteStorageController(sf::Out<sf::SharedPointer<IRemoteStorageController>> outValue) NN_NOEXCEPT
{
    *outValue = sf::CreateSharedObjectWithoutManagement<IRemoteStorageController, RemoteStorageControllerImpl>(&m_ObjectMemoryResource, &m_RemoteStorageController);
    NN_RESULT_SUCCESS;
}

Result Service::GetDaemonController(sf::Out<sf::SharedPointer<IDaemonController>> outValue) NN_NOEXCEPT
{
    *outValue = sf::CreateSharedObjectWithoutManagement<IDaemonController, DaemonControllerImpl>(&m_ObjectMemoryResource, &m_DaemonController);
    NN_RESULT_SUCCESS;
}

Result Service::ListTransferTaskErrorInfo(const sf::OutArray<nn::olsc::TransferTaskErrorInfo>& outValue, sf::Out<int32_t> outCount, int32_t offset) const NN_NOEXCEPT
{
    *outCount = m_ErrorHistoryDatabase.ListLastErrorInfoAsTransferTaskErrorInfo(outValue.GetData(), static_cast<int>(outValue.GetLength()), offset);
    NN_RESULT_SUCCESS;
}

Result Service::GetTransferTaskErrorInfoCount(sf::Out<int32_t> outValue) const NN_NOEXCEPT
{
    outValue.Set(m_ErrorHistoryDatabase.GetLastErrorInfoCount());
    NN_RESULT_SUCCESS;
}

Result Service::DeleteAllTransferTask() NN_NOEXCEPT
{
    m_TransferTaskAgent.Stop();
    NN_UTIL_SCOPE_EXIT{
        m_TransferTaskAgent.Start();
    };

    NN_RESULT_DO(m_TransferTaskListController.DeleteAllTransferTask());

    NN_RESULT_SUCCESS;
}

Result Service::DeleteAllSeriesInfo(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto writeMount = m_SeriesInfoDatabaseManager.AcquireWriteMount(uid);
    NN_RESULT_DO(m_SeriesInfoDatabaseManager.Acquire(uid)->Clear());
    NN_RESULT_DO(writeMount.Commit());

    NN_RESULT_SUCCESS;
}

Result Service::DeleteAllSdaInfoCache(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
    NN_RESULT_DO(m_SdaInfoCacheManager.Acquire(uid)->DeleteAll());
    NN_RESULT_DO(writeMount.Commit());

    NN_RESULT_SUCCESS;
}

Result Service::DeleteAllApplicationSetting(const nn::account::Uid& uid) NN_NOEXCEPT
{
    m_DaemonController.DeleteAllApplicationSetting(uid);
    NN_RESULT_SUCCESS;
}

Result Service::RemoveTransferTaskErrorInfo(const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    auto writeMount = m_ErrorHistoryDatabase.AcquireWriteMount();

    m_ErrorHistoryDatabase.RemoveLastErrorInfo(uid, appId);

    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result Service::DeleteAllTransferTaskErrorInfo() NN_NOEXCEPT
{
    auto writeMount = m_ErrorHistoryDatabase.AcquireWriteMount();

    m_ErrorHistoryDatabase.Cleanup();

    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result Service::RegisterTransferTaskErrorInfo(const TransferTaskErrorInfo& errorInfo) NN_NOEXCEPT
{
    auto writeMount = m_ErrorHistoryDatabase.AcquireWriteMount();

    NN_ABORT_UNLESS(m_ErrorHistoryDatabase.SetLastError(errorInfo));

    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result Service::AddSaveDataArchiveInfoCache(const account::Uid& uid, const SaveDataArchiveInfo& sda) NN_NOEXCEPT
{
    auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
    NN_RESULT_DO(m_SdaInfoCacheManager.Acquire(uid)->Add(sda));
    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result Service::DeleteSeriesInfo(const nn::account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    auto writeMount = m_SeriesInfoDatabaseManager.AcquireWriteMount(uid);
    NN_RESULT_DO(m_SeriesInfoDatabaseManager.Acquire(uid)->Remove(appId));
    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result Service::GetSeriesInfo(const sf::Out<SeriesInfo>& outValue, const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    auto si = m_SeriesInfoDatabaseManager.Acquire(uid)->Get(appId);
    outValue.Set(si);
    NN_RESULT_SUCCESS;
}

Result Service::GetDataTransferPolicy(sf::Out<DataTransferPolicy> outValue, ApplicationId appId) NN_NOEXCEPT
{
    auto policyInfo = m_PolicyInfoCache.Get(appId);
    NN_RESULT_THROW_UNLESS(policyInfo
        && policyInfo->launchRequiredVersion == util::GetApplicationLaunchRequiredVersion(appId)
        && policyInfo->latestVersion == util::GetApplicationLatestVersion(appId), ResultPolicyInfoCacheNoLongerExists());

    outValue.Set(ConvertToDataTransferPolicy(policyInfo->policyInfo));
    NN_RESULT_SUCCESS;
}

Result Service::DeleteDataTransferPolicy(ApplicationId appId) NN_NOEXCEPT
{
    auto writeMount = m_PolicyInfoCache.AcquireWriteMount();
    m_PolicyInfoCache.Remove(appId);
    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result Service::RequestUpdateDataTransferPolicyAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outValue, ApplicationId appId) NN_NOEXCEPT
{
    auto latestVersion = util::GetApplicationLatestVersion(appId);
    auto launchRequiredVersion = util::GetApplicationLaunchRequiredVersion(appId);
    auto p = sf::CreateSharedObjectEmplaced<IAsyncResult, AsyncResultImpl<UpdateDataTransferPolicyTask>>(&m_ObjectMemoryResource, appId, latestVersion, launchRequiredVersion, m_PolicyInfoCache);
    NN_RESULT_THROW_UNLESS(p, olsc::ResultOutOfObjectMemoryResource());
    NN_RESULT_DO(p.GetImpl().Initialize(m_Executor));
    *outValue = std::move(p);
    NN_RESULT_SUCCESS;
}

Result Service::RunAutonomyUploadTransferTaskRegistration(const account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW(olsc::ResultNotImplemented());
}

}}} //namespace nn::olsc::srv
