﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/fs/fs_Context.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/olsc/srv/olsc_AsyncResultImpl.h>
#include <nn/olsc/srv/olsc_Executor.h>
#include <nn/olsc/srv/olsc_RemoteStorageControllerImpl.h>
#include <nn/olsc/srv/olsc_SeriesInfoDatabaseManager.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveDownload.h>
#include <nn/olsc/srv/transfer/olsc_SaveDataArchiveUpload.h>
#include <nn/olsc/srv/util/olsc_Account.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/olsc/srv/util/olsc_SeriesPosition.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_Execution.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace olsc { namespace srv {

namespace
{
    struct Buffer
    {
        NsaIdToken token;
        std::array<char, 128 * 1024> ioBuffer;
    };

    Result CheckUidAvailability(const account::Uid& uid) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(uid != account::InvalidUid, ResultInvalidUser());
        bool isExist = false;
        NN_RESULT_DO(account::GetUserExistence(&isExist, uid));
        NN_RESULT_THROW_UNLESS(isExist, ResultInvalidUser());
        NN_RESULT_SUCCESS;
    }

    Result ClearDataInfoCache(const account::Uid& uid, SaveDataArchiveInfoCacheManager& sdaInfoCacheManager) NN_NOEXCEPT
    {
        auto writeMount = sdaInfoCacheManager.AcquireWriteMount(uid);
        NN_RESULT_DO(sdaInfoCacheManager.Acquire(uid)->DeleteAll());
        NN_RESULT_DO(writeMount.Commit());
        NN_RESULT_SUCCESS;
    }

    class RequestTaskBase : public nn::util::Executable
    {
    public:
        NN_IMPLICIT RequestTaskBase(const account::Uid& uid) NN_NOEXCEPT : m_Uid(uid)
        {}
    protected:
        virtual Result RequestImpl(const account::Uid& uid, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* ioBuffer, size_t ioBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT = 0;
        virtual Result ExecuteImpl(void* rawBuffer, size_t rawBufferSize) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(rawBufferSize == sizeof(ExecutionResource));
            auto er = reinterpret_cast<ExecutionResource*>(rawBuffer);

            NN_SDK_ASSERT(er->BufferSize >= sizeof(Buffer));
            auto buffer = reinterpret_cast<Buffer*>(er->buffer);

            NN_RESULT_DO(CheckUidAvailability(m_Uid));

            nn::util::Cancelable cancelable;
            size_t idTokenSize;
            NN_RESULT_DO(util::GetNsaIdToken(&idTokenSize, buffer->token.data, sizeof(buffer->token.data), m_Uid, cancelable));
            NN_RESULT_DO(RequestImpl(m_Uid, buffer->token, er->curlHandle, buffer->ioBuffer.data(), buffer->ioBuffer.size(), cancelable));
            NN_RESULT_SUCCESS;
        }
    private:
        account::Uid m_Uid;
    };

    class SaveDataArchiveInfoCacheUpdationTask : public RequestTaskBase
    {
    public:
        SaveDataArchiveInfoCacheUpdationTask(const account::Uid& uid, SaveDataArchiveInfoCacheManager& sdaInfoCacheManager) NN_NOEXCEPT
            : RequestTaskBase(uid), m_SdaInfoCacheManager(sdaInfoCacheManager)
        {}
    protected:
        virtual Result RequestImpl(const account::Uid& uid, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* ioBuffer, size_t ioBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT NN_OVERRIDE
        {
            auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
            NN_RESULT_DO(ClearDataInfoCache(uid, m_SdaInfoCacheManager));

            // TODO: 2段階で DB を更新する必要あり。下記タスクで改善予定。
            //       http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-80984
            NN_RESULT_DO(transfer::RequestSaveDataArchiveInfoList(m_SdaInfoCacheManager.Acquire(uid).Get(), nsaIdToken, curlHandle, ioBuffer, ioBufferSize, &cancelable));
            NN_RESULT_DO(writeMount.Commit());

            NN_RESULT_SUCCESS;
        }
    private:
        SaveDataArchiveInfoCacheManager& m_SdaInfoCacheManager;
    };

    class SaveDataArchiveInfoCacheUpdationTaskForSpecifiedApplication : public RequestTaskBase
    {
    public:
        SaveDataArchiveInfoCacheUpdationTaskForSpecifiedApplication(const account::Uid& uid, ApplicationId appId, SaveDataArchiveInfoCacheManager& sdaInfoCacheManager) NN_NOEXCEPT
            : RequestTaskBase(uid), m_AppId(appId), m_SdaInfoCacheManager(sdaInfoCacheManager)
        {}
    protected:
        virtual Result RequestImpl(const account::Uid& uid, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* ioBuffer, size_t ioBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::util::optional<SaveDataArchiveInfo> sdaInfo;
            NN_RESULT_DO(transfer::RequestFixedSaveDataArchiveInfo(&sdaInfo, nsaIdToken, m_AppId, curlHandle, ioBuffer, ioBufferSize, &cancelable));

            if (sdaInfo)
            {
                auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
                NN_RESULT_DO(m_SdaInfoCacheManager.Acquire(uid)->Add(*sdaInfo));
                NN_RESULT_DO(writeMount.Commit());
            }

            NN_RESULT_SUCCESS;
        }
    private:
        ApplicationId m_AppId;
        SaveDataArchiveInfoCacheManager& m_SdaInfoCacheManager;
    };

    class DeleteDataTask : public RequestTaskBase
    {
    public:
        DeleteDataTask(const account::Uid& uid, DataId dataId, SaveDataArchiveInfoCacheManager& sdaInfoCacheManager) NN_NOEXCEPT
            : RequestTaskBase(uid), m_DataId(dataId), m_SdaInfoCacheManager(sdaInfoCacheManager)
        {}
    protected:
        virtual Result RequestImpl(const account::Uid& uid, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* ioBuffer, size_t ioBufferSize, nn::util::Cancelable& cancelable) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(transfer::DeleteSaveDataArchive(m_DataId, nsaIdToken, curlHandle, ioBuffer, ioBufferSize, &cancelable));

            auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
            auto holder = m_SdaInfoCacheManager.Acquire(uid);
            NN_UTIL_LOCK_GUARD(holder.Get());

            NN_RESULT_TRY(holder->Delete(m_DataId))
                NN_RESULT_CATCH(olsc::ResultSaveDataArchiveInfoCacheNoLongerExists) {}
            NN_RESULT_END_TRY;
            NN_RESULT_DO(writeMount.Commit());

            NN_RESULT_SUCCESS;
        }
    private:
        DataId m_DataId;
        SaveDataArchiveInfoCacheManager& m_SdaInfoCacheManager;
    };


} // namespace

// -------------------------------------------------------------------------------------------

Result RemoteStorageControllerImpl::RegisterUploadSaveDataTransferTask(sf::Out<TransferTaskId> outValue, const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));

    nn::util::optional<SaveDataArchiveInfo> cachedSda;

    {
        auto sdaReadMount = m_SdaInfoCacheManager.AcquireReadMount(uid);
        auto siWriteMount = m_SeriesInfoDatabaseManager.AcquireWriteMount(uid);

        cachedSda = m_SdaInfoCacheManager.Acquire(uid)->GetSaveDataArchiveInfoByApplicationId(appId);
        if (cachedSda)
        {
            auto siHolder = m_SeriesInfoDatabaseManager.Acquire(uid);
            NN_UTIL_LOCK_GUARD(siHolder.Get());

            auto lastSi = siHolder->Get(appId);
            if (cachedSda->seriesInfo != lastSi)
            {
                NN_RESULT_DO(siHolder->Remove(appId));
                NN_RESULT_DO(siWriteMount.Commit());
            }
        }
    }

    TransferTaskConfig config = {};
    config.ulInfo.force = true;
    config.kind = TransferTaskKind::Upload;
    config.cachedSi = ((cachedSda) ? cachedSda->seriesInfo : InvalidSeriesInfo);

    TransferTaskId id;
    NN_RESULT_DO(m_Ttdm.RegisterTransferTask(&id,
        uid,
        appId,
        config,
        TransferTaskRank::Basic
    ));
    outValue.Set(id);

    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::RegisterDownloadSaveDataTransferTask(sf::Out<TransferTaskId> outValue, const account::Uid& uid, DataId dataId) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));

    auto sda = m_SdaInfoCacheManager.Acquire(uid)->GetSaveDataArchiveInfoByDataId(dataId);
    NN_RESULT_THROW_UNLESS(sda, olsc::ResultSaveDataArchiveInfoCacheNoLongerExists());

    TransferTaskConfig config = {};
    auto pSaveDataInfo = util::FindSaveData(uid, sda->applicationId);
    if (pSaveDataInfo)
    {
        fs::ScopedAutoAbortDisabler antiAbort;
        NN_RESULT_DO(fs::GetSaveDataCommitId(&config.dlInfo.currentCommitId, pSaveDataInfo->saveDataSpaceId, pSaveDataInfo->saveDataId));
        config.dlInfo.saveDataExists = true;
    }

    config.cachedSi = sda->seriesInfo;
    config.kind = TransferTaskKind::Download;

    TransferTaskId tid;
    NN_RESULT_DO(m_Ttdm.RegisterTransferTask(&tid,
        uid,
        sda->applicationId,
        config,
        TransferTaskRank::Basic));
    outValue.Set(tid);

    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::ListDataInfo(sf::Out<int32_t> outValue, const nn::sf::OutBuffer& outBuffer, int count, const account::Uid& uid, int offset) const NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));
    outValue.Set(m_SdaInfoCacheManager.Acquire(uid)->ListDataInfo(reinterpret_cast<DataInfo*>(outBuffer.GetPointerUnsafe()), count, offset));
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::DeleteDataInfo(const account::Uid& uid, DataId dataId) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));
    auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
    NN_RESULT_DO(m_SdaInfoCacheManager.Acquire(uid)->Delete(dataId));
    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::GetCount(sf::Out<int32_t> outValue, const account::Uid& uid) const NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));
    outValue.Set(m_SdaInfoCacheManager.Acquire(uid)->GetCount());
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::GetDataInfo(sf::Out<bool> outValue, sf::Out<DataInfo> outDataInfo, const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));
    auto dataInfo = m_SdaInfoCacheManager.Acquire(uid)->GetDataInfoByApplicationId(appId);
    if(dataInfo)
    {
        outDataInfo.Set(*dataInfo);
        outValue.Set(true);
    }
    else
    {
        outValue.Set(false);
    }
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::GetDataNewness(sf::Out<DataNewness> outValue, const account::Uid& uid, DataId dataId) const NN_NOEXCEPT
{
    NN_RESULT_DO(CheckUidAvailability(uid));
    auto readMountSda = m_SdaInfoCacheManager.AcquireReadMount(uid);
    auto readMountSi = m_SeriesInfoDatabaseManager.AcquireReadMount(uid);

    auto sda = m_SdaInfoCacheManager.Acquire(uid)->GetSaveDataArchiveInfoByDataId(dataId);
    NN_RESULT_THROW_UNLESS(sda, olsc::ResultSaveDataArchiveInfoCacheNoLongerExists());

    auto siHolder = m_SeriesInfoDatabaseManager.Acquire(uid);

    SeriesPosition seriesPosition;
    NN_RESULT_DO(util::EvaluateSaveDataInfoCacheSeriesInfo(&seriesPosition, uid, *sda, siHolder.Get()));
    outValue.Set(util::ConvertToDataNewness(seriesPosition));

    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::ClearDataInfoCache(const account::Uid& uid) NN_NOEXCEPT
{
    auto writeMount = m_SdaInfoCacheManager.AcquireWriteMount(uid);
    NN_RESULT_DO(m_SdaInfoCacheManager.Acquire(uid)->DeleteAll());
    NN_RESULT_DO(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::RequestUpdateDataInfoCacheAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outValue, const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto p = sf::CreateSharedObjectEmplaced<IAsyncResult, AsyncResultImpl<SaveDataArchiveInfoCacheUpdationTask>>(&m_ObjectMemoryResource, uid, m_SdaInfoCacheManager);
    NN_RESULT_THROW_UNLESS(p, olsc::ResultOutOfObjectMemoryResource());
    NN_RESULT_DO(p.GetImpl().Initialize(m_Executor));
    *outValue = std::move(p);
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::RequestUpdateDataInfoCacheOfSpecifiedApplicationAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outValue, const nn::account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
{
    auto p = sf::CreateSharedObjectEmplaced<IAsyncResult, AsyncResultImpl<SaveDataArchiveInfoCacheUpdationTaskForSpecifiedApplication>>(&m_ObjectMemoryResource, uid, appId, m_SdaInfoCacheManager);
    NN_RESULT_THROW_UNLESS(p, olsc::ResultOutOfObjectMemoryResource());
    NN_RESULT_DO(p.GetImpl().Initialize(m_Executor));
    *outValue = std::move(p);
    NN_RESULT_SUCCESS;
}

Result RemoteStorageControllerImpl::RequestDeleteData(sf::Out<sf::SharedPointer<IAsyncResult>> outValue, const nn::account::Uid& uid, DataId dataId) NN_NOEXCEPT
{
    auto p = sf::CreateSharedObjectEmplaced<IAsyncResult, AsyncResultImpl<DeleteDataTask>>(&m_ObjectMemoryResource, uid, dataId, m_SdaInfoCacheManager);
    NN_RESULT_THROW_UNLESS(p, olsc::ResultOutOfObjectMemoryResource());
    NN_RESULT_DO(p.GetImpl().Initialize(m_Executor));
    *outValue = std::move(p);
    NN_RESULT_SUCCESS;
}

}}} //namespace nn::olsc::srv

