﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/util/util_TFormatString.h>

namespace nn { namespace olsc { namespace srv { namespace database {

namespace {

} // namespace

//------------------------------------------------------------------------------
// Private
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// Public
//------------------------------------------------------------------------------

Result TransferTaskDatabaseManagerBase::PerformWriteTransaction(const WriteTransaction& transaction) NN_NOEXCEPT
{
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);

    auto writeMount = m_MountManager.AcquireDeviceSaveForWrite();

    NN_RESULT_DO(transaction(m_TaskDatabase, m_ContextDatabase));
    NN_ABORT_UNLESS_RESULT_SUCCESS(writeMount.Commit());
    NN_RESULT_SUCCESS;
}

void TransferTaskDatabaseManagerBase::PerformReadTransaction(const ReadTransaction& transaction) const NN_NOEXCEPT
{
    // TODO: read の時は shared_lock を使ってパフォーマンスを上げられる可能性がある
    std::lock_guard<os::SdkRecursiveMutex> lock(m_Lock);
    NN_ABORT_UNLESS_RESULT_SUCCESS(transaction(m_TaskDatabase, m_ContextDatabase));
}

os::SystemEvent& TransferTaskDatabaseManagerBase::GetExecutionListUpdatedEvent() NN_NOEXCEPT
{
    return m_TaskDatabase.GetExecutionListUpdatedEvent();
}

// ------------------------------------------------------------------------------------

nn::util::optional<TransferTaskDetailInfo> TransferTaskDatabaseManager::FindDetailInfo(const FindTaskPredicate& predicate) const NN_NOEXCEPT
{
    nn::util::optional<TransferTaskDetailInfo> detailInfo = nn::util::nullopt;
    auto transaction = [&](const database::TransferTaskDatabase& taskDb, const database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);

        detailInfo = taskDb.FindTaskIf(predicate);
        NN_RESULT_SUCCESS;
    };
    PerformReadTransaction(transaction);

    return detailInfo;
}

Result TransferTaskDatabaseManager::GetTransferTaskCount(uint32_t* out) const NN_NOEXCEPT
{
    auto transaction = [&](const database::TransferTaskDatabase& taskDb, const database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);
        *out = static_cast<uint32_t>(taskDb.GetTaskCount());
        NN_RESULT_SUCCESS;
    };
    PerformReadTransaction(transaction);

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::RemoveTransferTask(TransferTaskId id) NN_NOEXCEPT
{
    auto transaction = [&](TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_RESULT_DO(taskDb.RemoveTask(id));
        NN_RESULT_TRY(contextDb.RemoveTransferTaskContext(id))
            NN_RESULT_CATCH(olsc::ResultTransferTaskContextNotFound) {}
        NN_RESULT_END_TRY;
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(PerformWriteTransaction(transaction));

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::RegisterTransferTask(TransferTaskId* out, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config, TransferTaskRank rank) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);
        NN_RESULT_DO(taskDb.AppendTask(out, uid, appId, config, rank));
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(PerformWriteTransaction(transaction));

    NN_RESULT_SUCCESS;

}

nn::util::optional<database::TransferTaskContextDatabase::TransferTaskContext> TransferTaskDatabaseManager::GetTransferTaskContext(TransferTaskId id) NN_NOEXCEPT
{
    nn::util::optional<TransferTaskContext> context = nn::util::nullopt;
    auto transaction = [&](const database::TransferTaskDatabase& taskDb, const database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(taskDb);
        context = contextDb.GetTransferTaskContext(id);
        NN_RESULT_SUCCESS;
    };
    PerformReadTransaction(transaction);

    return context;
}

Result TransferTaskDatabaseManager::RemoveTransferTaskContext(TransferTaskId id) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(taskDb);
        NN_RESULT_DO(contextDb.RemoveTransferTaskContext(id));
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(PerformWriteTransaction(transaction));

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::SetCompleted(TransferTaskId id) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);
        NN_RESULT_TRY(taskDb.UpdateTaskState(id, TransferTaskDetailInfo::State::Success))
            NN_RESULT_CATCH(olsc::ResultTransferTaskNotFoundInExecutionList) {}
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    };
    NN_RESULT_TRY(PerformWriteTransaction(transaction))
        NN_RESULT_CATCH(ResultTransferTaskNotFound) {}
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::SetFailed(TransferTaskId id, Result result) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        bool needsUpdateContext = true;
        NN_RESULT_TRY(taskDb.UpdateTaskState(id, TransferTaskDetailInfo::State::Error))
            NN_RESULT_CATCH(olsc::ResultTransferTaskNotFoundInExecutionList) { needsUpdateContext = false; }
        NN_RESULT_END_TRY;

        if (needsUpdateContext)
        {
            TransferTaskContext context = {};
            context.id = id;
            context.lastResult = result;
            NN_RESULT_DO(contextDb.UpdateTransferTaskContext(context));
        }
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_TRY(PerformWriteTransaction(transaction))
        NN_RESULT_CATCH(ResultTransferTaskNotFound) {}
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::SetSuspended(TransferTaskId id, const TransferTaskContext& context) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        bool needsUpdateContext = true;
        NN_RESULT_TRY(taskDb.UpdateTaskState(id, TransferTaskDetailInfo::State::NotFinished))
            NN_RESULT_CATCH(olsc::ResultTransferTaskNotFoundInExecutionList) { needsUpdateContext = false; }
        NN_RESULT_END_TRY;

        if (needsUpdateContext)
        {
            NN_RESULT_DO(contextDb.UpdateTransferTaskContext(context));
        }
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_TRY(PerformWriteTransaction(transaction))
        NN_RESULT_CATCH(ResultTransferTaskNotFound) {}
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::SetRetry(TransferTaskId id) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_RESULT_TRY(contextDb.RemoveTransferTaskContext(id))
            NN_RESULT_CATCH(olsc::ResultTransferTaskContextNotFound) {}
        NN_RESULT_END_TRY;

        NN_RESULT_TRY(taskDb.UpdateTaskState(id, TransferTaskDetailInfo::State::NotFinished))
            NN_RESULT_CATCH(olsc::ResultTransferTaskNotFoundInExecutionList) {}
        NN_RESULT_END_TRY;

        NN_RESULT_SUCCESS;
    };
    NN_RESULT_TRY(PerformWriteTransaction(transaction))
        NN_RESULT_CATCH(ResultTransferTaskNotFound) {}
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::MoveTransferTaskToHead(TransferTaskId id) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);
        NN_RESULT_DO(taskDb.MoveTaskToHead(id));
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(PerformWriteTransaction(transaction));

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::RemoveAllTransferTask() NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_RESULT_DO(contextDb.RemoveAllTransferTaskContext());
        NN_RESULT_DO(taskDb.RemoveAllTask());
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(PerformWriteTransaction(transaction));

    NN_RESULT_SUCCESS;
}

Result TransferTaskDatabaseManager::RemoveAutoUploadTaskByUser(const account::Uid& uid) NN_NOEXCEPT
{
    auto transaction = [&](database::TransferTaskDatabase& taskDb, database::TransferTaskContextDatabase& contextDb) -> Result
    {
        auto shouldRemove = [&uid](const TransferTaskDetailInfo& di) -> bool {
            return di.uid == uid
                && di.config.kind == TransferTaskKind::Upload
                // TODO: force フラグではなくて自動 UL かどうかを表すフラグを使う
                && !di.config.ulInfo.force;
        };

        taskDb.ForEachDetailInfo(uid, [&shouldRemove, &contextDb](const TransferTaskDetailInfo& di) -> bool {
            if (shouldRemove(di))
            {
                contextDb.RemoveTransferTaskContext(di.id);
                return false;
            }
            return true;
        });

        taskDb.RemoveTaskIfMatch(uid, shouldRemove);

        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(PerformWriteTransaction(transaction));

    NN_RESULT_SUCCESS;
}

int TransferTaskDatabaseManager::ListDetailInfo(TransferTaskDetailInfo out[], int maxOutCount, int offset) NN_NOEXCEPT
{
    int outCount;
    auto transaction = [&](const database::TransferTaskDatabase& taskDb, const database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);
        outCount = taskDb.ListTasks(out, maxOutCount, offset);
        NN_RESULT_SUCCESS;
    };
    PerformReadTransaction(transaction);
    return outCount;
}

nn::util::optional<TransferTaskDetailInfo> TransferTaskDatabaseManager::GetTransferTaskDetailInfo(TransferTaskId id) NN_NOEXCEPT
{
    nn::util::optional<TransferTaskDetailInfo> detailInfo = nn::util::nullopt;
    auto transaction = [&](const database::TransferTaskDatabase& taskDb, const database::TransferTaskContextDatabase& contextDb) -> Result
    {
        NN_UNUSED(contextDb);
        detailInfo = taskDb.FindTaskIf([&](const TransferTaskDetailInfo& di) -> bool {
            return di.id == id;
        });
        NN_RESULT_SUCCESS;
    };
    PerformReadTransaction(transaction);
    return detailInfo;
}

}}}} //namespace nn::olsc::srv::database

