﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/olsc/srv/database/olsc_ExecutionList.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace olsc { namespace srv { namespace database {

//------------------------------------------------------------------------------
// Private
//------------------------------------------------------------------------------

ExecutionList::ListImpl::ListImpl(const char* prefix, TransferTaskRank rank, util::DefaultMountManager& mountManager) NN_NOEXCEPT
    : DataArray<util::DefaultMountManager, ExecutionInfoEntry, MaxTransferTaskCount, CacheCount, ReadBufferCount>(&m_CacheBuffer, &m_ReadBuffer, mountManager), m_Rank(rank)
{
    NN_ABORT_UNLESS(nn::util::TSNPrintf(m_MetadataPath, sizeof(m_MetadataPath), "ttdb_%s_el_meta", prefix) < sizeof(m_MetadataPath));
    NN_ABORT_UNLESS(nn::util::TSNPrintf(m_EntryPath, sizeof(m_EntryPath), "ttdb_%s_el_entry", prefix) < sizeof(m_EntryPath));
    NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUnsafe());
}

const char* ExecutionList::ListImpl::GetEntryFileRelativePath() const NN_NOEXCEPT
{
    return m_EntryPath;
}

const char* ExecutionList::ListImpl::GetMetadataFileRelativePath() const NN_NOEXCEPT
{
    return m_MetadataPath;
}

util::ReadMount ExecutionList::ListImpl::AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT
{
    return mountManager.AcquireDeviceSaveForRead();
}

util::WriteMount ExecutionList::ListImpl::AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT
{
    return mountManager.AcquireDeviceSaveForWrite();
}

TransferTaskRank ExecutionList::ListImpl::GetRank() const NN_NOEXCEPT
{
    return m_Rank;
}

// --------------------------------------------------------------------------

void ExecutionList::ForEachList(const std::function<bool(ListImpl& list)>& pred) NN_NOEXCEPT
{
    for (auto& target : {&m_BasicList, &m_PostponedList})
    {
        if (!pred(*target))
        {
            break;
        }
    }
}

void ExecutionList::ForEachList(const std::function<bool(const ListImpl& list)>& pred) const NN_NOEXCEPT
{
    for (auto& target : {&m_BasicList, &m_PostponedList})
    {
        if (!pred(*target))
        {
            break;
        }
    }
}

//------------------------------------------------------------------------------
// Public
//------------------------------------------------------------------------------

ExecutionList::ExecutionList(util::DefaultMountManager& mountManager) NN_NOEXCEPT
    : m_BasicList("basic", TransferTaskRank::Basic, mountManager), m_PostponedList("postponed", TransferTaskRank::Postponed, mountManager)
{
}

void ExecutionList::lock() NN_NOEXCEPT
{
    m_BasicList.lock();
    m_PostponedList.lock();
}

void ExecutionList::lock() const NN_NOEXCEPT
{
    const_cast<ExecutionList*>(this)->lock();
}

void ExecutionList::unlock() NN_NOEXCEPT
{
    m_PostponedList.unlock();
    m_BasicList.unlock();
}

void ExecutionList::unlock() const NN_NOEXCEPT
{
    const_cast<ExecutionList*>(this)->unlock();
}

nn::util::optional<ExecutionList::ExecutionInfo> ExecutionList::RemoveIf(const std::function<bool(const ExecutionInfo&)>& pred) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    nn::util::optional<ExecutionInfo> removed;
    auto remover = [&pred, &removed](ListImpl& list) -> bool {
        auto result = list.RemoveIf([&pred, &list, &removed](const ExecutionInfoEntry& entry) -> bool {
            auto ei = ExecutionInfo::Make(entry, list.GetRank());
            if (pred(ei))
            {
                removed.emplace(ei);
                return true;
            }
            return false;
        });

        if (!(result.IsSuccess() || olsc::ResultDatabaseEntryNotFound::Includes(result)))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        // 削除に成功した時点で forEach を抜けるため false を返す。
        return !result.IsSuccess();
    };

    ForEachList(remover);
    return removed;
}

void ExecutionList::RemoveAll(const std::function<bool(const ExecutionInfo&)>& pred) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    auto remover = [&pred](ListImpl& list) -> bool {
        auto result = list.RemoveAll([&pred, &list](const ExecutionInfoEntry& entry) -> bool {
            auto ei = ExecutionInfo::Make(entry, list.GetRank());
            return pred(ei);
        });

        if (!(result.IsSuccess() || olsc::ResultDatabaseEntryNotFound::Includes(result)))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        // 全リストを対象とするので forEach を抜けない。常に true を返す。
        return true;
    };

    ForEachList(remover);
}

nn::util::optional<ExecutionList::ExecutionInfo> ExecutionList::Remove(TransferTaskId tid) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    nn::util::optional<ExecutionInfo> removed;
    auto remover = [&tid, &removed](ListImpl& list) -> bool {
        auto result = list.RemoveIf([&tid, &removed, &list](const ExecutionInfoEntry& entry) -> bool {
            if (tid == entry.id)
            {
                removed.emplace(ExecutionInfo::Make(entry.id, entry.uid, list.GetRank()));
                return true;
            }
            return false;
        });

        if (!(result.IsSuccess() || olsc::ResultDatabaseEntryNotFound::Includes(result)))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        return !result.IsSuccess();
    };

    ForEachList(remover);
    return removed;
}

Result ExecutionList::Cleanup() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    auto cleanupper = [](ListImpl& list) -> bool {
        auto result = list.Cleanup();
        if (!(result.IsSuccess() || fs::ResultPathNotFound::Includes(result)))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        return true;
    };

    ForEachList(cleanupper);
    NN_RESULT_SUCCESS;
}


Result ExecutionList::PushFront(const ExecutionInfo& info) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    auto e = ExecutionInfoEntry::Make(info.id, info.uid);
    nn::util::optional<Result> result;
    ForEachList([&result, &e, &info](ListImpl& list) -> bool {
        if (list.GetRank() == info.rank)
        {
            result.emplace(list.PushFront(e));
            return false;
        }
        return true;
    });

    NN_ABORT_UNLESS(result);
    NN_RESULT_DO(*result);
    NN_RESULT_SUCCESS;
}

Result ExecutionList::PushBack(const ExecutionInfo& info) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    auto e = ExecutionInfoEntry::Make(info.id, info.uid);
    nn::util::optional<Result> result;
    ForEachList([&result, &e, &info](ListImpl& list) -> bool {
        if (list.GetRank() == info.rank)
        {
            result.emplace(list.PushBack(e));
            return false;
        }
        return true;
    });

    NN_ABORT_UNLESS(result);
    NN_RESULT_DO(*result);
    NN_RESULT_SUCCESS;
}

int ExecutionList::GetCount() const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    return m_BasicList.GetCount() + m_PostponedList.GetCount();
}

nn::util::optional<ExecutionList::ExecutionInfo> ExecutionList::FindIf(const std::function<bool(const ExecutionInfo&)>& pred) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    nn::util::optional<ExecutionInfo> found;
    auto finder = [&pred, &found](const ListImpl& list) -> bool {
        auto foundEntry = list.FindIf([&pred, &list](const ExecutionInfoEntry& entry) -> bool {
            auto ei = ExecutionInfo::Make(entry, list.GetRank());
            return pred(ei);
        });

        if (foundEntry)
        {
            found.emplace(ExecutionInfo::Make(*foundEntry, list.GetRank()));
            return false;
        }
        return true;
    };

    ForEachList(finder);
    return found;
}

bool ExecutionList::HasSpace(TransferTaskRank rank) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    nn::util::optional<bool> hasSpace;
    ForEachList([&hasSpace, rank](const ListImpl& list) -> bool {
        if (list.GetRank() == rank)
        {
            hasSpace.emplace(list.HasSpace());
            return false;
        }
        return true;
    });
    NN_ABORT_UNLESS(hasSpace);
    return *hasSpace;
}

Result ExecutionList::At(ExecutionList::ExecutionInfo* out, int index) const NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(*this);

    NN_ABORT_UNLESS(index < GetCount());

    auto toGet = index;
    Result result = olsc::ResultDatabaseEntryNotFound();
    auto getter = [&toGet, &out, &result](const ListImpl& list) -> bool {
        if (toGet < list.GetCount())
        {
            ExecutionInfoEntry entry;
            result = list.At(&entry, toGet);
            if (result.IsSuccess())
            {
                *out = ExecutionInfo::Make(entry, list.GetRank());
            }
            return false;
        }
        toGet -= list.GetCount();
        return true;
    };

    ForEachList(getter);
    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

void ExecutionList::ForEach(const std::function<bool(const ExecutionList::ExecutionInfo&)>& pred) NN_NOEXCEPT
{
    ForEachList([&pred](ListImpl& list) -> bool {
        bool needsContinue = true;
        list.ForEach([&pred, &list, &needsContinue](const ExecutionInfoEntry& e) -> bool {
            auto ei = ExecutionInfo::Make(e, list.GetRank());
            needsContinue = pred(ei);

            return needsContinue;
        });
        return needsContinue;
    });
}

}}}} //namespace nn::olsc::srv::database

