﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/service/core/ntc_SerializeTaskThread.h>
#include <nn/nn_Abort.h>

namespace nn { namespace ntc { namespace detail { namespace service { namespace core {

// TaskResourceList ---------------------------------------------------------------------------

TaskResourceList::TaskResourceList() NN_NOEXCEPT:
    m_Mutex(false),
    m_TaskEvent(nn::os::EventClearMode_ManualClear)
{
}

bool TaskResourceList::IsEmpty() const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_List.empty();
}

void TaskResourceList::ExecuteFrontTask() NN_NOEXCEPT
{
    TaskResourceNode node;
    TaskExecutor* pTaskExecutor = nullptr;

    m_Mutex.Lock();
    if(!m_List.empty())
    {
        node = m_List.front();

        // ロック解除後に node.GetTaskResourcePtr() が開放されることがあり得るのでタスク実行クラスをコピーする
        // 現時点では m_pTaskExecutor に大域的に確保したインスタンスへのポインタが格納されている前提

        pTaskExecutor  = node.GetTaskResourcePtr()->m_pTaskExecutor;

    }

    // ロック解除してからタスク実行
    m_Mutex.Unlock();

    nn::Result result = nn::ResultSuccess();
    if(pTaskExecutor)
    {
        result = pTaskExecutor->Execute(); // block API
    }
    this->Done(result, node); // このとき frontNode がすでにリストから削除されていれば、何も行われない
}

//!< 指定ノードを result で完了状態に遷移させる
bool TaskResourceList::Done(nn::Result result, TaskResourceNode& node) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto it = std::find(m_List.begin(), m_List.end(), node);
    if(it != m_List.end())
    {
        // NN_DETAIL_NTC_SERVER_LOG("Signal 0x%p\n", &(it->GetTaskResourcePtr()->m_SystemEvent));
        it->GetTaskResourcePtr()->m_Result = result; // 必ず Signal 前に Result を入れる
        it->GetTaskResourcePtr()->m_SystemEvent.Signal();

        m_List.erase(it);
        if(m_List.empty())
        {
            m_TaskEvent.Clear();
        }
        return true;
    }
    return false;
}

//!< すべてのノードを result で完了状態に遷移させる
void TaskResourceList::DoneAll(nn::Result result) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // NN_DETAIL_NTC_SERVER_LOG("DoneAll. list size: %zu\n", m_List.size());

    for(auto it = m_List.begin() ; it != m_List.end() ; ++it)
    {
        // NN_DETAIL_NTC_SERVER_LOG("Signal 0x%p\n", &(it->GetTaskResourcePtr()->m_SystemEvent));
        it->GetTaskResourcePtr()->m_Result = result; // 必ず Signal 前に Result を入れる
        it->GetTaskResourcePtr()->m_SystemEvent.Signal();
    }
    m_List.clear();
    m_TaskEvent.Clear();
}

bool TaskResourceList::PushBack(TaskResourceNode& node) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto it = std::find(m_List.begin(), m_List.end(), node);
    if(it == m_List.end())
    {
        // NN_DETAIL_NTC_SERVER_LOG("PushBack 0x%p\n", node.GetTaskResourcePtr());

        node.GetTaskResourcePtr()->m_SystemEvent.Clear(); // タスク未完了状態にする
        node.GetTaskResourcePtr()->m_Result = nn::time::ResultProcessing();

        m_List.push_back(node);
        m_TaskEvent.Signal();
        return true;
    }

    return false;
}

bool TaskResourceList::IsPushed(TaskResourceNode& node) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto it = std::find(m_List.begin(), m_List.end(), node);
    return it != m_List.end();
}

bool TaskResourceList::Remove(TaskResourceNode& node) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto it = std::find(m_List.begin(), m_List.end(), node);
    if(it != m_List.end())
    {
        // NN_DETAIL_NTC_SERVER_LOG("Remove 0x%p\n", it->GetTaskResourcePtr());
        m_List.erase(it);
        if(m_List.empty())
        {
            m_TaskEvent.Clear();
        }
        return true;
    }
    return false;
}

// SerializeTaskThread --------------------------------------------------------------

void SerializeTaskThread::ThreadFunctionImpl() NN_NOEXCEPT
{
    while(NN_STATIC_CONDITION(1))
    {
        // タスクが挿入されるまで待つ
        auto index = nn::os::WaitAny(m_StopEvent.GetBase(), m_TaskResourceList.GetTaskEvent().GetBase());
        if(index == 0)
        {
            break; // スレッド終了要求がきた
        }

        m_TaskResourceList.ExecuteFrontTask();
    }
}

SerializeTaskThread::SerializeTaskThread() NN_NOEXCEPT:
    m_StopEvent(nn::os::EventClearMode_ManualClear)
{
}

//!< スレッド開始.
void SerializeTaskThread::StartThread(char* stack, size_t stackSize, int priority, const char* threadName) NN_NOEXCEPT
{
    m_StopEvent.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&m_Thread, ThreadFunction, this, stack, stackSize, priority));
    nn::os::SetThreadNamePointer(&m_Thread, threadName);
    nn::os::StartThread(&m_Thread);
}

//! スレッド終了.登録済タスクリソースに渡す nn::Result を指定する.スレッド完了まで待機.
void SerializeTaskThread::StopThread(nn::Result remainTaskResult) NN_NOEXCEPT
{
    m_StopEvent.Signal();
    m_TaskResourceList.DoneAll(remainTaskResult);
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
}

//!< タスク登録. タスク処理終了時に taskResource.m_SystemEvent がシグナルする.
bool SerializeTaskThread::RegisterTask(TaskResourceNode& taskResourceNode) NN_NOEXCEPT
{
    // TaskResourceNode に紐づけられる TaskResource インスタンスが破棄されるときに
    // UnregisterTaskResource でリストから除く必要がある
    return m_TaskResourceList.PushBack(taskResourceNode);
}

//!< 登録済タスクリソースの削除.登録されていない場合は何もしない.
bool SerializeTaskThread::UnregisterTaskResource(TaskResourceNode& taskResourceNode) NN_NOEXCEPT
{
    return m_TaskResourceList.Remove(taskResourceNode);
}

//!< 指定タスクが登録されているかどうか
bool SerializeTaskThread::IsRegisteredTask(TaskResourceNode& taskResourceNode) NN_NOEXCEPT
{
    return m_TaskResourceList.IsPushed(taskResourceNode);
}

//!< タスクを指定の Result で終了させます.(キャンセルに使えます)
bool SerializeTaskThread::DoneTask(nn::Result doneResult, TaskResourceNode& taskResourceNode) NN_NOEXCEPT
{
    return m_TaskResourceList.Done(doneResult, taskResourceNode);
}

}}}}} // nn::ntc::detail::service::core
