﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ntc/detail/service/core/ntc_Download.h>

#include <curl/curl.h>
#include <nn/ssl/ssl_Context.h>
#include <nn/time/time_ResultPrivate.h>


namespace nn { namespace ntc { namespace detail { namespace service { namespace core {

namespace
{

    // TODO:タイムアウト時間要調整
    const long LowSpeedLimit = 1L; // 転送ビットレートは 1bps を下限とする
    const long LowSpeedTime = 10L; // LowSpeedLimit が 10 秒つづくと通信を中断する
    const long ConnectTimeout = 15L; // サーバとの接続(CONNECT)のタイムアウト (15秒)
    const long TimeoutSeconds = 30L; // 通信全体でのタイムアウト (30秒)

    // SSL検証有効化
    const long SslVerifyPeer = 1L;
    const long SslVerifyHost = 2L;
    // SSL時刻検証無効化
    const long SslVerifyDate = 0L;

    struct CurlWriteFunctionUserData
    {
        char* pWriteBuffer;
        size_t writePosition;
        size_t buffreSize;
    };

    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT;
    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT;

    nn::Result HandleHttpError(CURLcode code) NN_NOEXCEPT;

}

#define NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(handle, opt, ...) \
    do \
    { \
        CURLcode _code = curl_easy_setopt(handle, opt, __VA_ARGS__); \
        auto _result = HandleHttpError(_code); \
        if(_result.IsFailure()) \
        { \
            NN_DETAIL_NTC_SERVER_LOG("curl_easy_setopt(" #opt ") failed (%d)\n", _code); \
            NN_RESULT_THROW(_result); \
        } \
    } while (NN_STATIC_CONDITION(false))

//!< libcurl を叩いて実際にサーバーアクセスするブロック処理
nn::Result Download(
    const char* url,
    size_t* pOutDownloadSize,
    char* pDownloadBuffer,
    size_t downloadBufferSize) NN_NOEXCEPT
{
    // curl セッションインスタンスの生成
    CURL* curl = curl_easy_init();
    if (!curl)
    {
        NN_DETAIL_NTC_SERVER_LOG("curl_easy_init failed.\n");
        NN_RESULT_THROW(nn::time::ResultHttpErrorInterfaceFailed());
    }

    NN_UTIL_SCOPE_EXIT
    {
        curl_easy_cleanup(curl);
    };

    std::memset(pDownloadBuffer, 0, downloadBufferSize);
    CurlWriteFunctionUserData userData = { pDownloadBuffer, 0, downloadBufferSize };

    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_URL, url);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_WRITEFUNCTION, CurlWriteFunction);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_WRITEDATA, &userData);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_VERBOSE, 0L);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);

    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_LOW_SPEED_LIMIT, LowSpeedLimit);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_LOW_SPEED_TIME, LowSpeedTime);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_CONNECTTIMEOUT, ConnectTimeout);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_TIMEOUT, TimeoutSeconds);

    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_SSL_VERIFYPEER, SslVerifyPeer);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_SSL_VERIFYHOST, SslVerifyHost);
    NN_DETAIL_NTC_CURL_EASY_SETOPT_DO(curl, CURLOPT_SSL_VERIFYDATE, SslVerifyDate);

    // Perform the request
    CURLcode curlCode = curl_easy_perform(curl);

    if (curlCode == CURLE_OK)
    {
        // 念のため
        NN_SDK_ASSERT_GREATER_EQUAL(downloadBufferSize, userData.writePosition);
        NN_RESULT_THROW_UNLESS(downloadBufferSize >= userData.writePosition, nn::time::ResultInvalidServerResponse());

        *pOutDownloadSize = userData.writePosition;
    }
    else
    {
        NN_DETAIL_NTC_SERVER_LOG("Download failed. curlcode:%d\n", curlCode);
        NN_RESULT_THROW(HandleHttpError(curlCode));
    }

    NN_RESULT_SUCCESS;
}

namespace
{
    size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
    {
        CurlWriteFunctionUserData* userData = static_cast<CurlWriteFunctionUserData*>(userdata);

        auto count = blobsize * blobcount;
        // NN_TIME_SERVER_LOG("\n==== CurlWriteFunction count:%d userData->writePosition:%d\n", count, userData->writePosition);

        if (userData->writePosition + count <= userData->buffreSize)
        {
            memcpy(&userData->pWriteBuffer[userData->writePosition], pData, count);
            userData->writePosition += count;
        }
        else
        {
            NN_DETAIL_NTC_SERVER_LOG("At CurlWriteFunction, download buffer was overflowed. bufferSize:%zu\n", userData->buffreSize);
            return 0; // 処理を中断します
        }

        return count;
    }

    CURLcode CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
    {
        NN_UNUSED(pCurl);
        NN_UNUSED(pUserData);

        auto& context = *reinterpret_cast<nn::ssl::Context*>(pSslContext);
        if (context.Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
        {
            return CURLE_ABORTED_BY_CALLBACK;
        }
        return CURLE_OK;
    }

    nn::Result HandleHttpError(CURLcode code) NN_NOEXCEPT
    {
        if (code == CURLE_OK)
        {
            return nn::ResultSuccess();
        }

        switch (code)
        {
        case CURLE_UNSUPPORTED_PROTOCOL:
            return nn::time::ResultHttpErrorUnsupportedProtocol();
        case CURLE_FAILED_INIT:
            return nn::time::ResultHttpErrorFailedInit();
        case CURLE_URL_MALFORMAT:
            return nn::time::ResultHttpErrorUrlMalformat();
        case CURLE_NOT_BUILT_IN:
            return nn::time::ResultHttpErrorNotBuiltIn();
        case CURLE_COULDNT_RESOLVE_PROXY:
            return nn::time::ResultHttpErrorCouldntResolveProxy();
        case CURLE_COULDNT_RESOLVE_HOST:
            return nn::time::ResultHttpErrorCouldntResolveHost();
        case CURLE_COULDNT_CONNECT:
            return nn::time::ResultHttpErrorCouldntConnect();
        case CURLE_REMOTE_ACCESS_DENIED:
            return nn::time::ResultHttpErrorRemoteAccessDenied();
        case CURLE_HTTP2:
            return nn::time::ResultHttpErrorHttp2();
        case CURLE_PARTIAL_FILE:
            return nn::time::ResultHttpErrorPartialFile();
        case CURLE_QUOTE_ERROR:
            return nn::time::ResultHttpErrorQuoteError();
        case CURLE_HTTP_RETURNED_ERROR:
            return nn::time::ResultHttpErrorHttpReturnedError();
        case CURLE_WRITE_ERROR:
            return nn::time::ResultHttpErrorWriteError();
        case CURLE_UPLOAD_FAILED:
            return nn::time::ResultHttpErrorUploadFailed();
        case CURLE_READ_ERROR:
            return nn::time::ResultHttpErrorReadError();
        case CURLE_OUT_OF_MEMORY:
            return nn::time::ResultHttpErrorOutOfMemory();
        case CURLE_OPERATION_TIMEDOUT:
            return nn::time::ResultHttpErrorOperationTimedout();
        case CURLE_RANGE_ERROR:
            return nn::time::ResultHttpErrorRangeError();
        case CURLE_HTTP_POST_ERROR:
            return nn::time::ResultHttpErrorHttpPostError();
        case CURLE_SSL_CONNECT_ERROR:
            return nn::time::ResultHttpErrorSslConnectError();
        case CURLE_BAD_DOWNLOAD_RESUME:
            return nn::time::ResultHttpErrorBadDownloadResume();
        case CURLE_FUNCTION_NOT_FOUND:
            return nn::time::ResultHttpErrorFunctionNotFound();
        case CURLE_ABORTED_BY_CALLBACK:
            return nn::time::ResultHttpErrorAbortedByCallback();
        case CURLE_BAD_FUNCTION_ARGUMENT:
            return nn::time::ResultHttpErrorBadFunctionArgument();
        case CURLE_INTERFACE_FAILED:
            return nn::time::ResultHttpErrorInterfaceFailed();
        case CURLE_TOO_MANY_REDIRECTS:
            return nn::time::ResultHttpErrorTooManyRedirects();
        case CURLE_UNKNOWN_OPTION:
            return nn::time::ResultHttpErrorUnknownOption();
        case CURLE_PEER_FAILED_VERIFICATION:
            return nn::time::ResultHttpErrorPeerFailedVerification();
        case CURLE_GOT_NOTHING:
            return nn::time::ResultHttpErrorGotNothing();
        case CURLE_SSL_ENGINE_NOTFOUND:
            return nn::time::ResultHttpErrorSslEngineNotFound();
        case CURLE_SSL_ENGINE_SETFAILED:
            return nn::time::ResultHttpErrorSslEngineSetFailed();
        case CURLE_SEND_ERROR:
            return nn::time::ResultHttpErrorSendError();
        case CURLE_RECV_ERROR:
            return nn::time::ResultHttpErrorRecvError();
        case CURLE_SSL_CERTPROBLEM:
            return nn::time::ResultHttpErrorSslCertProblem();
        case CURLE_SSL_CIPHER:
            return nn::time::ResultHttpErrorSslCipher();
        case CURLE_SSL_CACERT:
            return nn::time::ResultHttpErrorSslCaCert();
        case CURLE_BAD_CONTENT_ENCODING:
            return nn::time::ResultHttpErrorBadContentEncoding();
        case CURLE_FILESIZE_EXCEEDED:
            return nn::time::ResultHttpErrorFileSizeExceeded();
        case CURLE_USE_SSL_FAILED:
            return nn::time::ResultHttpErrorUseSslFailed();
        case CURLE_SEND_FAIL_REWIND:
            return nn::time::ResultHttpErrorSendFailRewind();
        case CURLE_SSL_ENGINE_INITFAILED:
            return nn::time::ResultHttpErrorSslEngineInitFailed();
        case CURLE_LOGIN_DENIED:
            return nn::time::ResultHttpErrorLoginDenied();
        case CURLE_CONV_FAILED:
            return nn::time::ResultHttpErrorConvFailed();
        case CURLE_CONV_REQD:
            return nn::time::ResultHttpErrorConvReqd();
        case CURLE_SSL_CACERT_BADFILE:
            return nn::time::ResultHttpErrorSslCaCertBadFile();
        case CURLE_SSL_SHUTDOWN_FAILED:
            return nn::time::ResultHttpErrorSslShutdownFailed();
        case CURLE_AGAIN:
            return nn::time::ResultHttpErrorAgain();
        case CURLE_SSL_CRL_BADFILE:
            return nn::time::ResultHttpErrorSslCrlBadFile();
        case CURLE_SSL_ISSUER_ERROR:
            return nn::time::ResultHttpErrorSslIssuerError();
        case CURLE_CHUNK_FAILED:
            return nn::time::ResultHttpErrorChunkFailed();
        case CURLE_NO_CONNECTION_AVAILABLE:
            return nn::time::ResultHttpErrorNoConnectionAvailable();
        case CURLE_SSL_PINNEDPUBKEYNOTMATCH:
            return nn::time::ResultHttpErrorSslPinnedPubkeyNotMatch();
        case CURLE_SSL_INVALIDCERTSTATUS:
            return nn::time::ResultHttpErrorSslInvalidCertStatus();
        case CURLE_SSL_CTX_FUNCTION_NOT_SET:
            return nn::time::ResultHttpErrorSslCtxFunctionNotSet();
        case CURLE_SSL_CTX_INVALID:
            return nn::time::ResultHttpErrorSslCtxInvalid();
        case CURLE_SSL_CTX_FATAL:
            return nn::time::ResultHttpErrorSslCtxFatal();
        case CURLE_SSL_ALREADY_CONNECTED:
            return nn::time::ResultHttpErrorSslAlreadyConnected();
        case CURLE_SSL_INVALID_REFERENCE:
            return nn::time::ResultHttpErrorSslInvalidReference();
        default:
            return nn::time::ResultUnexpectedHttpError();
        }
    } // NOLINT(impl/function_size)
}

}}}}} // nn::ntc::detail::service::core
