﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/fs.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_Result.private.h>
#include <nn/ns/srv/ns_VersionManagementDatabase.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonApi.h>
#include <nn/ns/srv/detail/json/ns_RapidJsonInputStream.h>
#include <nn/ns/srv/detail/json/ns_JsonAdaptor.h>
#include <nn/ns/srv/detail/json/ns_ResultForJson.h>
#include <rapidjson/encodedstream.h>

#include "ns_StringUtil.h"
#include "ns_JsonUtil.h"
namespace nn { namespace ns { namespace srv {
    namespace {
        const int MaxPathBufferLength = 48;
        const int MaxNodeDepth = 6;

        template <int NodeDepthMax, int PathLengthMax>
        class VersionListAdaptorBase
        {
            NN_STATIC_ASSERT(NodeDepthMax >= MaxNodeDepth && PathLengthMax >= MaxPathBufferLength);
            NN_DISALLOW_COPY(VersionListAdaptorBase);

        public:
            typedef detail::json::JsonPath<NodeDepthMax, PathLengthMax> JsonPathType;

        public:
            explicit VersionListAdaptorBase() NN_NOEXCEPT
            {
            }

            Result Adapt(int32_t) const NN_NOEXCEPT
            {
                NN_RESULT_SUCCESS;
            }

            void Update(const JsonPathType&, int64_t) NN_NOEXCEPT
            {
            }
            void Update(const JsonPathType&, const char*, int) NN_NOEXCEPT
            {
            }

            void Update(const JsonPathType&, std::nullptr_t) NN_NOEXCEPT
            { /*NOP*/
            }
            void Update(const JsonPathType&, bool) NN_NOEXCEPT
            { /*NOP*/
            }
            void Update(const JsonPathType&, uint64_t) NN_NOEXCEPT
            { /*NOP*/
            }
            void Update(const JsonPathType&, double) NN_NOEXCEPT
            { /*NOP*/
            }
            void Update(const JsonPathType&, detail::json::Node, bool) NN_NOEXCEPT
            {
            }

        };

        class VersionListEntryAdaptor :
            public detail::json::ExtensibleJsonAdaptorBase<VersionListAdaptorBase<MaxNodeDepth, MaxPathBufferLength>>
        {
        public:
            VersionListEntryAdaptor() NN_NOEXCEPT :
                m_TitleIndex(0), m_RequiredVersionFound(false), m_IdFound(false), m_VersionFound(false)
            {
            }

            virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE
            {
                UpdateLookupPathForRequiredVersion();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    m_Entry.notifiedVersion = static_cast<int32_t>(value);
                    m_RequiredVersionFound = true;
                }

                UpdateLookupPathForVersion();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    m_Entry.recommendedVersion = static_cast<int32_t>(value);
                    m_VersionFound = true;
                }
                return true;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE
            {
                NN_UNUSED(valueLength);
                UpdateLookupPathForId();
                if (jsonPath.Match(m_LookupPathBuffer))
                {
                    JsonParser::MakeBit64FromString(&m_Entry.id, value);
                    m_IdFound = true;
                }
                return true;
            }

            Result Validate() NN_NOEXCEPT {
                if (!m_IdFound || !m_VersionFound) {
                    NN_RESULT_THROW(nn::ns::ResultCommunicationFailed());
                }
                if (!m_RequiredVersionFound) {
                    m_Entry.notifiedVersion = m_Entry.recommendedVersion;
                }
                NN_RESULT_SUCCESS;
            }

            void UpdateIndex(int32_t index) NN_NOEXCEPT
            {
                m_TitleIndex = index;
                m_VersionFound = false;
                m_RequiredVersionFound = false;
                m_IdFound = false;
            }

            VersionListEntry& GetEntry() NN_NOEXCEPT {
                return m_Entry;
            }

        private:
            void UpdateLookupPathForId() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.titles[%d].id", m_TitleIndex);
            }
            void UpdateLookupPathForVersion() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.titles[%d].version", m_TitleIndex);
            }
            void UpdateLookupPathForRequiredVersion() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.titles[%d].required_version", m_TitleIndex);
            }

            char m_LookupPathBuffer[MaxPathBufferLength];
            int m_TitleIndex;
            VersionListEntry m_Entry;
            bool m_RequiredVersionFound;
            bool m_IdFound;
            bool m_VersionFound;
        };




        class VersionListAdaptor :
            public detail::json::ExtensibleJsonAdaptorBase<VersionListAdaptorBase<MaxNodeDepth, MaxPathBufferLength>>,
            public detail::json::Cancellable
        {
        public:
            explicit VersionListAdaptor(VersionListDatabase& db) NN_NOEXCEPT :
                m_TitleIndex(0), m_pDb(&db), m_LastError(ResultSuccess())
            {
            }

            virtual Result AdaptImpl() NN_NOEXCEPT NN_OVERRIDE
            {
                NN_RESULT_SUCCESS;
            }

            virtual bool UpdateImpl(const JsonPathType&, bool) NN_NOEXCEPT NN_OVERRIDE
            {
                return false;
            }
            virtual bool UpdateImpl(const JsonPathType& jsonPath, int64_t value) NN_NOEXCEPT NN_OVERRIDE
            {

                if (jsonPath.Match("$.last_modified"))
                {
                    m_pDb->SetLastModified(value);
                }

                m_VersionListEntryAdaptor.UpdateImpl(jsonPath, value);

                return true;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, const char* value, int valueLength) NN_NOEXCEPT NN_OVERRIDE
            {
                m_VersionListEntryAdaptor.UpdateImpl(jsonPath, value, valueLength);

                return true;
            }

            virtual bool UpdateImpl(const JsonPathType& jsonPath, detail::json::Node node, bool isStart) NN_NOEXCEPT NN_OVERRIDE
            {
                if (node.kind == detail::json::Node::Kind::Array && isStart)
                {
                }

                if (node.kind == detail::json::Node::Kind::Array && !isStart)
                {
                }

                if (node.kind == detail::json::Node::Kind::Object && isStart)
                {
                    UpdateLookupPathForTitle();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        m_VersionListEntryAdaptor.UpdateIndex(m_TitleIndex);
                    }
                }
                if (node.kind == detail::json::Node::Kind::Object && !isStart)
                {
                    UpdateLookupPathForTitle();
                    if (jsonPath.Match(m_LookupPathBuffer))
                    {
                        auto validationResult = m_VersionListEntryAdaptor.Validate();
                        if (validationResult.IsFailure()) {
                            m_LastError = validationResult;
                            this->Cancel();
                            return false;
                        }

                        auto importResult = m_pDb->Add(m_VersionListEntryAdaptor.GetEntry());
                        if (importResult.IsFailure()) {
                            m_LastError = importResult;
                            this->Cancel();
                            return false;
                        }
                        m_TitleIndex++;
                    }
                }
                return true;
            }

            Result GetLastError() NN_NOEXCEPT
            {
                return m_LastError;
            }
        private:
            void UpdateLookupPathForTitle() NN_NOEXCEPT
            {
                util::TSNPrintf(m_LookupPathBuffer, sizeof(m_LookupPathBuffer), "$.titles[%d]", m_TitleIndex);
            }

            int m_TitleIndex;
            char m_LookupPathBuffer[MaxPathBufferLength];
            VersionListDatabase* m_pDb;
            Result m_LastError;
            VersionListEntryAdaptor m_VersionListEntryAdaptor;
        };
    } // namespace

    template<typename InputStreamType>
    Result VersionListImporter::Import(InputStreamType& inputStream) NN_NOEXCEPT
    {
        VersionListAdaptor adaptor(*m_pDb);
        NN_RESULT_TRY(detail::json::ImportJsonByRapidJson(adaptor, inputStream, &adaptor))
            NN_RESULT_CATCH(nn::ns::ResultCanceled)
            {
            }
        NN_RESULT_END_TRY;
        NN_RESULT_DO(adaptor.GetLastError());

        NN_RESULT_SUCCESS;
    }

    template Result VersionListImporter::Import<detail::json::EncodedMemoryInputStreamForRapidJson>(detail::json::EncodedMemoryInputStreamForRapidJson&);
    template Result VersionListImporter::Import<detail::json::AsyncDataInputStreamForRapidJson<>>(detail::json::AsyncDataInputStreamForRapidJson<>&);

}}}
