﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/srv/ns_UserResourceManager.h>
#include <nn/ns/detail/ns_IApplicationManagerInterface.sfdl.h>

#include <memory>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/ec/system/ec_DeviceLinkApi.h>
#include <nn/ec/system/ec_DeviceLinkSystemApi.h>
#include <nn/friends/friends_ApiAdmin.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_Utility.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/npns/npns_ApiSystem.h>
#endif
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/pdm/pdm_NotifyEventApi.h>

namespace nn { namespace ns { namespace srv {

namespace {

// ---------------------------------------------------------------------------------------------
// 単体のシステムセーブデータ削除

class DeleteSystemSaveDataOperator
{
public:
    static const fs::SaveDataSpaceId TargetSpaceIds[];
    static const bool PostProcessRefreshRequired = false;

private:
    const fs::UserId m_UserId;
    const fs::SystemSaveDataId m_SysSaveId;
    Result m_Result;

public:
    DeleteSystemSaveDataOperator(const account::Uid& uid, const fs::SystemSaveDataId& sysSaveId) NN_NOEXCEPT
        : m_UserId(fs::ConvertAccountUidToFsUserId(uid))
        , m_SysSaveId(sysSaveId)
        , m_Result(ResultSuccess())
    {
    }
    Result GetResult() const NN_NOEXCEPT
    {
        return m_Result;
    }

    bool IsCanceled() const NN_NOEXCEPT
    {
        return false;
    }
    bool Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT
    {
        return true
            && info.saveDataType == fs::SaveDataType::System
            && info.saveDataUserId == m_UserId
            && info.systemSaveDataId == m_SysSaveId;
    }
    bool Process(const fs::SaveDataInfo& info) NN_NOEXCEPT
    {
        auto r = fs::DeleteSaveData(info.saveDataId);
        if (!(r.IsSuccess() || fs::ResultTargetNotFound::Includes(r)))
        {
            m_Result = r;
        }
        return false;
    }
};
const fs::SaveDataSpaceId DeleteSystemSaveDataOperator::TargetSpaceIds[] = {
    nn::fs::SaveDataSpaceId::System};

// ---------------------------------------------------------------------------------------------
// ユーザーのセーブデータ一覧の統計情報 (進捗 % の分母)

class CalculateUserSaveDataStatisticsOperator
{
public:
    static const fs::SaveDataSpaceId TargetSpaceIds[];
    static const bool PostProcessRefreshRequired = false;

private:
    const fs::UserId m_UserId;
    UserSaveDataStatistics m_Stat;

public:
    explicit CalculateUserSaveDataStatisticsOperator(const account::Uid& uid) NN_NOEXCEPT
        : m_UserId(fs::ConvertAccountUidToFsUserId(uid))
    {
        m_Stat.count = 0u;
        m_Stat.sizeInBytes = 0ull;
    }
    UserSaveDataStatistics GetStatistics() const NN_NOEXCEPT
    {
        return m_Stat;
    }

    bool IsCanceled() const NN_NOEXCEPT
    {
        return false;
    }
    bool Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT
    {
        return info.saveDataUserId == m_UserId;
    }
    bool Process(const fs::SaveDataInfo& info) NN_NOEXCEPT
    {
        ++ m_Stat.count;
        m_Stat.sizeInBytes += info.saveDataSize;
        return true;
    }
};
const fs::SaveDataSpaceId CalculateUserSaveDataStatisticsOperator::TargetSpaceIds[] = {
    nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};


// ---------------------------------------------------------------------------------------------
// ユーザーのセーブデータを一括削除 (削除処理と進捗 % の分子)

class DeleteSaveDataOperator
    : CalculateUserSaveDataStatisticsOperator
{
public:
    using CalculateUserSaveDataStatisticsOperator::TargetSpaceIds;
    static const bool PostProcessRefreshRequired = true;

private:
    ProgressMonitorForDeleteUserSaveDataAll& m_Progress;
    Result m_Result;

public:
    DeleteSaveDataOperator(ProgressMonitorForDeleteUserSaveDataAll& progress, const account::Uid& uid) NN_NOEXCEPT
        : CalculateUserSaveDataStatisticsOperator(uid)
        , m_Progress(progress)
        , m_Result(ResultSuccess())
    {
    }
    Result GetResult() const NN_NOEXCEPT
    {
        return m_Result;
    }

    bool IsCanceled() const NN_NOEXCEPT
    {
        return m_Progress.IsCanceled();
    }
    bool Filter(const fs::SaveDataInfo& info) const NN_NOEXCEPT
    {
        return CalculateUserSaveDataStatisticsOperator::Filter(info);
    }
    bool Process(const fs::SaveDataInfo& info) NN_NOEXCEPT
    {
        if (!CalculateUserSaveDataStatisticsOperator::Process(info))
        {
            return false;
        }

        // 削除
        auto r = fs::DeleteSaveData(info.saveDataId);
        if (!(r.IsSuccess() || fs::ResultTargetNotFound::Includes(r)))
        {
            m_Result = r;
            return false;
        }

        switch (info.saveDataSpaceId)
        {
        case fs::SaveDataSpaceId::User:
            // ユーザー領域にある場合はアプリケーション ID を与える
            // (アプリケーションに関連付かないユーザー領域のアプリケーション ID は 0x00ull になる。)
            m_Progress.Update(CalculateUserSaveDataStatisticsOperator::GetStatistics(), info.applicationId);
            break;
        default:
            m_Progress.Update(CalculateUserSaveDataStatisticsOperator::GetStatistics());
            break;
        }
        return true;
    }
};

// ---------------------------------------------------------------------------------------------
// 汎用セーブデータウォーカー

template <typename Operator>
Result WalkSaveDataList(Operator& op) NN_NOEXCEPT
{
    for (auto spaceId : Operator::TargetSpaceIds)
    {
        std::unique_ptr<fs::SaveDataIterator> iter;
        NN_RESULT_DO(fs::OpenSaveDataIterator(&iter, spaceId));

        while (NN_STATIC_CONDITION(true))
        {
            NN_RESULT_THROW_UNLESS(!op.IsCanceled(), ResultCanceled());

            int64_t count;
            fs::SaveDataInfo info;
            NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }

            if (op.Filter(info))
            {
                if (!op.Process(info))
                {
                    // 終了
                    break;
                }

                if (NN_STATIC_CONDITION(Operator::PostProcessRefreshRequired))
                {
                    // TODO 毎回再検索するのは無駄が多いので、スキップできるようにする
                    NN_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, spaceId));
                }
            }
        }
    }
    NN_RESULT_SUCCESS;
}

} // ~namesapce nn::ns::srv

Result UserResourceManager::DeleteUserSystemSaveData(const account::Uid& uid, const fs::SystemSaveDataId& sysSaveId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserAccount());

    DeleteSystemSaveDataOperator op(uid, sysSaveId);
    NN_RESULT_DO(WalkSaveDataList(op));
    NN_RESULT_DO(op.GetResult());
    NN_RESULT_SUCCESS;
}

Result UserResourceManager::CalculateUserSaveDataStatistics(UserSaveDataStatistics* pOut, const account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOut != nullptr, ResultNullptr());
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserAccount());

    CalculateUserSaveDataStatisticsOperator op(uid);
    NN_RESULT_DO(WalkSaveDataList(op));

    *pOut = op.GetStatistics();
    NN_RESULT_SUCCESS;
}

Result UserResourceManager::DeleteUserSaveDataAll(ProgressMonitorForDeleteUserSaveDataAll& progress, const account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUserAccount());
    NN_SDK_ASSERT(progress.IsInitialized());

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    bool pdmSuspended = true;
    NN_RESULT_TRY(pdm::SuspendUserAccountEventService(uid))
        NN_RESULT_CATCH(pdm::ResultServiceNotAvailable)
        {
            pdmSuspended = false;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT
    {
        if (pdmSuspended)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::ResumeUserAccountEventService(uid));
        }
    };
#endif
    friends::DaemonSuspension friendsSuspension;
    NN_RESULT_DO(friends::SuspendDaemon(&friendsSuspension));

    // 削除処理
    DeleteSaveDataOperator op(progress, uid);
    NN_RESULT_DO(WalkSaveDataList(op));
    NN_RESULT_DO(op.GetResult());
    NN_RESULT_SUCCESS;
}

/*
    セーブデータ消さない版の NSA 連係解除関数
    (Psel だけが使用する予定)
*/
Result UserResourceManager::UnregisterNetworkServiceAccount(const account::Uid& uid) NN_NOEXCEPT
{
    friends::SetOption(friends::OptionAdmin_CheckUserStatus, 0);

    // -----------------------------------------------------------------------------------------
    // 0. 対象のユーザーアカウントが存在し、ネットワークサービスアカウントが利用可能であることを確認する。
    account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool nsaAvailable = true; // NSA を利用可能かどうか
    NN_UNUSED(nsaAvailable); // TORIAEZU: 今のところ使わない
    auto r = admin.CheckNetworkServiceAccountAvailability();
    if (!r.IsSuccess())
    {
        // 「NSA 利用不可 (NSA 未登録以外)」の場合を除きエラー
        NN_RESULT_THROW_UNLESS(true
            && account::ResultNetworkServiceAccountUnavailable::Includes(r)
            && !account::ResultNetworkServiceAccountRegistrationRequired::Includes(r),
            r);
        nsaAvailable = false;
    }
    NN_DETAIL_NS_INFO("[ns] NSA Availability: %04d\n", r.GetDescription());

    // -----------------------------------------------------------------------------------------
    // 1. NSA が消えた後に残っているとまずいものを消す

#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 機器認証解除 (ec)
    if(ec::system::HasDeviceLink(uid))
    {
        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestUnlinkDevicePrivate(&async, uid));
        NN_RESULT_DO(async.Get());
        NN_DETAIL_NS_INFO("[ns] (ec) Device unlink ensured\n");
    }
#endif

    // 遊んだ記録の削除 (friends)
    NN_RESULT_DO(friends::DeletePlayHistory(uid));
    NN_DETAIL_NS_INFO("[ns] (friends) PlayHistory removed\n");

    // Cruiser のセーブデータの削除 (Shop, LoginShare)
    NN_RESULT_DO(DeleteUserSystemSaveData(uid, static_cast<fs::SystemSaveDataId>(0x8000000000001061ull /* Shop */)));
    NN_RESULT_DO(DeleteUserSystemSaveData(uid, static_cast<fs::SystemSaveDataId>(0x8000000000001091ull /* LoginShare */)));
    NN_DETAIL_NS_INFO("[ns] (Cruiser) savedata removed for Shop and LoginShare\n");

    // -----------------------------------------------------------------------------------------
    // 2. NSA の登録解除処理
    account::AsyncContext ctx;
    NN_RESULT_DO(admin.UnregisterAsync(&ctx));
    os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();
    NN_RESULT_DO(ctx.GetResult());
    NN_DETAIL_NS_INFO("[ns] (account) NSA Unregistered\n");

    // -----------------------------------------------------------------------------------------
    // 3. NSA を消した後に、できれば消したいものを消す

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    //    - NPNS の通知トークンを削除
    //      (この処理はエラーを無視できる。)
    NN_RESULT_DO(npns::Suspend());
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(npns::Resume());
    };
    npns::DestroyTokenForBaaS(uid);
#endif

    NN_RESULT_SUCCESS;
}

/*
    NA連携解除処理 (ネットワークサービスアカウント の登録解除処理)

    関連 JIRA タスク:
        - http://spdlybra.nintendo.co.jp/jira/browse/SWHUID-3533
        - http://spdlybra.nintendo.co.jp/jira/browse/IAAD-436
        - http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-32258
        - http://spdlybra.nintendo.co.jp/jira/browse/IAAD-1566
        - http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-81695

    処理概要:
        1. NSA が消えた後に残っているとまずいものを消す
          - 権利同期・機器認証解除 (DRM)
          - セーブデータ
        2. NSA の登録解除処理
        3. NSA を消した後に、できれば消したいものを消す
          - NPNS の通知トークンを削除
*/
Result UserResourceManager::UnregisterNetworkServiceAccountWithUserSaveDataDeletion(const account::Uid& uid) NN_NOEXCEPT
{
    // -----------------------------------------------------------------------------------------
    // 0. 対象のユーザーアカウントが存在し、ネットワークサービスアカウントが利用可能であることを確認する。
    account::NetworkServiceAccountAdministrator admin;
    NN_RESULT_DO(account::GetNetworkServiceAccountAdministrator(&admin, uid));

    bool nsaAvailable = true; // NSA を利用可能かどうか
    NN_UNUSED(nsaAvailable); // TORIAEZU: 今のところ使わない
    auto r = admin.CheckNetworkServiceAccountAvailability();
    if (!r.IsSuccess())
    {
        // 「NSA 利用不可 (NSA 未登録以外)」の場合を除きエラー
        NN_RESULT_THROW_UNLESS(true
            && account::ResultNetworkServiceAccountUnavailable::Includes(r)
            && !account::ResultNetworkServiceAccountRegistrationRequired::Includes(r),
            r);
        nsaAvailable = false;
    }
    NN_DETAIL_NS_INFO("[ns] NSA Availability: %04d\n", r.GetDescription());

    // -----------------------------------------------------------------------------------------
    // 1. NSA が消えた後に残っているとまずいものを消す
#ifdef NN_BUILD_CONFIG_OS_HORIZON
    // 機器認証解除 (ec)
    if(ec::system::HasDeviceLink(uid))
    {
        ec::system::AsyncResult async;
        NN_RESULT_DO(ec::system::RequestUnlinkDevicePrivate(&async, uid));
        NN_RESULT_DO(async.Get());
        NN_DETAIL_NS_INFO("[ns] (ec) Device unlink ensured\n");
    }
#endif

    // セーブデータ
    ProgressMonitorForDeleteUserSaveDataAll progress;
    progress.Initialize();
    progress.MarkAsFinished(DeleteUserSaveDataAll(progress, uid));
    auto p = progress.GetProgress();
    NN_DETAIL_NS_INFO(
        "[ns] SaveData deletion: %u (%llu bytes), %lld seconds\n",
        p.GetStatistics().count, p.GetStatistics().sizeInBytes, p.GetElapsedTime().GetSeconds());
    NN_UNUSED(p);
    NN_RESULT_DO(progress.GetResult());

    // -----------------------------------------------------------------------------------------
    // 2. NSA の登録解除処理
    account::AsyncContext ctx;
    NN_RESULT_DO(admin.UnregisterAsync(&ctx));
    os::SystemEvent e;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
    e.Wait();
    NN_RESULT_DO(ctx.GetResult());
    NN_DETAIL_NS_INFO("[ns] (account) NSA Unregistered\n");

    // -----------------------------------------------------------------------------------------
    // 3. NSA を消した後に、できれば消したいものを消す
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_RESULT_DO(npns::Suspend());
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(npns::Resume());
    };
    npns::DestroyTokenForBaaS(uid);
#endif

    NN_RESULT_SUCCESS;
}


}}}
