﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_Config.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/es.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ec/system/ec_VirtualAccountApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_TicketManager.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include <nn/time/time_StandardNetworkSystemClock.h>

#include "ns_AsyncImpl.h"
#include "ns_PrepurchasedContent.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        Result HasContentRightsImpl(bool* outValue, Result result) NN_NOEXCEPT
        {
            if (result.IsSuccess())
            {
                *outValue = true;
            }
            else if (ResultApplicationLaunchRightsNotFound::Includes(result))
            {
                *outValue = false;
            }
            else
            {
                NN_RESULT_THROW(result);
            }

            NN_RESULT_SUCCESS;
        }
    }

    typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

    Result TicketManager::CheckAccountRights(es::AccountId accountId) const NN_NOEXCEPT
    {
        // コモンチケットの場合
        // 権利が有効
        if (accountId == 0)
        {
            NN_RESULT_SUCCESS;
        }
        // パーソナライズドチケットの場合
        else
        {
            ec::system::DeviceAccountInfo deviceAccountInfo;
            NN_RESULT_DO(ec::system::GetDeviceAccountInfo(&deviceAccountInfo));

            char* endPtr;
            uint32_t deviceAccountId = std::strtoul(deviceAccountInfo.id, &endPtr, 10);
            NN_RESULT_THROW_UNLESS(*endPtr == '\0', ResultInvalidDeviceAccountId());

            // パーソナライズドチケットで DA に紐付いた権利の場合、権利が有効
            if (accountId == deviceAccountId)
            {
                NN_RESULT_SUCCESS;
            }
            // パーソナライズドチケットで VA に紐付いた権利の場合、対応する NA が有効ならば権利が有効
            else
            {
                account::Uid uid;
                NN_RESULT_TRY(ec::system::GetAccountByVirtualAccount(&uid, accountId))
                    NN_RESULT_CATCH(ec::ResultRegisteredVirtualAccountNotFound)
                    {
                        NN_RESULT_THROW(ResultRightsOwnerVirtualAccountNotFound());
                    }
                NN_RESULT_END_TRY

                account::NetworkServiceAccountManager accountManager;
                NN_RESULT_DO(account::GetNetworkServiceAccountManager(&accountManager, uid));

                // 「NA 連携状態が正常」 or 「NA がエラー状態で、それが退会や削除でない」場合は権利を利用可能
                NN_RESULT_TRY(accountManager.CheckNetworkServiceAccountAvailability())
                    NN_RESULT_CATCH_CONVERT(account::ResultNintendoAccountStateDeleted, ResultInactiveNintendoAccount())    // 削除
                    NN_RESULT_CATCH_CONVERT(account::ResultNintendoAccountStateWithdrawn, ResultInactiveNintendoAccount())  // 退会手続きを行った
                    NN_RESULT_CATCH_CONVERT(account::ResultNintendoAccountStateInteractionRequired, ResultSuccess())        // 上記以外の NA のエラー状態
                NN_RESULT_END_TRY

                NN_RESULT_SUCCESS;
            }
        }
    }

    Result TicketManager::CheckTicketRights(es::RightsIdIncludingKeyId rightsId) const NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // 1つの RightsId に対して最大で持ちえるチケットの数(最大でリンクされる VA の数 + DA + コモンチケット)
        const int MaxTicketNum = nn::account::UserCountMax + 1 + 1;

        // 権利を持っているかを確認
        es::LightTicketInfo ticketInfo[MaxTicketNum];

        int ticketCount = es::ListLightTicketInfo(ticketInfo, MaxTicketNum, rightsId);
        // チケットがない場合、権利を持っていない
        if (ticketCount == 0)
        {
            NN_RESULT_THROW(ResultApplicationTicketNotFound());
        }

        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(rightsId);

        NN_RESULT_SUCCESS;
#endif
    }

    Result TicketManager::CheckPrepurchaseRights(es::RightsIdIncludingKeyId rightsId) const NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // 1つの RightsId に対して最大で持ちえる予約購入情報の数(最大でリンクされる VA の数 + DA + コモンチケット)
        const int MaxPrepurchaseRecordNum = nn::account::UserCountMax + 1 + 1;

        es::PrepurchaseRecord record[MaxPrepurchaseRecordNum];
        int recordCount = es::ListPrepurchaseRecordInfo(record, MaxPrepurchaseRecordNum, rightsId);

        if (recordCount == 0)
        {
            NN_RESULT_THROW(ResultApplicationTicketNotFound());
        }

        bool isFoundInactiveNintendoAccount = false;

        // 取得した予約購入情報に対して、権利が有効であるかを判定する
        // 権利が有効な予約購入情報を1つでも持っていたら予約購入状態である
        for (int i = 0; i < recordCount; i++)
        {
            NN_RESULT_TRY(CheckAccountRights(record[i].accountId))
                // 予約購入情報に対応する NA が無効だった場合、次の予約購入情報の権利があるか確認する
                NN_RESULT_CATCH(ResultInactiveNintendoAccount)
                {
                    isFoundInactiveNintendoAccount = true;
                    NN_DETAIL_NS_TRACE("[ApplicationLaunchManager] Inactive NintendoAccount\n");
                    continue;
                }
                // 予約購入情報に対応する VA が見つからなかった場合、次のチケットの権利があるか確認する
                // 通常は VA が見つからないことはないはずだが、VA と予約購入情報の不整合が起こったときに他のアカウントの権利を行使できないようになってしまうのを避けるための措置
                NN_RESULT_CATCH(ResultRightsOwnerVirtualAccountNotFound)
                {
                    NN_DETAIL_NS_TRACE("[ApplicationLaunchManager] Virtual Account not found.\n");
                    continue;
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        // 権利を利用可能でない NA が1つ以上あった場合
        if (isFoundInactiveNintendoAccount)
        {
            NN_RESULT_THROW(ResultInactiveNintendoAccount());
        }
        // 予約購入情報に紐付く VA が見つからなかった場合
        else
        {
            NN_RESULT_THROW(ResultRightsOwnerVirtualAccountNotFound());
        }
#else
        NN_UNUSED(rightsId);

        NN_RESULT_SUCCESS;
#endif
    }

    Result TicketManager::VerifyContentRights(
        const ncm::ContentMetaKey& key,
        ncm::ContentType contentType,
        uint8_t programIndex) const NN_NOEXCEPT
    {
        ncm::StorageId storageId;
        NN_RESULT_DO(m_pIntegrated->GetInstalledContentMetaStorage(&storageId, key));

        return VerifyContentRights(key, contentType, programIndex, storageId);
    }

    Result TicketManager::VerifyContentRights(
        const ncm::ContentMetaKey& key,
        ncm::ContentType contentType,
        uint8_t programIndex,
        ncm::StorageId storageId) const NN_NOEXCEPT
    {
        fs::RightsId fsRightsId;
        NN_RESULT_DO(m_pIntegrated->GetContentRightsId(&fsRightsId, key, contentType, programIndex, storageId));
        NN_RESULT_DO(VerifyFsRightsId(fsRightsId));

        NN_RESULT_SUCCESS;
    }

    Result TicketManager::VerifyFsRightsId(const fs::RightsId& fsRightsId) const NN_NOEXCEPT
    {
        es::RightsIdIncludingKeyId rightsId = es::RightsIdIncludingKeyId::Construct(fsRightsId);
        if (!rightsId.IsExternalKey())
        {
            NN_RESULT_SUCCESS;
        }

        NN_RESULT_TRY(CheckTicketRights(rightsId))
            NN_RESULT_CATCH(ResultApplicationTicketNotFound)
            {
                Result result = CheckPrepurchaseRights(rightsId);
                if (result.IsSuccess())
                {
                    NN_RESULT_THROW(ResultApplicationPrepurchased());
                }

                NN_RESULT_THROW(result);
            }
        NN_RESULT_END_TRY

        NN_RESULT_SUCCESS;
    }

    Result TicketManager::HasContentRights(bool* outHasRights, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex) const NN_NOEXCEPT
    {
        NN_RESULT_DO(HasContentRightsImpl(outHasRights, VerifyContentRights(key, contentType, programIndex)));
        NN_RESULT_SUCCESS;
    }

    Result TicketManager::HasContentRights(bool* outHasRights, const ncm::ContentMetaKey& key, ncm::ContentType contentType, uint8_t programIndex, ncm::StorageId storageId) const NN_NOEXCEPT
    {
        NN_RESULT_DO(HasContentRightsImpl(outHasRights, VerifyContentRights(key, contentType, programIndex, storageId)));
        NN_RESULT_SUCCESS;
    }

    Result TicketManager::RequestDownloadApplicationPrepurchasedRights(sf::Out<sf::NativeHandle> outHandle, sf::Out<nn::sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT
    {
        ApplicationRecordAccessor list;
        NN_RESULT_DO(m_pRecordDb->Open(&list, id));
        int contentCount = 0;
        es::RightsIdIncludingKeyId tmpRightsId;
        std::unique_ptr<es::RightsIdIncludingKeyId[]> prepurchasedContentRightsIds(new es::RightsIdIncludingKeyId[MaxPrepurchasedContentsCount]);

        // アプリケーションは共通の外部鍵で暗号化されており、コンテンツを利用するためには Program のチケットを取得すればいいので、Program のチケットを取得する
        // この前提が変わるときには他の ContentType のチケットの取得も必要となる
        NN_RESULT_DO(GetRightsId(&tmpRightsId, id.value, ncm::ContentType::Program));
        if (IsPrepurchased(tmpRightsId))
        {
            prepurchasedContentRightsIds[contentCount] = tmpRightsId;
            contentCount++;
        }

        for (int i = 0; i < list.GetDataCount(); i++)
        {
            auto& key = list.GetDataAt(i).key;
            if (key.type != ncm::ContentMetaType::AddOnContent)
            {
                continue;
            }

            NN_RESULT_DO(GetRightsId(&tmpRightsId, key.id, ncm::ContentType::Data));
            if (IsPrepurchased(tmpRightsId))
            {
                prepurchasedContentRightsIds[contentCount] = tmpRightsId;
                contentCount++;

                if (contentCount >= MaxPrepurchasedContentsCount)
                {
                    break;
                }
            }
        }

        // 予約状態のコンテンツが1つもない場合
        NN_RESULT_THROW_UNLESS(contentCount > 0, ResultPrepurchasedContentNotFound());

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncDownloadApplicationPrepurchasedRightsImpl>(this, m_RequestServer);
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        // 取得する必要があるチケットの RightsId の一覧を渡して初期化する
        // ns::MaxPrepurchasedContentsCount で規定される上限を超える数のチケットを取得しようとした場合でも復旧が困難にならないようにするため取得可能な範囲でチケットを取得する
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(prepurchasedContentRightsIds.get(), contentCount));
        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result TicketManager::GetApplicationTicketInfo(nn::sf::Out<ns::ApplicationTicketInfo> outValue, ncm::ApplicationId id) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // 権利を保有している場合
        if (es::CountTicket(id.value) > 0)
        {
            *outValue = { ns::ApplicationTicketInfo::Status::Owned };
        }
        // 予約状態の場合
        else if (es::CountPrepurchaseRecord(id.value) > 0)
        {
            es::PrepurchaseRecord record;
            {
                es::RightsIdIncludingKeyId rightsId;
                es::ListPrepurchaseRecordRightsIds(&rightsId, 1, id.value);
                es::ListPrepurchaseRecordInfo(&record, 1, rightsId);
            }

            time::PosixTime currentTime;
            NN_RESULT_DO(time::StandardNetworkSystemClock::GetCurrentTime(&currentTime));

            // 発売済みの場合
            if (record.deliveryScheduledTime <= currentTime)
            {
                *outValue = { ns::ApplicationTicketInfo::Status::PrepurchasedAndMaybeReleased };
            }
            // 未発売の場合
            else
            {
                *outValue = { ns::ApplicationTicketInfo::Status::PrepurchasedButMaybeNotReleased };
            }
        }
        // 権利を保有していない場合
        else
        {
            *outValue = { ns::ApplicationTicketInfo::Status::NotOwned };
        }
#else
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_ABORT("Unsupported platform.");
#endif

        NN_RESULT_SUCCESS;
    }

    Result TicketManager::GetRightsId(es::RightsIdIncludingKeyId* outValue, Bit64 contentMetaId, ncm::ContentType contentType) NN_NOEXCEPT
    {
        ncm::ContentMetaKey key;
        NN_RESULT_DO(m_pIntegrated->GetLatest(&key, contentMetaId));
        ncm::StorageId storageId;
        NN_RESULT_DO(m_pIntegrated->GetInstalledContentMetaStorage(&storageId, key));

        // アプリケーションは共通の外部鍵で暗号化されているので、最小の ProgramIndex を持つコンテンツのチケットを取得する
        // この前提が変わるときには他の ProgramIndex のチケットの取得も必要となる
        uint8_t programIndex;
        NN_RESULT_DO(m_pIntegrated->GetMinimumProgramIndex(&programIndex, key));

        fs::RightsId fsRightsId;
        NN_RESULT_DO(m_pIntegrated->GetContentRightsId(&fsRightsId, key, contentType, programIndex, storageId));

        *outValue = es::RightsIdIncludingKeyId::Construct(fsRightsId);

        NN_RESULT_SUCCESS;
    }

    bool TicketManager::IsPrepurchased(es::RightsIdIncludingKeyId rightsId) NN_NOEXCEPT
    {
        return CheckTicketRights(rightsId).IsFailure() && CheckPrepurchaseRights(rightsId).IsSuccess();
    }
}}}

