﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include "ns_DebugUtil.h"
#include "ns_InstallUtil.h"
#include "ns_SystemUpdateUtil.h"
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>

namespace nn { namespace ns { namespace srv {
    namespace {
        Result HasAllContentMetaInSystemUpdateMeta(bool* outValue, ncm::ContentMetaDatabase* db, const ncm::ContentMetaKey& systemUpdateMetaKey, bool requiresExFat) NN_NOEXCEPT
        {
            ncm::ContentMetaInfo metaList[64];
            int offset = 0;
            while (NN_STATIC_CONDITION(true))
            {
                int count;
                NN_RESULT_DO(db->ListContentMetaInfo(&count, metaList, sizeof(metaList) / sizeof(metaList[0]), systemUpdateMetaKey, offset));
                if (count == 0)
                {
                    break;
                }
                for (int i = 0; i < count; i++)
                {
                    auto& meta = metaList[i];
                    auto key = meta.ToKey();

                    if ((meta.attributes & ncm::ContentMetaAttribute_IncludesExFatDriver) && !requiresExFat)
                    {
                        NN_DETAIL_NS_TRACE("[NeedsUpdate] Ignore 0x%016llx version %u since exFAT driver is not required\n", key.id, key.version);
                        continue;
                    }

                    ncm::ContentMetaKey ownedKey;
                    NN_RESULT_TRY(db->GetLatest(&ownedKey, key.id))
                        NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
                        {
                            *outValue = false;
                            NN_RESULT_SUCCESS;
                        }
                    NN_RESULT_END_TRY

                    if(key.version > ownedKey.version)
                    {
                        *outValue = false;
                        NN_RESULT_SUCCESS;
                    }
                }

                offset += count;
            }

            *outValue = true;
            NN_RESULT_SUCCESS;
        }
    }

    Result GetSystemUpdateTaskIdInfo(util::optional<SystemUpdateTaskIdInfo>* outValue) NN_NOEXCEPT
    {
        nim::SystemUpdateTaskId id;
        auto count = nim::ListSystemUpdateTask(&id, 1);
        if (count == 0)
        {
            *outValue = util::nullopt;
            NN_RESULT_SUCCESS;
        }

        nim::SystemUpdateTaskInfo info;
        NN_RESULT_DO(nim::GetSystemUpdateTaskInfo(&info, id));

        SystemUpdateTaskIdInfo idInfo = { id, info };
        *outValue = idInfo;
        NN_RESULT_SUCCESS;
    }

    Result DestroyAllSystemUpdateTask() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            nim::SystemUpdateTaskId id;
            auto count = nim::ListSystemUpdateTask(&id, 1);
            if (count == 0)
            {
                break;
            }

            NN_RESULT_DO(nim::DestroySystemUpdateTask(id));
        }

        NN_RESULT_SUCCESS;
    }

    Result CreateOrReuseSystemUpdateTask(nim::SystemUpdateTaskId* outValue, const ncm::ContentMetaKey& latest, bool requiresExFatDriver) NN_NOEXCEPT
    {
        util::optional<SystemUpdateTaskIdInfo> idInfo;
        NN_RESULT_DO(GetSystemUpdateTaskIdInfo(&idInfo));

        if (idInfo)
        {
            auto& id = idInfo->id;
            auto& key = idInfo->info.key;
            auto result = idInfo->info.progress.GetLastResult();
            if (IsFatalSystemUpdate(result))
            {
                NN_DETAIL_NS_TRACE("[CreateOrReuseSystemUpdateTask] Destroy fatal task 0x%08x\n", result.GetInnerValueForDebug());
                NN_RESULT_DO(nn::nim::DestroySystemUpdateTask(id));
            }
            else if (key != latest)
            {
                NN_DETAIL_NS_TRACE("[CreateOrReuseSystemUpdateTask] Destroy old task 0x%016llx version %u\n", key.id, key.version);
                NN_RESULT_DO(nn::nim::DestroySystemUpdateTask(id));
            }
            else
            {
                NN_DETAIL_NS_TRACE("[CreateOrReuseSystemUpdateTask] Reuse existing task 0x%016llx version %u\n", key.id, key.version);
                *outValue = id;
                NN_RESULT_SUCCESS;
            }
        }

        nim::SystemUpdateTaskId newId;
        NN_RESULT_DO(nn::nim::CreateSystemUpdateTask(&newId, latest, requiresExFatDriver));
        *outValue = newId;
        NN_RESULT_SUCCESS;
    }

    Result GetLatestSystemUpdateMeta(ncm::ContentMetaKey* outValue, nim::AsyncContentMetaKey* async) NN_NOEXCEPT
    {
        ncm::ContentMetaKey latestKey;
        NN_RESULT_DO(async->Get(&latestKey));

        auto debugKey = GetDebugSystemUpdateMeta();
        if (debugKey)
        {
            latestKey = *debugKey;
        }

        *outValue = latestKey;
        NN_RESULT_SUCCESS;
    }

    Result NeedsUpdate(bool* outValue, const ncm::ContentMetaKey& latest, bool requiresExFat) NN_NOEXCEPT
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::BuildInSystem));

        ncm::ContentMetaKey current;
        NN_RESULT_TRY(db.GetLatest(&current, latest.id))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        if (current.version < latest.version)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        if (current.version == latest.version)
        {
            bool hasAll;
            NN_RESULT_DO(HasAllContentMetaInSystemUpdateMeta(&hasAll, &db, current, requiresExFat));
            if (!hasAll)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    Result NeedsUpdateVulnerability(bool* outValue, const ncm::ContentMetaKey& latest) NN_NOEXCEPT
    {
        ncm::ContentMetaDatabase db;
        ncm::StorageId storage = nn::ncm::StorageId::BuildInSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storage));

        ncm::ContentMetaKey current;
        NN_RESULT_TRY(db.GetLatest(&current, latest.id))
            NN_RESULT_CATCH(ncm::ResultContentMetaNotFound)
            {
                *outValue = true;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        if (current.version < latest.version)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }

        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    util::optional<ncm::ContentMetaKey> GetSystemUpdateMetaKey() NN_NOEXCEPT
    {
        // INFO: 現状は実機ビルドの場合だけデバッグシステムバージョンの設定を有効化
        //       Windows 版は必ずストレージを見に行く
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        auto debugSystemUpdateMetaKey = GetDebugSystemUpdateMeta();
        if (debugSystemUpdateMetaKey)
        {
            return debugSystemUpdateMetaKey;
        }
#endif
        ncm::ContentMetaDatabase db;
        ncm::StorageId storage = nn::ncm::StorageId::BuildInSystem;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storage));

        ncm::ContentMetaKey key;
        auto listCount = db.ListContentMeta(
            &key, 1, ncm::ContentMetaType::SystemUpdate);

        if (listCount.listed == 1)
        {
            ncm::ContentMetaKey latestKey;
            NN_ABORT_UNLESS_RESULT_SUCCESS(db.GetLatest(&latestKey, key.id));

            return latestKey;
        }

        return util::nullopt;
    }

    uint32_t GetSystemUpdateVersion() NN_NOEXCEPT
    {
        auto metaKey = GetSystemUpdateMetaKey();

        if (metaKey)
        {
            return metaKey->version;
        }
        else
        {
            // SystemUpdate タイトルが無い場合はバージョン 0 とする。
            return 0;
        }
    }
}}}
