﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/srv/ns_DevelopInterfaceServerFactory.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/srv/ns_Event.h>
#include <nn/ns/srv/ns_Shell.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/pm/pm_ShellApi.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/ldr/ldr_Result.public.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace ns { namespace srv {


    namespace
    {
        const int ProcessDataCount = 32;
        const os::ProcessId IdOfEmpty = { 0 };
        bool s_IsJitDebugEnabled = false;
        bool s_IsApplicationCrashReportEnabled = true;
        util::optional<bool> s_pIsEnabledApplicationAllThreadDumpOnCrash;
        ns::CrashReport s_AplicationCrashReportPolicy = ns::CrashReport::Deny;


        struct ProcessData
        {
            os::ProcessId   processId;
            Event*          pEvent;
        };

        os::ThreadType              g_Thread;
        os::Mutex                   g_Lock(false);
        NN_ALIGNAS(4096) nn::Bit8   g_Stack[16 * 1024];

        util::IntrusiveList<ShellEventObserverHolder, util::IntrusiveListBaseNodeTraits<ShellEventObserverHolder>> g_ObserverList;

        ProcessData                 g_ProcessData[ProcessDataCount];

        ProcessData* Find(os::ProcessId id)
        {
            for( auto& pd : g_ProcessData )
            {
                if( pd.processId == id )
                {
                    return &pd;
                }
            }

            return NULL;
        }

        // 10進数digit桁決め打ちで文字列に変換
        // 終端文字を付加しない
        void SetBit64ToChar(char* pOut, Bit64 number, int digit)
        {
            while (digit > 0)
            {
                digit--;
                pOut[digit] = '0' + number % 10;
                number /= 10;
            }
            NN_SDK_ASSERT(number == 0, "Overflow in SetBit64ToChar");
        }

        bool IsConfiguredToDump()
        {
            bool toDump;
            size_t size = nn::settings::fwdbg::GetSettingsItemValue(&toDump, sizeof(toDump), "snap_shot_dump", "auto_dump");
            NN_SDK_ASSERT(size == sizeof(toDump), "Failed to read auto_dump setting\n");
            NN_UNUSED(size);
            return toDump;
        }

        bool IsRequiredFullDump()
        {
            bool fullDump;
            size_t size = nn::settings::fwdbg::GetSettingsItemValue(&fullDump, sizeof(fullDump), "snap_shot_dump", "full_dump");
            NN_SDK_ASSERT(size == sizeof(fullDump), "Failed to read full_dump setting\n");
            NN_UNUSED(size);
            return fullDump;
        }

        bool IsRequiredAllLog()
        {
            if (s_pIsEnabledApplicationAllThreadDumpOnCrash)
            {
                return *s_pIsEnabledApplicationAllThreadDumpOnCrash;
            }

            bool allLog;
            size_t size = nn::settings::fwdbg::GetSettingsItemValue(&allLog, sizeof(allLog), "snap_shot_dump", "output_all_log");
            NN_SDK_ASSERT(size == sizeof(allLog), "Failed to read log setting\n");
            NN_UNUSED(size);
            return allLog;
        }

        void SetLogOption(char* argument, int logOption, int digit)
        {
            NN_SDK_ASSERT(0 <= logOption && logOption < 10);
            argument[1 + 1 + digit + 6] = '0' + logOption;
        }

        void SetDumpType(char* argument, int dumpType, int digit)
        {
            NN_SDK_ASSERT(-10 < dumpType && dumpType < 10);
            if (dumpType < 0)
            {
                argument[1 + 1 + digit + 7 + 7] = '-';
                dumpType = -dumpType;
            }
            argument[1 + 1 + digit + 7 + 8] = '0' + dumpType;
        }

        void CreateSsdArguments(char* argument, os::ProcessId pIdToDump, int digit)
        {
            SetBit64ToChar(argument + 2, pIdToDump.value, digit);

            SetLogOption(argument, IsRequiredAllLog() ? 1 : 0, digit);
            if (!IsConfiguredToDump())
            {
                SetDumpType(argument, -1, digit);
            }
            else if (IsRequiredFullDump())
            {
                SetDumpType(argument, 1, digit);
            }
            else
            {
                SetDumpType(argument, 0, digit);
            }
        }

        void TriggerSnapShotDumper(os::ProcessId pIdToDump)
        {
            static os::ProcessId dumpedPid = os::ProcessId::GetInvalidId();
            if (dumpedPid == pIdToDump)
            {
                return;
            }

            const int Digit = 10;
            char argument[] = "D 0123456789 -log 0 -dump  0";// 実行ファイルパスが入るはずの第0引数にダミーを入れておく。0123456789 は後でプロセスIDでつぶされる
            size_t argumentLen = sizeof(argument);
            CreateSsdArguments(argument, pIdToDump, Digit);

            const ncm::ProgramId ssdProgramId = { 0x0100000000002071 };
            Result result = nn::ldr::SetProgramArgument(ssdProgramId, argument, argumentLen);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[SSD] Failed to launch SnapShotDumper. Failed to set arguments.\n");
                return;
            }

            os::ProcessId pId;
            result = pm::LaunchProgram(&pId, ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, ssdProgramId), pm::LaunchProgramFlags_None);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[SSD] Failed to launch SnapShotDumper.\n");
                return;
            }
            dumpedPid = pIdToDump;
        }

        void TriggerCrashReport(os::ProcessId processId)
        {
            static os::ProcessId creportPid = os::ProcessId::GetInvalidId();
            static os::ProcessId crashedProcessPid = os::ProcessId::GetInvalidId();

            const ncm::ProgramId creportProgramId = { 0x0100000000000036 }; // creport
            char argument[64];

            //
            // creport がクラッシュした場合
            //
            if (processId == creportPid)
            {
                // 元々クラッシュしたプロセスを creport の代わりに終了
                TerminateProcess(crashedProcessPid);

                // クラッシュした creport を終了
                TerminateProcess(processId);

                // pid を初期値に戻す
                creportPid = os::ProcessId::GetInvalidId();
                crashedProcessPid = os::ProcessId::GetInvalidId();

                return;
            }

            bool isDetailed = IsApplicationCrashReportEnabled() && ((s_AplicationCrashReportPolicy == ns::CrashReport::Allow) ? true : false);

            // creport に渡す引数
            nn::util::SNPrintf(argument, sizeof(argument), "%lld %d", processId.value, isDetailed ? 1 : 0);
            Result result = nn::ldr::SetProgramArgument(creportProgramId, argument, strnlen(argument, sizeof(argument)));
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[ns] Failed to launch creport process. Failed to set arguments.\n");
                return;
            }

            os::ProcessId pId;
            result = pm::LaunchProgram(&pId, ncm::MakeProgramLocation(ncm::StorageId::BuildInSystem, creportProgramId), pm::LaunchProgramFlags_None);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[ns] Failed to launch creport process.\n");
                return;
            }

            creportPid = pId;
            crashedProcessPid = processId;
        }

        void HandleException(os::ProcessId pId)
        {
            // settings の jit_debug が有効の場合は SSD, 無効の場合は CrashReport を起動
            if (s_IsJitDebugEnabled)
            {
                std::lock_guard<os::Mutex> scope(g_Lock);

#if defined(NN_DETAIL_NS_FORCE_TRIGGER_SNAP_SHOT_DUMPER)
                // SnapShotDumper を強制的に起動するビルドオプションが指定された場合
                TriggerSnapShotDumper(pId);
#else
                if (Find(pId) != NULL)
                {
                    // ns で管理しているプロセスの場合のみ SSD を起動
                    // システムプロセスは ns を経由せず起動されるので管理外（起動しない）
                    TriggerSnapShotDumper(pId);
                }
                else
                {
                    // ns 管理外のプロセス（システムプロセス）がクラッシュした場合は終了する
                    TerminateProcess(pId);
                }
#endif
            }
            else
            {
                // アプリの場合でも creport を起動する
                // ホワイトリストのアプリ以外はエラーレポートのみ作成するが、例外情報は保存しない
                TriggerCrashReport(pId);
            }
        }

        void OnExceptionEvent(pm::ProcessEventInfo eventInfo)
        {
            {
                std::lock_guard<os::Mutex> scope(g_Lock);

                auto pData = Find(eventInfo.processId);
                if( pData != NULL )
                {
                    pData->pEvent->Signal(pm::ProcessEvent_Exception);
                }
                else
                {
                    NN_DETAIL_NS_TRACE("[ProcessLaunchControl] ProcessEvent_Exception listener not found.\n");
                }
            }

            // SnapShotDumper や CrashReport の起動をハンドリング
            HandleException(eventInfo.processId);
        }

        void OnExitEvent(pm::ProcessEventInfo eventInfo)
        {
            std::lock_guard<os::Mutex> scope(g_Lock);

            auto pData = Find(eventInfo.processId);
            if( pData != NULL )
            {
                pData->pEvent->Signal(pm::ProcessEvent_Exit);
                pData->processId = IdOfEmpty;
            }
            else
            {
                NN_DETAIL_NS_TRACE("[ProcessLaunchControl] ProcessEvent_Exit listener not found.\n");
            }
        }

        void OnStartedEvent(pm::ProcessEventInfo eventInfo)
        {
            std::lock_guard<os::Mutex> scope(g_Lock);

            auto pData = Find(eventInfo.processId);
            if( pData != NULL )
            {
                pData->pEvent->Signal(pm::ProcessEvent_Started);
            }
            else
            {
                NN_DETAIL_NS_TRACE("[ProcessLaunchControl] ProcessEvent_Started listener not found.\n");
            }
        }

//        void OnDebugRunningEvent(pm::ProcessEventInfo eventInfo)
//        {
//            std::lock_guard<os::Mutex> scope(g_Lock);
//
//            auto pData = Find(eventInfo.processId);
//            if( pData != NULL )
//            {
//                pData->pEvent->Signal(pm::ProcessEvent_DebugRunning);
//            }
//            else
//            {
//                NN_DETAIL_NS_TRACE("[ProcessLaunchControl] ProcessEvent_DebugRunning listener not found.\n");
//            }
//        }

        void ProcessLaunchControl(void*)
        {
            Result result;
            os::SystemEvent processEvent;

            result = pm::GetProcessEventEvent(&processEvent);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            for(;;)
            {
                processEvent.Wait();
                processEvent.Clear();

                for(;;)
                {
                    pm::ProcessEventInfo eventInfo;
                    result = pm::GetProcessEventInfo(&eventInfo);
                    if( result.IsFailure() )
                    {
                        NN_DETAIL_NS_TRACE("[ProcessLaunchControl] pm::GetProcessEventInfo failed.(%08x)\n", result.GetInnerValueForDebug());
                        break;
                    }

                    for (auto& item : g_ObserverList)
                    {
                        item.Notify(eventInfo);
                    }

                    if( eventInfo.event == pm::ProcessEvent_Exception )
                    {
                        OnExceptionEvent(eventInfo);
                    }
                    else if( eventInfo.event == pm::ProcessEvent_Exit )
                    {
                        OnExitEvent(eventInfo);
                    }
                    else if( eventInfo.event == pm::ProcessEvent_Started ||
                             eventInfo.event == pm::ProcessEvent_DebugRunning )
                    {
                        OnStartedEvent(eventInfo);
                    }
//                    else if( eventInfo.event == pm::ProcessEvent_DebugRunning )
//                    {
//                        OnDebugRunningEvent(eventInfo);
//                    }
                    else if( eventInfo.event == pm::ProcessEvent_None )
                    {
                        break;
                    }
                    else if( eventInfo.event == pm::ProcessEvent_DebugBreaked )
                    {
                        // am には既に通知済みなので、ここでは何もしない。
                    }
                    else
                    {
                        NN_DETAIL_NS_TRACE("[ProcessLaunchControl] pm::GetProcessEventInfo unknown event.(%08x)\n", eventInfo.event);
                    }
                }
            }
        }

        int MakeProcessManagerFlags(int flags)
        {
            int pmFlags = 0;

            if( (flags & ns::LaunchProgramFlags_NotifyStart) != 0 )
            {
                pmFlags |= pm::LaunchProgramFlags_NotifyStart;
            }
            if( (flags & ns::LaunchProgramFlags_NotifyExit) != 0 )
            {
                pmFlags |= pm::LaunchProgramFlags_NotifyExit;
            }
            if( (flags & ns::LaunchProgramFlags_NotStart) != 0 )
            {
                pmFlags |= pm::LaunchProgramFlags_NotStart;
            }
            if( (flags & ns::LaunchProgramFlags_NotifyException) != 0 )
            {
                pmFlags |= pm::LaunchProgramFlags_NotifyException;
            }
            if( (flags & ns::LaunchProgramFlags_DisableAslr) != 0 )
            {
                pmFlags |= pm::LaunchProgramFlags_DisableAslr;
            }
            if( (flags & ns::LaunchProgramFlags_NotifyDebug) != 0 )
            {
                pmFlags |= pm::LaunchProgramFlags_NotifyDebug;
            }

            return pmFlags;
        }

    }   // anonymous namespace


    void StartLaunchControl() NN_NOEXCEPT
    {
        Result result;
        result = nn::os::CreateThread(
                    &g_Thread,
                    ProcessLaunchControl,
                    nullptr,
                    g_Stack,
                    sizeof(g_Stack),
                    NN_SYSTEM_THREAD_PRIORITY(nssrv, ProcessLaunchControlTask));
        NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

        // LaunchProgram を呼ぶ際に settings にアクセスさせた場合、InitializeSdev による SystemUpdaterHostFs.nca の実行時にデッドロックする
        // ので最初にキャッシングしてこれを使いまわす
        nn::settings::fwdbg::GetSettingsItemValue(&s_IsJitDebugEnabled, sizeof(s_IsJitDebugEnabled), "jit_debug", "enable_jit_debug");

        nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(nssrv, ProcessLaunchControlTask));
        nn::os::StartThread(&g_Thread);
    }

    nn::Result LaunchProgram(
        nn::os::ProcessId*              pOut,
        const nn::ncm::ProgramLocation& location,
        int                             flags) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scope(g_Lock);

        auto pData = Find(IdOfEmpty);
        if( pData == NULL )
        {
            NN_ABORT("[ns] ProcessData overflow");
        }

        // アプリケーション起動時にクラッシュレポートの取得を有効化する
        EnableApplicationCrashReport(true);
        // アプリケーション起動時に、クラッシュログの設定を未設定状態に戻す
        s_pIsEnabledApplicationAllThreadDumpOnCrash = util::nullopt;

        if (s_IsJitDebugEnabled)
        {
            flags |= ns::LaunchProgramFlags_NotifyException;
        }

        int pmFlags = MakeProcessManagerFlags(flags);
        auto ret = pm::LaunchProgram(pOut, location, pmFlags);
        if( ret.IsSuccess() )
        {
            pData->processId = *pOut;
            GetSharedDevelopInterfaceServerImpl().RegisterState(&pData->pEvent, pData->processId);
        }

        return ret;
    }

    nn::Result TerminateProcess(nn::os::ProcessId id) NN_NOEXCEPT
    {
        return pm::TerminateProcess(id);
    }

    nn::Result TerminateProgram(nn::ncm::ProgramId id) NN_NOEXCEPT
    {
        return pm::TerminateProgram(id);
    }

    nn::Result TerminateApplication() NN_NOEXCEPT
    {
        // アプリケーションのプロセス ID を取得
        os::ProcessId processId;
        auto result = nn::pm::GetApplicationProcessIdForShell(&processId);

        if( result.IsFailure() )
        {
            return ns::ResultApplicationNotRunning();
        }

        return pm::TerminateProcess(processId);
    }

    void RegisterShellEventObserver(ShellEventObserverHolder* holder) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(g_Lock);

        g_ObserverList.push_back(*holder);
    }

    void UnregisterShellEventObserver(ShellEventObserverHolder* holder) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> guard(g_Lock);

        for (auto& item : g_ObserverList)
        {
            if (&item == holder)
            {
                g_ObserverList.erase(g_ObserverList.iterator_to(item));
                break;
            }
        }
    }

    util::optional<os::ProcessId> GetRunningApplicationProcessId() NN_NOEXCEPT
    {
        os::ProcessId processId;
        auto result = nn::pm::GetApplicationProcessIdForShell(&processId);

        return result.IsSuccess() ? util::optional<os::ProcessId>(processId) : util::nullopt;
    }

    void EnableApplicationCrashReport(bool isEnabled) NN_NOEXCEPT
    {
        s_IsApplicationCrashReportEnabled = isEnabled;
    }

    bool IsApplicationCrashReportEnabled() NN_NOEXCEPT
    {
        return s_IsApplicationCrashReportEnabled;
    }

    void EnableApplicationAllThreadDumpOnCrash(bool isEnabled) NN_NOEXCEPT
    {
        s_pIsEnabledApplicationAllThreadDumpOnCrash = isEnabled;
    }

    void RegisterApplicationCrashReportPolicy(ns::CrashReport policy) NN_NOEXCEPT
    {
        s_AplicationCrashReportPolicy = policy;
    }

}}}  // namespace nn::ns::srv

