﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <malloc>
#endif
#include <nn/nn_StaticAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/es.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_GameCardManager.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_Result.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/util/util_ScopedTransaction.h>

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
#include <nn/hid/system/hid_InputDetection.h>
#endif // NN_BUILD_CONFIG_OS_WIN

#include "ns_DebugUtil.h"
#include "ns_GameCardStopperImpl.h"
#include "ns_LogUtil.h"
#include "ns_ProgramIndexUtil.h"
#include "ns_SystemUpdateUtil.h"
#include "ns_ForEachUtil.h"

namespace nn { namespace ns { namespace srv {
    namespace {
        static const char GameCardMountName[] = "@GcApp";
        static const char GameCardRootPath[] = "@GcApp:/";

        Result BuildGameCardContentMetaDatabase(ncm::ContentMetaDatabase* db) NN_NOEXCEPT
        {
            ncm::ContentMetaDatabaseBuilder builder(db);

            NN_RESULT_DO(builder.BuildFromPackage(GameCardRootPath));
            NN_RESULT_SUCCESS;
        }

        bool Includes(fs::GameCardAttribute attribute, fs::GameCardAttribute flag) NN_NOEXCEPT
        {
            return (static_cast<uint8_t>(attribute) & static_cast<uint8_t>(flag)) == static_cast<uint8_t>(flag);
        }
        Result LoadFile(ncm::AutoBuffer* buffer, const char* fileName) NN_NOEXCEPT
        {
            char path[256];
            util::SNPrintf(path, sizeof(path), "%s:/%s", GameCardMountName, fileName);

            fs::FileHandle file;
            NN_RESULT_TRY(fs::OpenFile(&file, path, fs::OpenMode_Read))
                NN_RESULT_CATCH_CONVERT(fs::ResultPathNotFound, ResultOnCardPatchRightsNotFound())
            NN_RESULT_END_TRY
            NN_UTIL_SCOPE_EXIT { fs::CloseFile(file); };

            int64_t fileSize;
            NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
            NN_RESULT_DO(buffer->Initialize(static_cast<size_t>(fileSize)));
            NN_RESULT_DO(fs::ReadFile(file, 0, buffer->Get(), buffer->GetSize()));

            NN_RESULT_SUCCESS;
        }
    }

    Result GameCardManager::Initialize(IntegratedContentManager* contentManager, ApplicationRecordDatabase* recordDb, ApplicationControlDataManager* dataManager, AppletLauncher* appletLauncher) NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> stateLock(m_StateLock);

            m_ContentManager = contentManager;
            m_RecordDb = recordDb;
            m_ControlDataManager = dataManager;
            m_AppletLauncher = appletLauncher;

            auto result = TryLoadAutoBoot();
            if(result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[GameCardManager] Failed to try load auto boot 0x%08x\n", result.GetInnerValueForDebug());
                HandleMountResult(result);
            }

            if (m_ApplicationIdForAutoBoot || m_IsCardUpdateAppliedOnAutoBoot)
            {
                NN_DETAIL_NS_TRACE("[GameCardManager] Auto boot game card is detected. Stop other game card detection\n");
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_DO(fs::OpenGameCardDetectionEventNotifier(&m_Notifier));
            NN_RESULT_DO(m_Notifier->BindEvent(&m_CardDetectEvent, os::EventClearMode_ManualClear));
        }

        {
            std::lock_guard<os::Mutex> threadLock(m_ThreadLock);

            StartAutoMountThread();
        }

        NN_RESULT_SUCCESS;
    }

    void GameCardManager::HandleGameCardEvent() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);

        // アタッチされているのにここに来ているということは、一度抜けたということ
        if (m_IsCardAttached)
        {
            NN_DETAIL_NS_TRACE("[GameCardManager] Game card remove detected\n");
            HandleRemoved();
        }
        else if (fs::IsGameCardInserted())
        {
            NN_DETAIL_NS_TRACE("[GameCardManager] Game card insert detected\n");
            HandleInserted();
        }
        else // アタッチされてないし、カードもささってない
        {
            NN_DETAIL_NS_TRACE("[GameCardManager] Clear update detection event\n");
            m_UpdateDetectionEvent.Clear();
        }
    }

    void GameCardManager::Run() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            HandleGameCardEvent();

            int index = os::WaitAny(&m_CardDetectEvent, m_StopEvent.GetBase());
            if (index == 0)
            {
                os::ClearSystemEvent(&m_CardDetectEvent);
            }
            else if (index == 1)
            {
                m_StopEvent.Clear();
                break;
            }
            else
            {
                NN_ABORT("[GameCardManager] Receive unknown signal.\n");
            }

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
            // 状態変化があったら無操作状態を解除
            nn::hid::system::NotifyInputDetector(nn::hid::system::InputSourceId::GameCardSlot::Mask);
            NN_DETAIL_NS_TRACE("[GameCardManager] Notified hid input detector\n");
#endif // NN_BUILD_CONFIG_OS_WIN
        }
    }

    os::NativeHandle GameCardManager::GetAttachmentEvent() NN_NOEXCEPT
    {
        return m_SignalEvent.GetReadableHandle();
    }

    os::NativeHandle GameCardManager::GetMountFailureEvent() NN_NOEXCEPT
    {
        return m_MountErrorEvent.GetReadableHandle();
    }

    void GameCardManager::GetAttachmentInfo(GameCardAttachmentInfo* pOutInfo) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        pOutInfo->sequence = m_EventCount;
        pOutInfo->isAttached = m_IsCardAttached || m_ApplicationIdForAutoBoot;
    }

    void GameCardManager::HandleInserted() NN_NOEXCEPT
    {
        auto result = GetMountGameCardResultForDebug();
        if (result.IsFailure())
        {
            HandleMountResult(result);
            return;
        }

        {
            bool needsUpdate;
            result = CheckSystemUpdateNeed(&needsUpdate);
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[GameCardManager] Failed to check system update need as 0x%08x.\n", result.GetInnerValueForDebug());
                HandleMountResult(result);

                return;
            }

            if (needsUpdate)
            {
                NN_DETAIL_NS_TRACE("[GameCardManager] Need system update\n");
                m_UpdateDetectionEvent.Signal();

                return;
            }
        }

        {
            result = Load();
            if (result.IsFailure())
            {
                NN_DETAIL_NS_TRACE("[GameCardManager] Failed to load game card as 0x%08x.\n", result.GetInnerValueForDebug());
                HandleMountResult(result);
                return;
            }

            m_IsCardAttached = true;
            m_EventCount++;
            m_SignalEvent.Signal();

            NN_DETAIL_NS_TRACE("[GameCardManager] Game card attached\n");
        }
    }

    void GameCardManager::HandleRemoved() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(Unload());

        m_IsCardAttached = false;
        m_EventCount++;
        if (!m_IsSignalEventDisabledOnRemove)
        {
            m_SignalEvent.Signal();
        }
        else
        {
            m_IsSignalEventDisabledOnRemove = false;
        }
        m_RecordDb->NotifyUpdate();

        NN_DETAIL_NS_TRACE("[GameCardManager] Game card detached\n");
    }

    Result GameCardManager::CheckSystemUpdateNeed(bool* outValue) NN_NOEXCEPT
    {
        fs::GameCardHandle handle;
        NN_RESULT_DO(fs::GetGameCardHandle(&handle));
        fs::GameCardUpdatePartitionInfo info;
        NN_RESULT_DO(fs::GetGameCardUpdatePartitionInfo(&info, handle));
        if (info.id == 0)
        {
            NN_DETAIL_NS_TRACE("[GameCardManager] No update partition found\n");
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        NN_DETAIL_NS_TRACE("[GameCardManager] Update partition info id 0x%016llx version %u\n", info.id, info.version);

        // CUP 判定は exFAT ドライバを考慮しない
        bool needsUpdate;
        NN_RESULT_DO(NeedsUpdate(&needsUpdate, ncm::ContentMetaKey::Make(info.id, info.version, ncm::ContentMetaType::SystemUpdate), false));

        *outValue = needsUpdate;
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::TryLoadAutoBoot() NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        bool isAutoBootEnabled;
        nn::settings::fwdbg::GetSettingsItemValue(&isAutoBootEnabled, sizeof(isAutoBootEnabled), "ns.autoboot", "enabled");

        if (!isAutoBootEnabled)
        {
            NN_RESULT_SUCCESS;
        }

        if (!fs::IsGameCardInserted())
        {
            NN_RESULT_SUCCESS;
        }

        fs::GameCardHandle handle;
        NN_RESULT_DO(fs::GetGameCardHandle(&handle));
        fs::GameCardAttribute attribute;
        NN_RESULT_DO(fs::GetGameCardAttribute(&attribute, handle));

        if (!Includes(attribute, fs::GameCardAttribute::AutoBootFlag))
        {
            NN_RESULT_SUCCESS;
        }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        bool needsUpdate = false;
        NN_RESULT_DO(CheckSystemUpdateNeed(&needsUpdate));

        // 本体更新が必要なら自動 CUP を行う
        if(needsUpdate)
        {
            NN_DETAIL_NS_TRACE("[GameCardManager] Need system update\n");

            // ここで生成するスレッドが終了するときにはシステム全体が再起動するはずなので、
            // 特に破棄処理はしないで確保し続ける
            m_ThreadLock.lock();

            m_Thread.emplace();

            NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread.value(), [](void* p)
            {
                reinterpret_cast<GameCardManager*>(p)->ApplyCardUpdateForAutoBoot();
            }, this, m_Stack, m_StackSize, NN_SYSTEM_THREAD_PRIORITY(nssrv, AsyncApplyCardUpdateForAutoBoot)));
            os::SetThreadNamePointer(&m_Thread.value(), NN_SYSTEM_THREAD_NAME(nssrv, AsyncApplyCardUpdateForAutoBoot));
            os::StartThread(&m_Thread.value());

            m_IsCardUpdateAppliedOnAutoBoot = true;
            NN_RESULT_SUCCESS;
        }
#endif // NN_BUILD_CONFIG_OS_HORIZON

        NN_RESULT_DO(fs::MountGameCardPartition(GameCardMountName, handle, fs::GameCardPartition::Secure));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(GameCardMountName); };

        NN_RESULT_DO(ActivateContentManager());
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            InactivateContentManager();
        };

        NN_RESULT_DO(RegisterToIntegratedContentManager());
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            UnregisterFromIntegratedContentManager();
        };

        int count;
        ncm::ApplicationId idList[2];
        NN_RESULT_DO(ncm::ListApplicationFromPackage(&count, idList, static_cast<int>(NN_ARRAY_SIZE(idList)), GameCardRootPath));
        NN_RESULT_THROW_UNLESS(count > 0, ResultNoApplicationContentGameCardNotSupported());
        NN_RESULT_THROW_UNLESS(count == 1, ResultMultiApplicationGameCardNotSupported());

        ncm::ApplicationId appId = { idList[0] };
        m_ApplicationIdForAutoBoot = appId;

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::ActivateContentManager() NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            InactivateContentManager();
        };

        NN_RESULT_DO(ncm::ActivateContentStorage(ncm::StorageId::GameCard));
        NN_RESULT_DO(ncm::ActivateContentMetaDatabase(ncm::StorageId::GameCard));

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::GameCard));
        NN_RESULT_DO(BuildGameCardContentMetaDatabase(&db));

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::InactivateContentManager() NN_NOEXCEPT
    {
        NN_RESULT_DO(ncm::InactivateContentMetaDatabase(ncm::StorageId::GameCard));
        NN_RESULT_DO(ncm::InactivateContentStorage(ncm::StorageId::GameCard));

        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::RegisterContentManager() NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        NN_RESULT_DO(RegisterToIntegratedContentManager());
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            UnregisterFromIntegratedContentManager();
        };

        NN_RESULT_DO(RegisterToApplicationRecordDatabaseManager());
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            UnregisterFromApplicationRecordDatabaseManager();
        };

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    void GameCardManager::UnregisterContentManager() NN_NOEXCEPT
    {
        UnregisterFromApplicationRecordDatabaseManager();
        UnregisterFromIntegratedContentManager();
    }

    Result GameCardManager::RecordGameCardInsertedEvent() NN_NOEXCEPT
    {
        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::GameCard));

        std::lock_guard<nn::os::Mutex> lock(m_GameCardInfoLock);
        if (m_GameCardInfo)
        {
            for (int i = 0; i < m_GameCardInfo->idCount; i++)
            {
                NN_RESULT_DO(RecordInsertedEvent(m_GameCardInfo->idList[i], &db));
            }
            m_RecordDb->NotifyUpdate();
        }

        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::RecordInsertedEvent(ncm::ApplicationId appId, ncm::ContentMetaDatabase* pDb) NN_NOEXCEPT
    {
        // 現状 ncm::ContentMetaType::Application しか登録を行わないので、 Application に限定して処理を行う
        // Patch や Aoc の情報は登録された ContentMetaDatabase から直接参照される
        ncm::ContentMetaKey key;
        auto listCount = pDb->ListContentMeta(&key, 1, ncm::ContentMetaType::Application, appId);
        NN_ABORT_UNLESS(listCount.total == listCount.listed);

        ncm::StorageContentMetaKey storageKey = { key, ncm::StorageId::GameCard };

        auto isNewRecord = !m_RecordDb->Has(appId);
        NN_RESULT_DO(m_RecordDb->Push(appId, ApplicationEvent::GameCardInserted, &storageKey, 1, true, true));
        if (isNewRecord)
        {
            NN_NS_TRACE_RESULT_IF_FAILED(m_RecordDb->EnableAutoUpdate(appId), "[GameCardManager] Failed to enable auto update 0x%016llx\n", appId);
        }

        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::RecordGameCardRemovedEvent() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_GameCardInfoLock);
        if (m_GameCardInfo)
        {
            // 抜かれたことを記録に伝えるため、UpdateEvent する
            for (int i = 0; i < m_GameCardInfo->idCount; i++)
            {
                NN_RESULT_DO(RecordRemovedEvent(m_GameCardInfo->idList[i]));
            }
            m_RecordDb->NotifyUpdate();
        }
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::RecordRemovedEvent(ncm::ApplicationId appId) NN_NOEXCEPT
    {
        bool hasRecord = m_RecordDb->Has(appId);
        if (hasRecord)
        {
            NN_RESULT_TRY(m_RecordDb->UpdateEvent(appId, ApplicationEvent::GameCardRemoved, false, true))
                NN_RESULT_CATCH(ResultApplicationRecordNotFound)
                {
                    NN_DETAIL_NS_TRACE("[GameCardManager] Not found application record 0x%016llx\n", appId);
                }
            NN_RESULT_END_TRY
        }
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::Load() NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        fs::GameCardHandle handle;
        NN_RESULT_DO(fs::GetGameCardHandle(&handle));

        fs::GameCardAttribute attribute;
        NN_RESULT_DO(fs::GetGameCardAttribute(&attribute, handle));
        if (Includes(attribute, fs::GameCardAttribute::HistoryEraseFlag))
        {
            NN_RESULT_THROW(ResultGameCardNotRecordable());
        }

        NN_RESULT_DO(fs::MountGameCardPartition(GameCardMountName, handle, fs::GameCardPartition::Secure));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(GameCardMountName); };

        NN_RESULT_DO(LoadGameCardInfo(handle));
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            ClearGameCardInfo();
        };

        NN_RESULT_DO(ActivateContentManager());
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            InactivateContentManager();
        };

        NN_RESULT_DO(LoadControlData());

        NN_RESULT_DO(RegisterContentManager());
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            UnregisterContentManager();
        };

        NN_RESULT_DO(RecordGameCardInsertedEvent());

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::Unload() NN_NOEXCEPT
    {
        // ここに来る場合は、既にゲームカードが抜けている
        // この関数が処理されるまでの間に GetApplicationView() を呼ばれると、ゲームカードが挿入されていると判定されるデータが渡ってしまう。
        // そのため、一時的にゲームカードのアプリケーションが抜去状態なのに挿入状態と判定される。
        // RecordGameCardRemovedEvent() が実行されると、
        // 記録が更新されて View のキャッシュも更新されるので、次回 GetApplicationView() 取得時に正常状態に戻る

        UnregisterContentManager();
        NN_RESULT_DO(InactivateContentManager());

        // GameCardInfo のキャッシュを使うので、先に記録を更新する
        NN_RESULT_DO(RecordGameCardRemovedEvent());
        ClearGameCardInfo();
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::LoadGameCardInfo(const fs::GameCardHandle& handle) NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        int idCount;
        ncm::ApplicationId idList[MaxApplicationCountOnGameCard];
        NN_RESULT_DO(ncm::ListApplicationFromPackage(&idCount, idList, static_cast<int>(NN_ARRAY_SIZE(idList)), GameCardRootPath));
        NN_RESULT_THROW_UNLESS(idCount <= MaxApplicationCountOnGameCard, ResultTooManyApplicationContentOnGameCard());

        // カードに書き込まれている順通りにアプリケーション記録を並べたいので、
        // 先頭にしたいアプリケーションが最後に登録されるように配列を逆順に並べ替える
        std::reverse(idList, idList + idCount);

        std::lock_guard<nn::os::Mutex> lock(m_GameCardInfoLock);

        m_GameCardInfo.emplace();
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            m_GameCardInfo = util::nullopt;
        };

        NN_RESULT_DO(fs::GetGameCardDeviceId(m_GameCardInfo->deviceId, sizeof(m_GameCardInfo->deviceId)));
        NN_RESULT_DO(fs::GetGameCardDeviceCertificate(m_GameCardInfo->cert, sizeof(m_GameCardInfo->cert), handle));
        std::copy(idList, idList + idCount, m_GameCardInfo->idList);
        m_GameCardInfo->idCount = idCount;

        NN_DETAIL_NS_TRACE("[GameCardManager] GameCardInfo cached\n");

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::LoadControlData() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_GameCardInfoLock);
        if (m_GameCardInfo)
        {
            ncm::ContentMetaDatabase db;
            NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::GameCard));
            ncm::ContentStorage storage;
            NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::GameCard));
            for (int i = 0; i < m_GameCardInfo->idCount; i++)
            {
                NN_RESULT_DO(LoadControlDataImpl(m_GameCardInfo->idList[i], &db, &storage));
            }
        }

        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::LoadControlDataImpl(ncm::ApplicationId appId, ncm::ContentMetaDatabase* pDb, ncm::ContentStorage* pStorage) NN_NOEXCEPT
    {
        ncm::ContentMetaKey appKey;
        NN_RESULT_DO(pDb->GetLatest(&appKey, appId.value));
        ncm::PatchId patchId;
        NN_RESULT_DO(pDb->GetPatchId(&patchId, appKey));
        ncm::ContentMetaKey patchKey;
        Result result = pDb->GetLatest(&patchKey, patchId.value);
        ncm::ContentMetaKey mainKey = result.IsSuccess() ? patchKey : appKey;

        util::optional<ncm::ContentInfo> controlInfo;
        ForEachContentInfo(pDb, mainKey, [&controlInfo](bool* outIsEnd, const ncm::ContentInfo& info) NN_NOEXCEPT
        {
            if (info.type == ncm::ContentType::Control)
            {
                controlInfo = info;
                *outIsEnd = true;
            }
            else
            {
                *outIsEnd = false;
            }
            NN_RESULT_SUCCESS;
        });
        NN_RESULT_THROW_UNLESS(controlInfo, ResultApplicationControlDataNotFound());

        ncm::Path path;
        pStorage->GetPath(&path, controlInfo->id);
        NN_RESULT_DO(m_ControlDataManager->PutPath(appId, mainKey.version, GetProgramIndex(*controlInfo), path, true));

        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::RegisterToIntegratedContentManager() NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::GameCard));
        ncm::ContentStorage storage;
        NN_RESULT_DO(ncm::OpenContentStorage(&storage, ncm::StorageId::GameCard));

        m_ContentManager->Register(ncm::StorageId::GameCard, std::move(db), std::move(storage));
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            UnregisterFromIntegratedContentManager();
        };

        lr::LocationResolver resolver;
        NN_RESULT_DO(lr::OpenLocationResolver(&resolver, ncm::StorageId::GameCard));
        NN_RESULT_DO(lr::RefreshLocationResolver(ncm::StorageId::GameCard));

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    void GameCardManager::UnregisterFromIntegratedContentManager() NN_NOEXCEPT
    {
        m_ContentManager->Unregister(ncm::StorageId::GameCard);
    }

    Result GameCardManager::RegisterToApplicationRecordDatabaseManager() NN_NOEXCEPT
    {
        util::ScopedTransaction transaction;

        ncm::ContentMetaDatabase db;
        NN_RESULT_DO(ncm::OpenContentMetaDatabase(&db, ncm::StorageId::GameCard));
        m_RecordDb->RegisterGameCardDatabase(std::move(db));
        NN_UTIL_RESERVE_SCOPED_ROLLBACK(transaction)
        {
            UnregisterFromIntegratedContentManager();
        };

        transaction.Commit();
        NN_RESULT_SUCCESS;
    }

    void GameCardManager::UnregisterFromApplicationRecordDatabaseManager() NN_NOEXCEPT
    {
        m_RecordDb->UnregisterGameCardDatabase();
    }

    void GameCardManager::EnableAutoMount() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> threadLock(m_ThreadLock);

        if (m_Thread || m_ApplicationIdForAutoBoot)
        {
            return;
        }

        {
            std::lock_guard<os::Mutex> stateLock(m_StateLock);

            // m_Stopper は管理しなくて良いので、Detach してから無効化する
            m_Stopper.Detach();
            m_Stopper = nullptr;
        }

        // ゲームカードの書き込みをしている間にシグナルされてしまう。
        // そのため、挿入後すぐに抜去されたようにハンドリングしてしまうので、
        // 一旦クリアしておく
        if (os::TryWaitSystemEvent(&m_CardDetectEvent))
        {
            os::ClearSystemEvent(&m_CardDetectEvent);
        }

        StartAutoMountThread();
    }

    void GameCardManager::DisableAutoMount() NN_NOEXCEPT
    {
        {
            std::lock_guard<os::Mutex> threadLock(m_ThreadLock);

            if (!m_Thread)
            {
                return;
            }

            m_StopEvent.Signal();
            os::WaitThread(&m_Thread.value());
            os::DestroyThread(&m_Thread.value());
            m_Thread = util::nullopt;
        }

        {
            std::lock_guard<os::Mutex> stateLock(m_StateLock);
            if (m_IsCardAttached)
            {
                HandleRemoved();
            }
        }
    }

    void GameCardManager::StartAutoMountThread() NN_NOEXCEPT
    {
        m_Thread.emplace();
        NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Thread.value(), [](void* p)
        {
            reinterpret_cast<GameCardManager*>(p)->Run();
        }, this, m_Stack, m_StackSize, NN_SYSTEM_THREAD_PRIORITY(nssrv, DetectGameCardTask)));
        os::SetThreadNamePointer(&m_Thread.value(), NN_SYSTEM_THREAD_NAME(nssrv, DetectGameCardTask));
        os::StartThread(&m_Thread.value());
    }

    sf::SharedPointer<ns::detail::IGameCardStopper> GameCardManager::GetStopper() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);

        if (!m_Stopper)
        {
            auto sharedPointer = sf::CreateSharedObjectEmplaced<ns::detail::IGameCardStopper, GameCardStopperImpl>(this);
            auto sharedObject = sharedPointer.Detach();
            m_Stopper = sf::SharedPointer<ns::detail::IGameCardStopper>(sharedObject, false);
            return sf::SharedPointer<ns::detail::IGameCardStopper>(sharedObject, false);
        }
        return m_Stopper;
    }
    Result GameCardManager::ImportTicket(const fs::RightsId& id) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        std::lock_guard<os::Mutex> lock(m_StateLock);

        fs::GameCardHandle handle;
        NN_RESULT_DO(fs::GetGameCardHandle(&handle));

        NN_RESULT_DO(fs::MountGameCardPartition(GameCardMountName, handle, fs::GameCardPartition::Secure));
        NN_UTIL_SCOPE_EXIT{ fs::Unmount(GameCardMountName); };

        ncm::AutoBuffer ticket;
        ncm::AutoBuffer cert;
        {
            char fileName[ncm::TicketFileStringLength + 1];
            ncm::GetTicketFileStringFromRightsId(fileName, sizeof(fileName), id);
            NN_RESULT_DO(LoadFile(&ticket, fileName));
        }
        {
            char fileName[ncm::CertFileStringLength + 1];
            ncm::GetCertificateFileStringFromRightsId(fileName, sizeof(fileName), id);
            NN_RESULT_DO(LoadFile(&cert, fileName));
        }
        NN_RESULT_DO(es::ImportTicket(ticket.Get(), ticket.GetSize(), cert.Get(), cert.GetSize()));
#else
        NN_UNUSED(id);
#endif
        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::GetGameCardInfo(GameCardInfo* out) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_GameCardInfoLock);

        NN_RESULT_THROW_UNLESS(m_GameCardInfo, ns::ResultGameCardInfoNotFound());

        *out = *m_GameCardInfo;

        NN_RESULT_SUCCESS;
    }

    void GameCardManager::ClearGameCardInfo() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_GameCardInfoLock);
        NN_DETAIL_NS_TRACE("[GameCardManager] GameCardInfo cleared\n");
        m_GameCardInfo = util::nullopt;
    }

    void GameCardManager::HandleMountResult(Result result) NN_NOEXCEPT
    {
        if (fs::ResultGameCardInitializeAsicFailure::Includes(result))
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        if (result.IsFailure())
        {
            m_LastMountUnexpectedResult = result;
            m_MountErrorEvent.Signal();
        }
    }

    bool GameCardManager::IsGameCardInserted() const NN_NOEXCEPT
    {
        return fs::IsGameCardInserted();
    }

    Result GameCardManager::EnsureAccess() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsCardAttached, ResultGameCardNotInserted());
        NN_RESULT_DO(GetTryGameCardAccessResultForDebug());

        fs::GameCardHandle handle;
        NN_RESULT_DO(fs::GetGameCardHandle(&handle));
        NN_RESULT_DO(fs::CheckGameCardPartitionAvailability(handle, fs::GameCardPartition::Secure));

        NN_RESULT_SUCCESS;
    }

    Result GameCardManager::GetLastMountFailureResult() const NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(m_StateLock);
        return m_LastMountUnexpectedResult;
    }

    Result GameCardManager::ListApplicationId(int* outCount, ncm::ApplicationId* outIdList, int numList) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_IsCardAttached, ResultGameCardNotInserted());
        NN_RESULT_THROW_UNLESS(numList >= 0, ResultInvalidListCount());

        std::lock_guard<os::Mutex> lock(m_GameCardInfoLock);
        NN_RESULT_THROW_UNLESS(m_GameCardInfo, ResultGameCardNotInserted());

        *outCount = std::min(numList, m_GameCardInfo->idCount);
        std::copy(m_GameCardInfo->idList, m_GameCardInfo->idList + *outCount, outIdList);

        NN_RESULT_SUCCESS;
    }

    void GameCardManager::ApplyCardUpdateForAutoBoot() NN_NOEXCEPT
    {
        settings::system::AppletLaunchFlagSet flagSet;
        settings::system::GetAppletLaunchFlags(&flagSet);
        if (flagSet.Test<settings::system::AppletLaunchFlag::Migration>())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(ResultMigrationIsRunning());
        }

        m_AppletLauncher->NotifyStartCardUpdate();

        SystemUpdateControl systemUpdateControl;
        NN_ABORT_UNLESS_RESULT_SUCCESS(systemUpdateControl.Occupy());

#if defined(NN_BUILD_CONFIG_OS_WIN)
        void* const buf = _aligned_malloc(nn::os::MemoryPageSize, nn::os::MemoryPageSize);
#else
        void* const buf = std::aligned_alloc(nn::os::MemoryPageSize, nn::os::MemoryPageSize);
#endif
        NN_UTIL_SCOPE_EXIT{ systemUpdateControl.Relieve(); free(buf); };
        NN_ABORT_UNLESS_RESULT_SUCCESS(systemUpdateControl.SetupCardUpdate(buf, nn::os::MemoryPageSize));

        AsyncResult asyncResult;
        NN_ABORT_UNLESS_RESULT_SUCCESS(systemUpdateControl.RequestPrepareCardUpdate(&asyncResult));

        while(!asyncResult.TryWait())
        {
            auto systemUpdateProgress = systemUpdateControl.GetPrepareCardUpdateProgress();
            NN_UNUSED(systemUpdateProgress);
            NN_DETAIL_NS_TRACE("[GameCardManager] SystemUpdateProgress %d / %d\n", systemUpdateProgress.loaded, systemUpdateProgress.total);
            m_AppletLauncher->RequestShowCardUpdateProcessing();

            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1500) );
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(asyncResult.Get());
        systemUpdateControl.ApplyCardUpdate();
        m_AppletLauncher->NotifyEndCardUpdate();
        m_AppletLauncher->RequestReboot();
    }
}}}
