﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/es/es_RightsApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/detail/ns_IAsync.sfdl.h>
#include <nn/ns/detail/ns_Log.h>
#include <nn/ns/srv/ns_DynamicRightsManager.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_StaticOneAllocator.h>
#include "ns_AccountUtil.h"
#include "ns_AsyncDynamicRightsImpl.h"

namespace nn { namespace ns { namespace srv {

    typedef sf::ObjectFactory<sf::impl::StaticOneAllocationPolicy> StaticOneFactory;

    DynamicRightsManager::DynamicRightsManager(
        ApplicationRecordDatabase* pDb, IntegratedContentManager* pIntegrated,
        RequestServer* pRequestServer, RightsEnvironmentManager* pRightsManager) NN_NOEXCEPT
        : m_pRecordDb(pDb), m_pIntegrated(pIntegrated),
          m_pRequestServer(pRequestServer), m_pRightsEnvironmentManager(pRightsManager)
    {
    }

    Result DynamicRightsManager::RequestApplicationRightsOnServer(
        sf::Out<sf::NativeHandle> outHandle,
        sf::Out<sf::SharedPointer<ns::detail::IAsyncValue>> outAsync,
        ncm::ApplicationId id,
        const account::Uid& uid,
        Bit32 flags) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pRecordDb->Has(id), ResultApplicationRecordNotFound());

        ApplicationRightsQueryFlags queryFlags;
        queryFlags._storage[0] = flags;
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncValue, AsyncApplicationRightsOnServerImpl>(m_pRecordDb, m_pIntegrated, id, uid, queryFlags, m_pRequestServer->Stop());
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result DynamicRightsManager::RequestAssignRights(
        sf::Out<sf::NativeHandle> outHandle,
        sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync,
        sf::InArray<ApplicationRightsOnServer> list) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(list.GetLength() > 0, ResultNotImplemented());
        for (size_t i = 0; i < list.GetLength(); i++)
        {
            NN_RESULT_THROW_UNLESS(m_pRecordDb->Has(list[i].id), ResultApplicationRecordNotFound());
        }

        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncAssignRightsImpl>();
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());
        NN_RESULT_DO(emplacedRef.GetImpl().Initialize(m_pRecordDb, m_pIntegrated, list.GetData(), static_cast<int>(list.GetLength()), m_pRequestServer->Stop()));

        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result DynamicRightsManager::RegisterUserOfAccountRestrictedRights(const account::Uid& uid) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        account::NintendoAccountId naId;
        NN_RESULT_DO(GetNintendoAccountId(&naId, uid));
        NN_RESULT_DO(es::RegisterAccountRestrictedRightsUser(naId));
#else
        NN_UNUSED(uid);
#endif
        NN_RESULT_SUCCESS;
    }

    void DynamicRightsManager::UnregisterAllUsersOfAccountRestrictedRights() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        es::UnregisterAllAccountRestrictedRightsUser();
#endif
    }

    Result DynamicRightsManager::RequestAssignRightsToResume(
        sf::Out<sf::NativeHandle> outHandle,
        sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync,
        RightsEnvironmentHandle handle,
        const account::Uid& uid) NN_NOEXCEPT
    {
        auto emplacedRef = StaticOneFactory::CreateSharedEmplaced<ns::detail::IAsyncResult, AsyncAssignRightsToResumeImpl>(m_pRightsEnvironmentManager, handle, uid, m_pRequestServer->Stop());
        NN_RESULT_THROW_UNLESS(emplacedRef, ResultOutOfMaxRunningTask());

        NN_RESULT_DO(emplacedRef.GetImpl().Run());

        *outHandle = sf::NativeHandle(emplacedRef.GetImpl().GetEvent().GetReadableHandle(), false);
        *outAsync = emplacedRef;

        NN_RESULT_SUCCESS;
    }

    Result DynamicRightsManager::VerifyActivatedRightsOwners(RightsEnvironmentHandle) NN_NOEXCEPT
    {
        // TODO: es の実装が入ったら対応する
        NN_RESULT_SUCCESS;
    }
}}}
