﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/srv/ns_DataStore.h>

namespace nn { namespace ns { namespace srv {
    Result DataStore::Initialize(const SaveDataInfo& info) NN_NOEXCEPT
    {
        fs::DisableAutoSaveDataCreation();
        NN_RESULT_TRY(fs::MountSystemSaveData(info.mountName, info.id))
            NN_RESULT_CATCH(fs::ResultTargetNotFound)
            {
                NN_RESULT_DO(fs::CreateSystemSaveData(info.id, info.saveDataSize, info.saveDataJournalSize, info.saveDataFlags));
                NN_RESULT_DO(fs::MountSystemSaveData(info.mountName, info.id));
            }
        NN_RESULT_END_TRY
        m_MountName.emplace();
        m_MountName->Assign(info.mountName);

        bool success = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                fs::Unmount(*m_MountName);
                m_MountName = util::nullopt;
            }
        };

        NN_RESULT_DO(HasLivingMark(&m_DetectsForceShutdown));
        if (!m_DetectsForceShutdown)
        {
            NN_RESULT_DO(MarkLiving());
        }

        success = true;

        NN_RESULT_SUCCESS;
    }

    DataStore::~DataStore() NN_NOEXCEPT
    {
        if (m_MountName)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(UnmarkLiving());
            fs::Unmount(*m_MountName);
        }
    }

    DataStore::Path DataStore::MakeLivingMarkPath() NN_NOEXCEPT
    {
        Path path;
        path.AssignFormat("%s:/%s", m_MountName->Get(), "living");

        return path;
    }

    Result DataStore::HasLivingMark(bool* outValue) NN_NOEXCEPT
    {
        fs::DirectoryEntryType entry;
        NN_RESULT_TRY(fs::GetEntryType(&entry, MakeLivingMarkPath()))
            NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY

        *outValue =true;
        NN_RESULT_SUCCESS;
    }

    Result DataStore::MarkLiving() NN_NOEXCEPT
    {
        NN_RESULT_DO(fs::CreateFile(MakeLivingMarkPath(), 0));
        NN_RESULT_DO(fs::CommitSaveData(*m_MountName));
        NN_RESULT_SUCCESS;
    }

    Result DataStore::UnmarkLiving() NN_NOEXCEPT
    {
        NN_RESULT_DO(fs::DeleteFile(MakeLivingMarkPath()));
        NN_RESULT_DO(fs::CommitSaveData(*m_MountName));
        NN_RESULT_SUCCESS;
    }
}}}
